#Model Processor and Core I/O
function FetchDataToInput(md::model,inputs::Inputs,elements::Vector{Tria},data::Vector{Float64},enum::IssmEnum) #{{{
	for i in 1:length(elements)
		InputCreate(elements[i],inputs,data,enum)
	end
end#}}}
function ModelProcessor(md::model, solutionstring::String) #{{{

	#Initialize structures
	elements    = Vector{Tria}(undef,0)
	vertices    = Vector{Vertex}(undef,0)
	results     = Vector{Result}(undef,0)
	parameters  = Parameters(Dict{IssmEnum,Parameter}())
	inputs      = Inputs(md.mesh.numberofelements, md.mesh.numberofvertices, Dict{IssmEnum,Input}())

	#Create  elements, vertices and materials (independent of the analysis)
	CreateElements(elements, md)
	CreateVertices(vertices, md)
	CreateParameters(parameters, md)
	CreateInputs(inputs,elements, md)
	if solutionstring=="TransientSolution"
		UpdateParameters(TransientAnalysis(), parameters, md)
	end

	#Now create analysis specific data structure
	if solutionstring=="StressbalanceSolution"
		analyses = [StressbalanceAnalysis()]
	elseif solutionstring=="TransientSolution"
		analyses = [StressbalanceAnalysis(), MasstransportAnalysis()]
	else
		error(solutionstring, " not supported by ModelProcessor")
	end

	#Initialize analysis specific datasets
	numanalyses = length(analyses)
	nodes       = Vector{Vector{Node}}(undef,numanalyses)
	constraints = Vector{Vector{Constraint}}(undef,numanalyses)
	for i in 1:numanalyses
		analysis = analyses[i]
		println("   creating datasets for analysis ", typeof(analysis))
		nodes[i]       = Vector{Node}(undef,0)
		constraints[i] = Vector{Constraint}(undef,0)

		UpdateParameters(analysis, parameters, md)
		UpdateElements(analysis, elements, inputs, md)
		CreateNodes(analysis, nodes[i], md)
		CreateConstraints(analysis, constraints[i], md)

		#Configure objects
		ConfigureObjectx(elements, nodes[i], vertices, parameters, inputs, i)

		#Constrain and Number nodes
		SpcNodesx(nodes[i], constraints[i], parameters)
		NodesDofx(nodes[i], parameters)
	end

	#Build FemModel
	return FemModel(analyses, elements, vertices,
						 Vector{Node}(undef,0), nodes,
						 parameters, inputs,
						 Vector{Constraint}(undef,0), constraints,
						 results)
end# }}}
function CreateElements(elements::Vector{Tria},md::model) #{{{

	#Make sure elements is currently empty
	@assert length(elements)==0



	count = 0
	for i in 1:md.mesh.numberofelements

		#Assume Linear Elements for now
		vertexids = md.mesh.elements[i,:]
		nodeids   = md.mesh.elements[i,:]

		#Call constructor and add to dataset elements
		push!(elements,Tria(i,count, vertexids))
	end

end# }}}
function CreateVertices(vertices::Vector{Vertex},md::model) #{{{

	#Make sure vertices is currently empty
	@assert length(vertices)==0

	#Get data from md
	x = md.mesh.x
	y = md.mesh.y

	count = 0
	for i in 1:md.mesh.numberofvertices
		push!(vertices,Vertex(i,x[i],y[i],0.))
	end

end# }}}
function CreateParameters(parameters::Parameters,md::model) #{{{

	#Get data from md
	AddParam(parameters,md.materials.rho_ice,MaterialsRhoIceEnum)
	AddParam(parameters,md.materials.rho_water,MaterialsRhoSeawaterEnum)
	AddParam(parameters,md.materials.rho_freshwater,MaterialsRhoFreshwaterEnum)
	AddParam(parameters,md.constants.g,ConstantsGEnum)
	
	#Set step and time, this will be overwritten if we run a transient
	AddParam(parameters,1,StepEnum)
	AddParam(parameters,0.0,TimeEnum)

end# }}}
function CreateInputs(inputs::Inputs,elements::Vector{Tria},md::model) #{{{

	#Only assume we have Matice for now
	FetchDataToInput(md,inputs,elements,md.materials.rheology_B,MaterialsRheologyBEnum)
	FetchDataToInput(md,inputs,elements,md.materials.rheology_n,MaterialsRheologyNEnum)
end# }}}
function OutputResultsx(femmodel::FemModel, md::model, solution::String)# {{{


	if solution=="TransientSolution"

		#Compute maximum number of steps
		maxstep = 0
		for i in length(femmodel.results)
			if(femmodel.results[i].step>maxstep) maxstep = femmodel.results[i].step end
		end

		#Initialize vector now that we know the size
		output = Vector{Dict}(undef, maxstep)
		for i in 1:maxstep; output[i] = Dict() end

		#Insert results in vector
		for i in 1:length(femmodel.results)
			result = femmodel.results[i]
			step   = femmodel.results[i].step
			(output[step])[EnumToString(result.enum)] = result.value
		end
	else
		output = Dict()
		for i in length(femmodel.results)
			result = femmodel.results[i]
			output[EnumToString(result.enum)] = result.value
		end
	end

	md.results[solution] = output

end# }}}

#Other modules
function ConfigureObjectx(elements::Vector{Tria}, nodes::Vector{Node}, vertices::Vector{Vertex}, parameters::Parameters, inputs::Inputs, analysis::Int64) #{{{

	for i in 1:length(elements)
		Configure(elements[i], nodes, vertices, parameters, inputs, analysis)
	end

end# }}}
function SpcNodesx(nodes::Vector{Node},constraints::Vector{Constraint},parameters::Parameters) #{{{

	for i in 1:length(constraints)
		ConstrainNode(constraints[i],nodes,parameters)
	end

end# }}}
function NodesDofx(nodes::Vector{Node}, parameters::Parameters) #{{{

	#Do we have any nodes?
	if length(nodes)==0
		return
	end

	#Do we really need to update dof indexing
	if(~RequiresDofReindexing(nodes)) return end

	print("   Renumbering degrees of freedom\n")
	DistributeDofs(nodes,GsetEnum)
	DistributeDofs(nodes,FsetEnum)
	DistributeDofs(nodes,SsetEnum)

end# }}}
function GetSolutionFromInputsx(analysis::Analysis,femmodel::FemModel) #{{{

	#Get size of vector
	gsize = NumberOfDofs(femmodel.nodes,GsetEnum)

	#Initialize solution vector
	ug = IssmVector(gsize)

	#Go through elements and plug in solution
	for i=1:length(femmodel.elements)
		GetSolutionFromInputs(analysis,ug,femmodel.elements[i])
	end

	return ug

end#}}}
function InputUpdateFromSolutionx(analysis::Analysis,ug::IssmVector,femmodel::FemModel) #{{{

	#Go through elements and plug in solution
	for i=1:length(femmodel.elements)
		InputUpdateFromSolution(analysis,ug.vector,femmodel.elements[i])
	end

	return ug

end#}}}
function InputDuplicatex(femmodel::FemModel, oldenum::IssmEnum, newenum::IssmEnum) #{{{
	DuplicateInput(femmodel.inputs, oldenum, newenum)
end#}}}
function Reducevectorgtofx(ug::IssmVector,nodes::Vector{Node}) #{{{

	#Get size of output vector
	fsize = NumberOfDofs(nodes,FsetEnum)

	#Initialize output vector
	uf = IssmVector(fsize)

	#Go through elements and plug in solution
	for i=1:length(nodes)
		VecReduce(nodes[i],ug.vector,uf)
	end

	return uf

end#}}}
function Mergesolutionfromftogx(ug::IssmVector, uf::IssmVector, ys::IssmVector, nodes::Vector{Node}) #{{{

	#Go through elements and plug in solution
	for i=1:length(nodes)
		VecMerge(nodes[i],ug,uf.vector,ys.vector)
	end

	return ug

end#}}}
function Reduceloadx!(pf::IssmVector, Kfs::IssmMatrix, ys::IssmVector) #{{{

	#Is there anything to do?
	m, n = GetSize(Kfs)

	if(m*n>0)

		#Allocate Kfs*ys
		Kfsy_s=IssmVector(m)

		#Perform multiplication
		MatMult!(Kfs,ys,Kfsy_s)

		#Subtract Kfs*ys from pf
		AXPY!(pf,-1.0,Kfsy_s)

	end
end#}}}
function SystemMatricesx(femmodel::FemModel,analysis::Analysis)# {{{

	#Allocate matrices
	fsize = NumberOfDofs(femmodel.nodes,FsetEnum)
	ssize = NumberOfDofs(femmodel.nodes,SsetEnum)
	Kff = IssmMatrix(fsize,fsize)
	Kfs = IssmMatrix(fsize,ssize)
	pf  = IssmVector(fsize)

	#Construct Stiffness matrix and load vector from elements
	for i in 1:length(femmodel.elements)
		Ke = CreateKMatrix(analysis,femmodel.elements[i])
		pe = CreatePVector(analysis,femmodel.elements[i])

		AddToGlobal!(Ke,Kff,Kfs)
		AddToGlobal!(pe,pf)
	end

	Assemble!(Kff)
	Assemble!(Kfs)
	Assemble!(pf)
	
	return Kff, Kfs, pf
end# }}}
function CreateNodalConstraintsx(nodes::Vector{Node})# {{{

	#Allocate vector
	ssize=NumberOfDofs(nodes,SsetEnum)
	ys=IssmVector(ssize)

	#constraints vector with the constraint values
	for i in 1:length(nodes)
		CreateNodalConstraints(nodes[i],ys)
	end

	return ys
end# }}}
function RequestedOutputsx(femmodel::FemModel,outputlist::Vector{IssmEnum})# {{{

	#Get Step and Time from parameters
	step = FindParam(femmodel.parameters,StepEnum)
	time = FindParam(femmodel.parameters,TimeEnum)
	
	#Now fetch results
	for i in 1:length(outputlist)

		#See if outputlist[i] is an input
		if outputlist[i]>InputsSTARTEnum && outputlist[i]<InputsENDEnum

			#Create Result
			input  = GetInput(femmodel.inputs,outputlist[i])
			result = Result(outputlist[i], step, time, copy(input.values))

			#Add to femmodel.results dataset
			push!(femmodel.results, result)

		else
			println("Output ",outputlist[i]," not supported yet")
		end
	end
end# }}}
