#MasstransportAnalysis class definition
struct MasstransportAnalysis <: Analysis#{{{
end #}}}

#Model Processing
function CreateConstraints(analysis::MasstransportAnalysis,constraints::Vector{Constraint},md::model) #{{{

	#load constraints from model
	spcthickness = md.masstransport.spcthickness

	count = 1
	for i in 1:md.mesh.numberofvertices
		if ~isnan(spcthickness[i])
			push!(constraints,Constraint(count,i,1,spcthickness[i]))
			count+=1
		end
	end

end#}}}
function CreateNodes(analysis::MasstransportAnalysis,nodes::Vector{Node},md::model) #{{{

	numdof = 1
	for i in 1:md.mesh.numberofvertices
		push!(nodes,Node(i,i,true,numdof,-ones(Int64,numdof), ones(Int64,numdof), -ones(Int64,numdof), zeros(numdof)))
	end
end#}}}
function UpdateElements(analysis::MasstransportAnalysis,elements::Vector{Tria}, inputs::Inputs, md::model) #{{{

	#Provide node indices to element
	for i in 1:md.mesh.numberofelements
		Update(elements[i],inputs,i,md,P1Enum)
	end

	#Add necessary inputs to perform this analysis
	FetchDataToInput(md,inputs,elements,md.geometry.thickness,ThicknessEnum)
	FetchDataToInput(md,inputs,elements,md.geometry.surface,SurfaceEnum)
	FetchDataToInput(md,inputs,elements,md.geometry.base,BaseEnum)
	FetchDataToInput(md,inputs,elements,md.basalforcings.groundedice_melting_rate./md.constants.yts,BasalforcingsGroundediceMeltingRateEnum)
	FetchDataToInput(md,inputs,elements,md.basalforcings.floatingice_melting_rate./md.constants.yts,BasalforcingsFloatingiceMeltingRateEnum)
	FetchDataToInput(md,inputs,elements,md.mask.ice_levelset, MaskIceLevelsetEnum)
	FetchDataToInput(md,inputs,elements,md.mask.ocean_levelset, MaskOceanLevelsetEnum)
	FetchDataToInput(md,inputs,elements,md.initialization.vx./md.constants.yts,VxEnum)
	FetchDataToInput(md,inputs,elements,md.initialization.vy./md.constants.yts,VyEnum)

end#}}}
function UpdateParameters(analysis::MasstransportAnalysis,parameters::Parameters,md::model) #{{{

	AddParam(parameters, md.masstransport.min_thickness, MasstransportMinThicknessEnum)
	AddParam(parameters, md.masstransport.stabilization, MasstransportStabilizationEnum)

end#}}}

#Finite Element Analysis
function Core(analysis::MasstransportAnalysis,femmodel::FemModel)# {{{

	error("not supported yet")

	#Save output
	RequestedOutputsx(femmodel, [ThicknessEnum, SurfaceEnum, BedEnum])

end #}}}
function CreateKMatrix(analysis::MasstransportAnalysis,element::Tria)# {{{

	error("not supported")

	#Internmediaries
	numnodes = 3
	
	#Initialize Element matrix and basis function derivatives
	Ke = ElementMatrix(element.nodes)
	dbasis = Matrix{Float64}(undef,numnodes,2)

	#Retrieve all inputs and parameters
	xyz_list = GetVerticesCoordinates(element.vertices)
	H_input  = GetInput(element, ThicknessEnum)

	#Prepare material object
	material = Matice(element)
	
	#Start integrating
	gauss = GaussTria(2)
	for ig in 1:gauss.numgauss

		Jdet = JacobianDeterminant(xyz_list, gauss)
		NodalFunctionsDerivatives(element,dbasis,xyz_list,gauss)

		H  = GetInputValue(H_input, gauss, ig)
		mu = ViscositySSA(material, xyz_list, gauss, ig)

		for i in 1:numnodes
			for j in 1:numnodes
				Ke.values[2*i-1,2*j-1] += gauss.weights[ig]*Jdet*mu*H*(4*dbasis[j,1]*dbasis[i,1] + dbasis[j,2]*dbasis[i,2])
				Ke.values[2*i-1,2*j  ] += gauss.weights[ig]*Jdet*mu*H*(2*dbasis[j,2]*dbasis[i,1] + dbasis[j,1]*dbasis[i,2])
				Ke.values[2*i  ,2*j-1] += gauss.weights[ig]*Jdet*mu*H*(2*dbasis[j,1]*dbasis[i,2] + dbasis[j,2]*dbasis[i,1])
				Ke.values[2*i  ,2*j  ] += gauss.weights[ig]*Jdet*mu*H*(4*dbasis[j,2]*dbasis[i,2] + dbasis[j,1]*dbasis[i,1])
			end
		end
	end

	#Add basal friction
	phi=GetGroundedPortion(element, xyz_list)

	if(phi>0)
		basis = Vector{Float64}(undef,numnodes)
		friction = CoreFriction(element)

		#Start integrating
		gauss = GaussTria(2)
		for ig in 1:gauss.numgauss

			Jdet = JacobianDeterminant(xyz_list, gauss)
			NodalFunctions(element, basis, gauss, ig, P1Enum)

			alpha2 = Alpha2(friction, gauss, ig)

			for i in 1:numnodes
				for j in 1:numnodes
					Ke.values[2*i-1,2*j-1] += gauss.weights[ig]*Jdet*phi*alpha2*basis[i]*basis[j]
					Ke.values[2*i  ,2*j  ] += gauss.weights[ig]*Jdet*phi*alpha2*basis[i]*basis[j]
				end
			end
		end
	end

	return Ke
end #}}}
function CreatePVector(analysis::MasstransportAnalysis,element::Tria)# {{{

	error("not supported")

	#Internmediaries
	numnodes = 3

	#Initialize Element vectro and basis functions
	pe = ElementVector(element.nodes)
	basis = Vector{Float64}(undef,numnodes)

	#Retrieve all inputs and parameters
	xyz_list = GetVerticesCoordinates(element.vertices)
	H_input  = GetInput(element, ThicknessEnum)
	s_input  = GetInput(element, SurfaceEnum)
	rho_ice  = FindParam(element, MaterialsRhoIceEnum)
	g        = FindParam(element, ConstantsGEnum)

	#Start integrating
	gauss = GaussTria(2)
	for ig in 1:gauss.numgauss

		Jdet = JacobianDeterminant(xyz_list, gauss)
		NodalFunctions(element, basis, gauss, ig, P1Enum)

		H  = GetInputValue(H_input, gauss, ig)
		ds = GetInputDerivativeValue(s_input, xyz_list, gauss, ig)

		for i in 1:numnodes
			pe.values[2*i-1] += -gauss.weights[ig]*Jdet*rho_ice*g*H*ds[1]*basis[i]
			pe.values[2*i  ] += -gauss.weights[ig]*Jdet*rho_ice*g*H*ds[2]*basis[i]
		end
	end

	if(IsIcefront(element))

		#Get additional parameters and inputs
		b_input   = GetInput(element, BaseEnum)
		rho_water = FindParam(element, MaterialsRhoSeawaterEnum)

		#Get normal and ice front coordinates
		xyz_list_front = Matrix{Float64}(undef,2,3)
		GetIcefrontCoordinates!(element, xyz_list_front, xyz_list, MaskIceLevelsetEnum)
		nx, ny = NormalSection(element, xyz_list_front)

		gauss = GaussTria(element, xyz_list, xyz_list_front, 3)
		for ig in 1:gauss.numgauss

			Jdet = JacobianDeterminantSurface(xyz_list_front, gauss)
			NodalFunctions(element, basis, gauss, ig, P1Enum)

			H  = GetInputValue(H_input, gauss, ig)
			b  = GetInputValue(b_input, gauss, ig)
			sl = 0

			term = 0.5*g*rho_ice*H^2 + 0.5*g*rho_water*(min(0, H+b-sl)^2 - min(0, b-sl)^2)

			for i in 1:numnodes
				pe.values[2*i-1] += gauss.weights[ig]*Jdet*term*nx*basis[i]
				pe.values[2*i  ] += gauss.weights[ig]*Jdet*term*ny*basis[i]
			end
		end
	end

	return pe
end #}}}
function GetSolutionFromInputs(analysis::MasstransportAnalysis,ug::IssmVector,element::Tria) #{{{

	#Get dofs for this finite element
	doflist = GetDofList(element,GsetEnum)
	@assert length(doflist)==3

	#Fetch inputs
	thickness_input = GetInput(element, ThicknessEnum)

	#Loop over each node and enter solution in ug
	count = 0
	gauss=GaussTria(P1Enum)
	for i in 1:gauss.numgauss
		thickness = GetInputValue(thickness_input, gauss, i)

		count += 1
		ug.vector[doflist[count]] = thickness
	end

	#Make sure we reached all the values
	@assert count==length(doflist)

end#}}}
function InputUpdateFromSolution(analysis::MasstransportAnalysis,ug::Vector{Float64},element::Tria) #{{{

	#Get dofs for this finite element
	doflist = GetDofList(element,GsetEnum)

	#Get solution vector for this element
	numdof   = 3
	values = Vector{Float64}(undef,numdof)
	for i in 1:numdof values[i]=ug[doflist[i]] end

	#Get some parameters
	rho_water = FindParam(element, MaterialsRhoSeawaterEnum)
	rho_ice   = FindParam(element, MaterialsRhoIceEnum)
	H_min     = FindParam(element, MasstransportMinThicknessEnum)

	#Now split solution vector into x and y components
	numnodes = 3
	thickness  = Vector{Float64}(undef,numnodes)
	for i in 1:numnodes 
		thickness[i]=values[i] 
		@assert isfinite(thickness[i])

		#Enforce minimum thickness
		if(thickness[i]<min_thickness)
			thickness[i] = min_thickness
		end
	end
	AddInput(element, ThicknessEnum,  thickness,  P1Enum)

	#Update bed and surface accordingly
	newthickness = Vector{Float64}(undef,3)
	oldthickness = Vector{Float64}(undef,3)
	oldbase      = Vector{Float64}(undef,3)
	oldsurface   = Vector{Float64}(undef,3)
	phi          = Vector{Float64}(undef,3)
	GetInputListOnVertices!(element, newthickness, ThicknessEnum)
   GetInputListOnVertices!(element, oldthickness, ThicknessOldEnum)
   GetInputListOnVertices!(element, oldbase, BaseOldEnum)
   GetInputListOnVertices!(element, oldsurface, SurfaceOldEnum)
   GetInputListOnVertices!(element, phi, MaskOceanLevelsetEnum)
   sealevel = zeros(3)

   for i in 1:3
      if(phi[i]>0.)
         #this is grounded ice: just add thickness to base.
         newsurface[i] = bed[i]+newthickness[i]; #surface = bed + newthickness
         newbase[i]    = bed[i];                 #new base at new bed
      else
         #this is an ice shelf: hydrostatic equilibrium
         newsurface[i] = newthickness[i]*(1-rho_ice/rho_water) + sealevel[i]
         newbase[i]    = newthickness[i]*(-rho_ice/rho_water)  + sealevel[i]
      end
   end

	AddInput(element, SurfaceEnum, newsurface, P1Enum)
	AddInput(element, BaseEnum,    newbase,    P1Enum)
end#}}}
