#Tria class definition
mutable struct Tria #{{{
	sid::Int64
	pid::Int64
	#vertexids::Int64[3]
	#vertices::Vertex[3]
	vertexids::Vector{Int64}
	vertices::Vector{Vertex}
	nodeids::Vector{Int64}
	nodes::Vector{Node}
	parameters::Parameters
	inputs::Inputs
end# }}}

#Element functions
function InputCreate(element::Tria,inputs::Inputs,data::Vector{Float64},enum::IssmEnum) #{{{
	if size(data,1)==inputs.numberofelements
		SetTriaInput(inputs,enum,P0Enum,element.sid,data[element.sid])
	elseif size(data,1)==inputs.numberofvertices
		SetTriaInput(inputs,enum,P1Enum,element.vertexids,data[element.vertexids])
	else
		error("size ",size(data,1)," not supported yet");
	end
end #}}}
function AddInput(element::Tria,inputenum::IssmEnum,data::Vector{Float64},interpolation::IssmEnum) #{{{
	if interpolation==P1Enum
		@assert length(data)==3
		SetTriaInput(element.inputs,inputenum,P1Enum,element.vertexids,data)
	else
		error("interpolation ", interpolation, " not supported yet");
	end
end #}}}
function Update(element::Tria,inputs::Inputs,index::Int64,md::model,finiteelement::IssmEnum) #{{{

	if finiteelement==P1Enum
		numnodes = 3
		element.nodeids    = Vector{Int64}(undef,numnodes)
		element.nodeids[1] = md.mesh.elements[index,1]
		element.nodeids[2] = md.mesh.elements[index,2]
		element.nodeids[3] = md.mesh.elements[index,3]
	else
		error("not supported yet")
	end
end #}}}
function Configure(element::Tria,nodes::Vector{Node},vertices::Vector{Vertex},parameters::Parameters,inputs::Inputs) # {{{

   #Configure vertices
   for i in 1:3
		element.vertices[i] = vertices[element.vertexids[i]]
   end

	#Configure nodes (assuming P1 finite elements)
	for i in 1:3
		element.nodes[i] = nodes[element.nodeids[i]]
	end

	#Point to real datasets
	element.parameters = parameters
	element.inputs     = inputs
	
end # }}}
function GetDofList(element::Tria,setenum::IssmEnum) # {{{

	#Define number of nodes
	numnodes = 3

	#Determine size of doflist
	numdofs = 0
	for i in 1:numnodes
		numdofs += GetNumberOfDofs(element.nodes[i],GsetEnum)
	end

	#Allocate doflist vector
	doflist = Vector{Int64}(undef,numdofs)

	#enter dofs in doflist vector
	count = 0
	for i in 1:numnodes
		count = GetDofList(element.nodes[i],doflist,count,GsetEnum)
	end

	return doflist
end # }}}
function GetInput(element::Tria,enum::IssmEnum) # {{{

	input = GetInput(element.inputs,enum)
	InputServe!(element,input)
	return input

end # }}}
function GetInputListOnVertices!(element::Tria, vector::Vector{Float64}, enum::IssmEnum) # {{{

	#Get Input first 
	input = GetInput(element, enum)

	#Get value at each vertex (i.e. P1 Nodes)
	gauss=GaussTria(P1Enum)
	for i in 1:gauss.numgauss
		vector[i] = GetInputValue(input, gauss, i)
	end

end # }}}
function FindParam(element::Tria,enum::IssmEnum) # {{{

	return FindParam(element.parameters, enum)

end # }}}
function InputServe!(element::Tria,input::ElementInput) # {{{

	if input.interp==P0Enum
		input.element_values[1] = input.values[element.sid]
	elseif input.interp==P1Enum
		for i in 1:3
			input.element_values[i] = input.values[element.vertices[i].sid]
		end
	else
		error("interpolation ",input.interp," not supported yet")
	end

end # }}}
function GetGroundedPortion(element::Tria, xyz_list::Matrix{Float64}) #{{{

	level = Vector{Float64}(undef,3)
	GetInputListOnVertices!(element, level, MaskOceanLevelsetEnum)

	#Be sure that values are not zero
	epsilon = 1.e-15
	for i in 1:3
		if(level[i]==0.) level[i]=level[i]+epsilon end
	end

	if level[1]>0 && level[2]>0 && level[3]>0
      #Completely grounded
		phi = 1.0
   elseif level[1]<0 && level[2]<0 && level[3]<0
      #Completely floating
      phi = 0.0
   else
		#Partially floating,
		if(level[1]*level[2]>0) #Nodes 0 and 1 are similar, so points must be found on segment 0-2 and 1-2
			s1=level[3]/(level[3]-level[2]);
			s2=level[3]/(level[3]-level[1]);
		elseif(level[2]*level[3]>0) #Nodes 1 and 2 are similar, so points must be found on segment 0-1 and 0-2
			s1=level[1]/(level[1]-level[2]);
			s2=level[1]/(level[1]-level[3]);
		elseif(level[1]*level[3]>0) #Nodes 0 and 2 are similar, so points must be found on segment 1-0 and 1-2
			s1=level[2]/(level[2]-level[1]);
			s2=level[2]/(level[2]-level[3]);
		else
			error("not supposed to be here...")
		end

		if(level[1]*level[2]*level[3]>0)
			phi = s1*s2
		else
			phi = (1-s1*s2)
		end
	end

	return phi
end#}}}
function IsIcefront(element::Tria) #{{{

	level = Vector{Float64}(undef,3)
	GetInputListOnVertices!(element, level, MaskIceLevelsetEnum)

	nbice = 0
	for i in 1:3
		if(level[i]<0.) nbice+=1 end
	end

	if(nbice==1)
		return true
	else
		return false
	end
end#}}}

#Finite Element stuff
function JacobianDeterminant(xyz_list::Matrix{Float64}, gauss::GaussTria) #{{{

	#Get Jacobian Matrix
	J = Jacobian(xyz_list)

	#Get its determinant
	Jdet = Matrix2x2Determinant(J)

	#check and return
	if(Jdet<0) error("negative Jacobian Determinant") end
	return Jdet

end#}}}
function Jacobian(xyz_list::Matrix{Float64}) #{{{

	J = Matrix{Float64}(undef,2,2)

	x1 = xyz_list[1,1]
	y1 = xyz_list[1,2]
	x2 = xyz_list[2,1]
	y2 = xyz_list[2,2]
	x3 = xyz_list[3,1]
	y3 = xyz_list[3,2]

	J[1,1] = .5*(x2-x1)
	J[1,2] = .5*(y2-y1)
	J[2,1] = sqrt(3)/6*(2*x3 -x1 -x2)
	J[2,2] = sqrt(3)/6*(2*y3 -y1 -y2)

	return J
end#}}}
function JacobianInvert(xyz_list::Matrix{Float64}, gauss::GaussTria) #{{{

	#Get Jacobian matrix
	J = Jacobian(xyz_list)

	#Get its determinant
	Jinv = Matrix2x2Invert(J)

	return Jinv
end#}}}
function NodalFunctions(element::Tria,basis::Vector{Float64}, gauss::GaussTria, ig::Int64, finiteelement::IssmEnum) #{{{

	if(finiteelement==P0Enum)
		#Nodal function 1
		basis[1]= 1
	elseif(finiteelement==P1Enum)
		basis[1] = gauss.coords1[ig]
		basis[2] = gauss.coords2[ig]
		basis[3] = gauss.coords3[ig]
	else
		error("Element type ",finiteelement," not supported yet")
	end


end#}}}
function NodalFunctionsDerivatives(element::Tria,dbasis::Matrix{Float64},xyz_list::Matrix{Float64}, gauss::GaussTria) #{{{

	#Get nodal function derivatives in reference element
	dbasis_ref = Matrix{Float64}(undef,3,2)
	NodalFunctionsDerivativesReferenceTria(dbasis_ref,gauss,P1Enum)

	#Get invert of the Jacobian
	Jinv = JacobianInvert(xyz_list,gauss)

	#Build dbasis:
	#[ dNi/dx ] = Jinv * [dNhat_i/dr]
	#[ dNi/dy ] =        [dNhat_i/ds]
	for i in 1:3
		dbasis[i,1] = Jinv[1,1]*dbasis_ref[i,1]+Jinv[1,2]*dbasis_ref[i,2]
		dbasis[i,2] = Jinv[2,1]*dbasis_ref[i,1]+Jinv[2,2]*dbasis_ref[i,2]
	end

end#}}}
function NodalFunctionsDerivativesReferenceTria(dbasis::Matrix{Float64}, gauss::GaussTria, finiteelement::IssmEnum) #{{{

	if(finiteelement==P0Enum)
		#Nodal function 1
		dbasis[1,1]= 0
		dbasis[1,2]= 0

	elseif(finiteelement==P1Enum)
		#Nodal function 1
		dbasis[1,1]= -.5
		dbasis[1,2]= -sqrt(3)/6
		#Nodal function 2
		dbasis[2,1]= .5
		dbasis[2,2]= -sqrt(3)/6
		#Nodal function 3
		dbasis[3,1]= 0
		dbasis[3,2]= sqrt(3)/3
	else
		error("Element type ",finiteelement," not supported yet")
	end
end#}}}
function NumberofNodesTria(finiteelement) #{{{

	if    (finiteelement==P0Enum) return 0
	elseif(finiteelement==P1Enum) return 3
	else
		error("Element type ",finiteelement," not supported yet")
	end
end#}}}
