#StressbalanceAnalysis class definition
struct StressbalanceAnalysis <: Analysis#{{{
end #}}}

#Model Processing
function CreateConstraints(analysis::StressbalanceAnalysis,constraints::Vector{Constraint},md::model) #{{{

	#load constraints from model
	spcvx = md.stressbalance.spcvx
	spcvy = md.stressbalance.spcvy

	count = 1
	for i in 1:md.mesh.numberofvertices
		if ~isnan(spcvx[i])
			push!(constraints,Constraint(count,i,1,spcvx[i]))
			count+=1
		end
		if ~isnan(spcvy[i])
			push!(constraints,Constraint(count,i,2,spcvy[i]))
			count+=1
		end
	end

end#}}}
function CreateNodes(analysis::StressbalanceAnalysis,nodes::Vector{Node},md::model) #{{{

	numdof = 2
	for i in 1:md.mesh.numberofvertices
		push!(nodes,Node(i,i,true,numdof,-ones(Int64,numdof), ones(Int64,numdof), -ones(Int64,numdof), zeros(numdof)))
	end
end#}}}
function UpdateElements(analysis::StressbalanceAnalysis,elements::Vector{Tria}, inputs::Inputs, md::model) #{{{

	#Provide node indices to element
	for i in 1:md.mesh.numberofvertices
		Update(elements[i],inputs,i,md,P1Enum)
	end

	#Add necessary inputs to perform this analysis
	FetchDataToInput(md,inputs,elements,md.materials.rheology_B,MaterialsRheologyBEnum)
	FetchDataToInput(md,inputs,elements,md.geometry.thickness,ThicknessEnum)
	FetchDataToInput(md,inputs,elements,md.geometry.surface,SurfaceEnum)
	FetchDataToInput(md,inputs,elements,md.geometry.base,BaseEnum)
	FetchDataToInput(md,inputs,elements,md.initialization.vx./md.constants.yts,VxEnum)
	FetchDataToInput(md,inputs,elements,md.initialization.vy./md.constants.yts,VyEnum)

end#}}}
function UpdateParameters(analysis::StressbalanceAnalysis,parameters::Parameters,md::model) #{{{
	AddParam(parameters,md.stressbalance.restol,StressbalanceRestolEnum)
	AddParam(parameters,md.stressbalance.reltol,StressbalanceReltolEnum)
	AddParam(parameters,md.stressbalance.abstol,StressbalanceAbstolEnum)
	AddParam(parameters,md.stressbalance.maxiter,StressbalanceMaxiterEnum)
end#}}}

#Finite Element Analysis
function Core(analysis::StressbalanceAnalysis,femmodel::FemModel)# {{{

	#Fetch parameters relevant to solution sequence
	maxiter = FindParam(femmodel.parameters,StressbalanceMaxiterEnum)
	restol  = FindParam(femmodel.parameters,StressbalanceRestolEnum)
	reltol  = FindParam(femmodel.parameters,StressbalanceReltolEnum)
	abstol  = FindParam(femmodel.parameters,StressbalanceAbstolEnum)

	solutionsequence_nonlinear(femmodel,analysis,maxiter,restol,reltol,abstol)
	error("STOP")

end #}}}
function CreateKMatrix(analysis::StressbalanceAnalysis,element::Tria)# {{{

	#Initialize Element matrix
	Ke = ElementMatrix(element.nodes)

	return Ke
end #}}}
function GetSolutionFromInputs(analysis::StressbalanceAnalysis,ug::Vector{Float64},element::Tria) #{{{

	#Get dofs for this finite element
	doflist = GetDofList(element,GsetEnum)
	@assert length(doflist)==6

	#Fetch inputs
	vx_input = GetInput(element, VxEnum)
	vy_input = GetInput(element, VyEnum)

	#Loop over each node and enter solution in ug
	count = 0
	gauss=GaussTria(P1Enum)
	for i in 1:gauss.numgauss
		vx = GetInputValue(vx_input, gauss, i)
		vy = GetInputValue(vy_input, gauss, i)

		count += 1
		ug[doflist[count]] = vx
		count += 1
		ug[doflist[count]] = vy
	end

	#Make sure we reached all the values
	@assert count==length(doflist)

end#}}}
function InputUpdateFromSolution(analysis::StressbalanceAnalysis,ug::Vector{Float64},element::Tria) #{{{

	#Get dofs for this finite element
	doflist = GetDofList(element,GsetEnum)

	#Get solution vector for this element
	numdof   = 3*2
	values = Vector{Float64}(undef,numdof)
	for i in 1:numdof values[i]=ug[doflist[i]] end

	#Now split solution vector into x and y components
	numnodes = 3
	vx= Vector{Float64}(undef,numnodes)
	vy= Vector{Float64}(undef,numnodes)
	for i in 1:numnodes 
		vx[i]=values[2*i-1] 
		vy[i]=values[2*i] 
		@assert isfinite(vx[i])
		@assert isfinite(vy[i])
	end

	AddInput(element,VxEnum,vx,P1Enum)
	AddInput(element,VyEnum,vy,P1Enum)
end#}}}
