include("classes.jl")
include("issmenums.jl")

#define classes first
abstract type Parameter end
struct DoubleParam <: Parameter #{{{
	enum::IssmEnum
	value::Float64
end# }}}
function GetParameterValue(param::DoubleParam) #{{{
	return param.value
end#}}}
struct IntParam <: Parameter #{{{
	enum::IssmEnum
	value::Int64
end# }}}
function GetParameterValue(param::IntParam) #{{{
	return param.value
end#}}}
mutable struct Parameters #{{{
	lookup::Dict{IssmEnum,Parameter}
end# }}}
function AddParam(parameters::Parameters,value::Float64,enum::IssmEnum) #{{{

	parameters.lookup[enum] = DoubleParam(enum,value)

end#}}}
function AddParam(parameters::Parameters,value::Int64,enum::IssmEnum) #{{{

	parameters.lookup[enum] = IntParam(enum,value)

end#}}}
function AddParam(parameters::Parameters,value::Int64,enum::IssmEnum) #{{{

	parameters.lookup[enum] = IntParam(enum,value)

end#}}}
function FindParam(parameters::Parameters,enum::IssmEnum) #{{{

	param = parameters.lookup[enum]
	return GetParameterValue(param)

end#}}}

abstract type Input end
struct BoolInput <: Input#{{{
	enum::IssmEnum
	values::Matrix{Bool}
end# }}}
struct IntInput <: Input#{{{
	enum::IssmEnum
	values::Matrix{Int64}
end# }}}
struct ElementInput <: Input#{{{
	enum::IssmEnum
	interp::IssmEnum
	values::Vector{Float64}
end# }}}
mutable struct Inputs #{{{
	numberofelements::Int64
	numberofvertices::Int64
	lookup::Dict{IssmEnum,Input}
end# }}}
function SetInput(inputs::Inputs,enum::IssmEnum,index::Int64,value::Bool) #{{{

	#Does this input exist
	if !haskey(inputs.lookup,enum)
		#it does not exist yet, we need to create it...
		@assert inputs.numberofelements > 0
		input = BoolInput(enum,zeros(Bool,inputs.numberofelements))
		inputs.lookup[enum] = BoolInput(enum,zeros(Bool,inputs.numberofelements))
	end

	#Get this input and check type
	input = inputs.lookup[enum]
	if typeof(input)!=BoolInput error("not consistent") end

	#set value
	@assert index>0 && index<=inputs.numberofelements
	input.values[index] = value
end#}}}
function SetTriaInput(inputs::Inputs,enum::IssmEnum,interp::IssmEnum,index::Int64,value::Float64) #{{{

	#Does this input exist
	if !haskey(inputs.lookup,enum)
		#it does not exist yet, we need to create it...
		@assert inputs.numberofelements > 0
		if interp==P0Enum
			inputs.lookup[enum] = ElementInput(enum,interp,zeros(inputs.numberofelements))
		elseif interp==P1Enum
			inputs.lookup[enum] = ElementInput(enum,interp,zeros(inputs.numberofvertices))
		else
			error("not supported yet")
		end
	end

	#Get this input and check type
	input = inputs.lookup[enum]
	if typeof(input)!=ElementInput error("input type not consistent") end
	if interp!=input.interp        error("input interpolations not consistent") end

	#set value
	input.values[index] = value
end#}}}
function SetTriaInput(inputs::Inputs,enum::IssmEnum,interp::IssmEnum,indices::Vector{Int64},values::Vector{Float64}) #{{{

	#Does this input exist
	if !haskey(inputs.lookup,enum)
		#it does not exist yet, we need to create it...
		@assert inputs.numberofvertices>0
		if interp==P1Enum
			inputs.lookup[enum] = ElementInput(enum,interp,zeros(inputs.numberofvertices))
		else
			error("not supported yet")
		end
	end

	#Get this input and check type
	input = inputs.lookup[enum]
	if typeof(input)!=ElementInput error("input type not consistent") end
	if interp!=input.interp        error("input interpolations not consistent") end

	#set value
	input.values[indices] = values
end#}}}

mutable struct Node #{{{
	id::Int64
	sid::Int64
	indexingupdate::Bool
	gsize::Int64
	gdoflist::Vector{Int64}
	fdoflist::Vector{Int64}
	sdoflist::Vector{Int64}
	svalues::Vector{Float64}
end# }}}
function Base.show(io::IO, this::Node)# {{{

	println(io,"Node:")
	println(io,"   id:  ",this.id)
	println(io,"   sid: ",this.sid)
	println(io,"   indexingupdate: ",this.indexingupdate)
	println(io,"   gsize: ",this.gsize)
	println(io,"   gdoflist: ",this.gdoflist)
	println(io,"   fdoflist: ",this.fdoflist)
	println(io,"   sdoflist: ",this.sdoflist)
	println(io,"   svalues: ",this.svalues)
end# }}}
function ApplyConstraint(node::Node,dof::Int8,value::Float64) #{{{

	node.indexingupdate = true
	node.fdoflist[dof]  = -1
	node.sdoflist[dof]  = +1
	node.svalues[dof]   = value

end# }}}
struct Constraint #{{{
	id::Int64
	nodeid::Int64
	dof::Int8
	value::Float64
end# }}}
function ConstrainNode(constraint::Constraint,nodes::Vector{Node},parameters::Parameters) #{{{

	#Chase through nodes and find the node to which this SpcStatic apply
	node = nodes[constraint.nodeid]

	#Apply Constraint
	ApplyConstraint(node,constraint.dof,constraint.value)

end# }}}
mutable struct Vertex#{{{
	sid::Int64
	x::Float64
	y::Float64
end# }}}
mutable struct Tria #{{{
	sid::Int64
	pid::Int64
	#vertexids::Int64[3]
	#vertices::Vertex[3]
	vertexids::Vector{Int64}
	vertices::Vector{Vertex}
	nodeids::Vector{Int64}
	nodes::Vector{Node}
	parameters::Parameters
	inputs::Inputs
end# }}}
mutable struct FemModel #{{{
	elements::Vector{Tria}
	vertices::Vector{Vertex}
	nodes::Vector{Node}
	parameters::Parameters
	inputs::Inputs
	constraints::Vector{Constraint}
	#loads::Vector{Loads}
end#}}}
struct GaussTria #{{{
	numgauss::Int64
	weights::Vector{Float64}
	coords1::Vector{Float64}
	coords2::Vector{Float64}
	coords3::Vector{Float64}
end #}}}
function GaussTria(order::Int64) #{{{

	#=Gauss quadrature points for the triangle.
	Higher-order points from D.A. Dunavant, "High Degree Efficient
	Symmetrical Gaussian Quadrature Rules for the Triangle", IJNME,
	Vol. 21, pp. 1129-1148 (1985), as transcribed for Probe rules3.=#

	if(order==1)
		npoints = 1
		weights = [1.732050807568877]
		coords1 = [0.333333333333333]
		coords2 = [0.333333333333333]
		coords3 = [0.333333333333333]
	elseif(order==2)
		weights = [0.577350269189625, 0.577350269189625, 0.577350269189625]
		coords1 = [0.666666666666667, 0.166666666666667, 0.166666666666667]
		coords2 = [0.166666666666667, 0.666666666666667, 0.166666666666667]
		coords3 = [0.166666666666667, 0.166666666666667, 0.666666666666667]
	elseif(order==3)
		npoints=4
		weights = [-0.974278579257493, 0.902109795608790, 0.902109795608790, 0.902109795608790]
		coords1 = [0.333333333333333, 0.600000000000000, 0.200000000000000, 0.200000000000000]
		coords2 = [0.333333333333333, 0.200000000000000, 0.600000000000000, 0.200000000000000]
		coords3 = [0.333333333333333, 0.200000000000000, 0.200000000000000, 0.600000000000000]
	else
		error("order ",order," not supported yet");
	end

	return GaussTria(npoints,weights,coords1,coords2,coords3)
end# }}}

abstract type Analysis end
struct StressbalanceAnalysis <: Analysis#{{{
end #}}}

#Modules
function FetchDataToInput(md::model,inputs::Inputs,elements::Vector{Tria},data::Vector{Float64},enum::IssmEnum) #{{{
	for i in 1:length(elements)
		InputCreate(elements[i],inputs,data,enum)
	end
end#}}}
function IssmCore(md::model) #{{{
	
	#Construct FemModel
	femmodel=ModelProcessor(md)

	#Solve
	analysis = StressbalanceAnalysis()
	Core(analysis,femmodel)

	#then what??
	error("STOP")

end# }}}
function ModelProcessor(md::model) #{{{

	#Initialize structures
	elements    = Vector{Tria}(undef,0)
	vertices    = Vector{Vertex}(undef,0)
	nodes       = Vector{Node}(undef,0)
	constraints = Vector{Constraint}(undef,0)
	parameters  = Parameters(Dict{IssmEnum,Parameter}())
	inputs      = Inputs(md.mesh.numberofelements,md.mesh.numberofvertices,Dict{IssmEnum,Input}())

	#Create  elements, vertices and materials (independent of the analysis)
	CreateElements(elements,md)
	CreateVertices(vertices,md)
	CreateParameters(parameters,md)
	CreateInputs(inputs,elements,md)

	#Now create analysis specific data structure
	analysis = StressbalanceAnalysis()
	UpdateParameters(analysis,parameters,md)
	CreateNodes(analysis,nodes,md)
	UpdateElements(analysis,elements,inputs,md)
	CreateConstraints(analysis,constraints,md)

	#Configure objects
	ConfigureObjectx(elements,nodes,vertices,parameters,inputs)

	#Constrain and Number nodes
	SpcNodesx(nodes,constraints,parameters)
	NodesDofx(nodes,parameters)
   error("STOP")

	#Build FemModel
	return FemModel(elements,vertices,nodes,parameters,inputs,constraints)
end# }}}
function CreateElements(elements::Vector{Tria},md::model) #{{{

	#Make sure elements is currently empty
	@assert length(elements)==0

	tempparams   = Parameters(Dict{IssmEnum,Parameter}())
	tempinputs   = Inputs(-1,-1,Dict{IssmEnum,Input}())
	tempvertices = Vector{Vertex}(undef,3)
	tempnodes    = Vector{Node}(undef,3)

	count = 0
	for i in 1:md.mesh.numberofelements

		#Assume Linear Elements for now
		vertexids = md.mesh.elements[i,:]
		nodeids   = md.mesh.elements[i,:]

		#Call constructor and add to dataset elements
		push!(elements,Tria(i,count,vertexids,tempvertices,nodeids,tempnodes,tempparams,tempinputs))
	end

end# }}}
function CreateVertices(vertices::Vector{Vertex},md::model) #{{{

	#Make sure vertices is currently empty
	@assert length(vertices)==0

	#Get data from md
	x = md.mesh.x
	y = md.mesh.y

	count = 0
	for i in 1:md.mesh.numberofvertices
		push!(vertices,Vertex(i,x[i],y[i]))
	end

end# }}}
function CreateParameters(parameters::Parameters,md::model) #{{{

	#Get data from md
	AddParam(parameters,md.materials.rho_ice,MaterialsRhoIceEnum)
	AddParam(parameters,md.materials.rho_water,MaterialsRhoSeawaterEnum)
	AddParam(parameters,md.materials.rho_freshwater,MaterialsRhoFreshwaterEnum)
end# }}}
function CreateInputs(inputs::Inputs,elements::Vector{Tria},md::model) #{{{

	#Only assume we have Matice for now
	FetchDataToInput(md,inputs,elements,md.materials.rheology_B,MaterialsRheologyBEnum)
	FetchDataToInput(md,inputs,elements,md.materials.rheology_n,MaterialsRheologyNEnum)
end# }}}
function ConfigureObjectx(elements::Vector{Tria},nodes::Vector{Node},vertices::Vector{Vertex},parameters::Parameters,inputs::Inputs) #{{{

	for i in 1:length(elements)
		Configure(elements[i],nodes,vertices,parameters,inputs)
	end

end# }}}
function SpcNodesx(nodes::Vector{Node},constraints::Vector{Constraint},parameters::Parameters) #{{{

	for i in 1:length(constraints)
		ConstrainNode(constraints[i],nodes,parameters)
	end

end# }}}
function NodesDofx(nodes::Vector{Node}, parameters::Parameters) #{{{

	error("not implemented yet")

end# }}}

#Analyses
function UpdateParameters(analysis::StressbalanceAnalysis,parameters::Parameters,md::model) #{{{
	AddParam(parameters,md.stressbalance.restol,StressbalanceRestolEnum)
	AddParam(parameters,md.stressbalance.reltol,StressbalanceReltolEnum)
	AddParam(parameters,md.stressbalance.abstol,StressbalanceAbstolEnum)
	AddParam(parameters,md.stressbalance.maxiter,StressbalanceMaxiterEnum)
end#}}}
function CreateNodes(analysis::StressbalanceAnalysis,nodes::Vector{Node},md::model) #{{{

	numdof = 2
	for i in 1:md.mesh.numberofvertices
		push!(nodes,Node(i,i,true,numdof,-ones(Int64,numdof), ones(Int64,numdof), -ones(Int64,numdof), zeros(numdof)))
	end
end#}}}
function UpdateElements(analysis::StressbalanceAnalysis,elements::Vector{Tria}, inputs::Inputs, md::model) #{{{

	#Provide node indices to element
	for i in 1:md.mesh.numberofvertices
		Update(elements[i],inputs,i,md,P1Enum)
	end

	#Add necessary inputs to perform this analysis
	FetchDataToInput(md,inputs,elements,md.materials.rheology_B,MaterialsRheologyBEnum)
	FetchDataToInput(md,inputs,elements,md.geometry.thickness,ThicknessEnum)
	FetchDataToInput(md,inputs,elements,md.geometry.surface,SurfaceEnum)
	FetchDataToInput(md,inputs,elements,md.geometry.base,BaseEnum)
	FetchDataToInput(md,inputs,elements,md.initialization.vx./md.constants.yts,VxEnum)
	FetchDataToInput(md,inputs,elements,md.initialization.vy./md.constants.yts,VyEnum)

end#}}}
function CreateConstraints(analysis::StressbalanceAnalysis,constraints::Vector{Constraint},md::model) #{{{

	#load constraints from model
	spcvx = md.stressbalance.spcvx
	spcvy = md.stressbalance.spcvy

	count = 1
	for i in 1:md.mesh.numberofvertices
		if ~isnan(spcvx[i])
			push!(constraints,Constraint(count,i,1,spcvx[i]))
			count+=1
		end
		if ~isnan(spcvy[i])
			push!(constraints,Constraint(count,i,2,spcvy[i]))
			count+=1
		end
	end

end#}}}
function Core(analysis::StressbalanceAnalysis,femmodel::FemModel)# {{{

	#Fetch parameters relevant to solution sequence
	maxiter = FindParam(femmodel.parameters,StressbalanceMaxiterEnum)
	restol  = FindParam(femmodel.parameters,StressbalanceRestolEnum)
	reltol  = FindParam(femmodel.parameters,StressbalanceReltolEnum)
	abstol  = FindParam(femmodel.parameters,StressbalanceAbstolEnum)

	solutionsequence_nonlinear(femmodel,analysis,maxiter,restol,reltol,abstol)
	error("STOP")

end #}}}
function solutionsequence_nonlinear(femmodel::FemModel,analysis::Analysis,maxiter::Int64,restol::Float64,reltol::Float64,abstol::Float64) # {{{

	#Initialize number of iterations
	count = 0
	converged = false

	#Get existing solution
	ug = GetSolutionFromInputsx(analysis,femmodel)

	#Loop until we reach convergence
	while(~converged)

		error("not implemented yet")

		#Increase count
		count += 1
		if(count>maxiter)
			println("   maximum number of nonlinear iterations (",maxiter,") exceeded")
			converged = true
		end
	end

	error("STOP")

end# }}}

#Element functions
function InputCreate(element::Tria,inputs::Inputs,data::Vector{Float64},enum::IssmEnum) #{{{
	if size(data,1)==inputs.numberofelements
		SetTriaInput(inputs,enum,P0Enum,element.sid,data[element.sid])
	elseif size(data,1)==inputs.numberofvertices
		SetTriaInput(inputs,enum,P1Enum,element.vertexids,data[element.vertexids])
	else
		error("size ",size(data,1)," not supported yet");
	end
end #}}}
function Update(element::Tria,inputs::Inputs,index::Int64,md::model,finiteelement::IssmEnum) #{{{

	if finiteelement==P1Enum
		numnodes = 3
		element.nodeids    = Vector{Int64}(undef,numnodes)
		element.nodeids[1] = md.mesh.elements[index,1]
		element.nodeids[2] = md.mesh.elements[index,2]
		element.nodeids[3] = md.mesh.elements[index,3]
	else
		error("not supported yet")
	end
end #}}}
function Configure(element::Tria,nodes::Vector{Node},vertices::Vector{Vertex},parameters::Parameters,inputs::Inputs) # {{{

   #Configure vertices
   for i in 1:3
      element.vertices[i] = vertices[element.vertexids[i]]
   end

	#Configure nodes (assuming P1 finite elements)
	for i in 1:3
		element.nodes[i] = nodes[element.nodeids[i]]
	end

	#Point to real datasets
	element.parameters = parameters
	element.inputs     = inputs
	
end # }}}
