/*!\file: solutionsequence_sampling.cpp
 * \brief: numerical core of linear solutions
 */

#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../analyses/analyses.h"
#include "../shared/Random/randomgenerator.h"
#include <random>

void GaussianVector(Vector<IssmDouble>* ppf,int seed){/*{{{*/


	/*Intermediaries*/
	double      rdnumber;

	/*Define seed*/
	rdngen::normal_distribution_rnd distribution;
	if(seed<0){
		std::random_device rd;
		seed = rd();
	}
	else
	{
		int my_rank;
		ISSM_MPI_Comm_rank(ISSM_MPI_COMM_WORLD,&my_rank);
		seed = seed + 783728*my_rank; // change default seed for parallel simulations (by considering an arbitrary shif based on the rank number)
	}
	distribution.seed(seed);

	int        *local_indices = NULL;
	IssmDouble *local_vector = NULL;
	ppf->GetLocalVector(&local_vector,&local_indices);

  int M;
	ppf->GetLocalSize(&M);
	for(int i=0;i<M;i++){
		rdnumber = distribution.generator();
		ppf->SetValue(local_indices[i],rdnumber,INS_VAL);
	}
	ppf->Assemble();

}/*}}}*/
void solutionsequence_sampling(FemModel* femmodel){

	/*intermediary: */
  Matrix<IssmDouble>*  Kff = NULL;
	Matrix<IssmDouble>*  Kfs = NULL;
  Vector<IssmDouble>*  ug  = NULL;
  Vector<IssmDouble>*  uf  = NULL;
  Vector<IssmDouble>*  old_uf  = NULL;     // previous solution vector (for transient)
  Vector<IssmDouble>*  pf  = NULL;
  Vector<IssmDouble>*  df  = NULL;
  Vector<IssmDouble>*  ys=NULL;

  SamplingAnalysis*    analysis = NULL;

  Vector<IssmDouble>*  Ml = NULL;      // diagonal lumped mass matrix
  Vector<IssmDouble>*  Mscale = NULL;  // square root of diagonal lumped factor matrix

  /*parameters:*/
  int solution_type, alpha, step, seed, nsize;
  IssmDouble phi, tau;

  /*Recover parameters: */
  femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
  femmodel->parameters->FindParam(&seed,SamplingSeedEnum);
  femmodel->parameters->FindParam(&alpha,SamplingAlphaEnum);
  femmodel->parameters->FindParam(&tau,SamplingTauEnum);

  /*Recover parameters for transient simulation: */
  if(solution_type==TransientSolutionEnum)
  {
    femmodel->parameters->FindParam(&step,StepEnum);
    femmodel->parameters->FindParam(&phi,SamplingPhiEnum);
		if(seed>=0) seed = seed + 13923272*step; // change default seed for transient simulations (by considering an arbitrary shif based on the step number)

    GetSolutionFromInputsx(&ug,femmodel);
    Reducevectorgtofx(&uf, ug, femmodel->nodes,femmodel->parameters);
    old_uf=uf->Duplicate();
  }

  /*CreateAnalysis*/

  analysis = new SamplingAnalysis();

  analysis->LumpedMassMatrix(&Ml,femmodel); //Create lumped mass matrix

  if(alpha%2!=0)  analysis->LumpedKMatrix(&Mscale,femmodel);   /* Compute square root of lump mass matrix (for alpha even) or stiffness matrix (for alpha odd) */
  else{
    Mscale=Ml->Duplicate();
    Ml->Copy(Mscale);
  }
  Mscale->Pow(0.5);

  femmodel->UpdateConstraintsx();
  SystemMatricesx(&Kff, &Kfs, &pf, &df, NULL,femmodel);
  CreateNodalConstraintsx(&ys,femmodel->nodes);
  Reduceloadx(pf, Kfs, ys);
  delete Kfs;

  /*Copy old solution for transient run */
  if(solution_type==TransientSolutionEnum) uf->Copy(old_uf);

  /* Generate random RHS */
  GaussianVector(pf,seed);

  /* Multiply random RHS by square root of mass matrix (for alpha even) or stiffness matrix (for alpha odd) */
  pf->PointwiseMult(pf,Mscale);

  /*Go solve SPDE */
  femmodel->profiler->Start(SOLVER);
  Solverx(&uf, Kff, pf, NULL, df, femmodel->parameters);
  femmodel->profiler->Stop(SOLVER);

  /* Iterate to compute a realization for alpha>2 */
  for(int i=3;i<=alpha;i+=2){

    /*Create RHS */
    uf->Copy(pf);
    pf->PointwiseMult(pf,Ml);

    /*Go solve SPDE */
    femmodel->profiler->Start(SOLVER);
    Solverx(&uf, Kff, pf, NULL, df, femmodel->parameters);
    femmodel->profiler->Stop(SOLVER);

  }

  /* Divide results by tau */
  uf->Scale(1.0/tau);

  /* Update solution x_{t+1} = phi x_{t} + noise for transient */
  if(solution_type==TransientSolutionEnum) uf->AXPY(old_uf,phi);

  /* Update input */
  Mergesolutionfromftogx(&ug, uf,ys,femmodel->nodes,femmodel->parameters);
  InputUpdateFromSolutionx(femmodel,ug);

  /*clean-up*/
  delete Kff; delete pf; delete df; delete uf; delete ys; delete old_uf;
  delete Ml; delete Mscale;
  delete analysis;
  delete ug;

}
