%FRONTAL FORCINGS rignot autoregression class definition
%
%   Usage:
%      frontalforcingsrignotautoregression=frontalforcingsrignotautoregression();

classdef frontalforcingsrignotautoregression
	properties (SetAccess=public) 
		num_basins           = 0;
		beta0                = NaN;
		beta1                = NaN;
		ar_order             = 0;
		ar_initialtime       = 0;
		ar_timestep          = 0;
		phi                  = NaN;
		basin_id             = NaN;
		subglacial_discharge = NaN;
	end
	methods
		function self = frontalforcingsrignot(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					inputstruct=varargin{1};
					list1 = properties('frontalforcingsrignotautoregression');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							self.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = extrude(self,md) % {{{
		    % nothing for now
		end % }}}
		function self = setdefaultparameters(self) % {{{

			self.basin_id             = NaN;
			self.num_basins           = 0;
			self.subglacial_discharge = NaN;
			self.ar_order             = 0.0; %autoregression model of order 0

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{
			%Early return
			if (~strcmp(solution,'TransientSolution') | md.transient.ismovingfront==0), return; end

			md = checkfield(md,'fieldname','frontalforcings.num_basins','numel',1,'NaN',1,'Inf',1,'>',0);
         md = checkfield(md,'fieldname','frontalforcings.basin_id','Inf',1,'>=',0,'<=',md.frontalforcings.num_basins,'size',[md.mesh.numberofelements 1]);
			md = checkfield(md,'fieldname','frontalforcings.subglacial_discharge','>=',0,'NaN',1,'Inf',1,'timeseries',1);
			md = checkfield(md,'fieldname','frontalforcings.beta0','NaN',1,'Inf',1,'size',[1,md.frontalforcings.num_basins],'numel',md.frontalforcings.num_basins); 
         md = checkfield(md,'fieldname','frontalforcings.beta1','NaN',1,'Inf',1,'size',[1,md.frontalforcings.num_basins],'numel',md.frontalforcings.num_basins); 
         md = checkfield(md,'fieldname','frontalforcings.ar_order','numel',1,'NaN',1,'Inf',1,'>=',0);
         md = checkfield(md,'fieldname','frontalforcings.ar_initialtime','numel',1,'NaN',1,'Inf',1);
         md = checkfield(md,'fieldname','frontalforcings.ar_timestep','numel',1,'NaN',1,'Inf',1,'>=',md.timestepping.time_step); %autoregression time step cannot be finer than ISSM timestep
			md = checkfield(md,'fieldname','frontalforcings.phi','NaN',1,'Inf',1,'size',[md.frontalforcings.num_basins,md.frontalforcings.ar_order]);

		end % }}}
		function disp(self) % {{{
			disp(sprintf('   Frontalforcings parameters:'));
			fielddisplay(self,'num_basins','number of different basins [unitless]');
         fielddisplay(self,'basin_id','basin number assigned to each element [unitless]');
         fielddisplay(self,'subglacial_discharge','sum of subglacial discharge for each basin [m/d]');
         fielddisplay(self,'beta0','basin-specific intercept values [∘C]');
         fielddisplay(self,'beta1','basin-specific trend values [∘C yr^(-1)]');
         fielddisplay(self,'ar_order','order of the autoregressive model [unitless]');
         fielddisplay(self,'ar_initialtime','initial time assumed in the autoregressive model parameterization [yr]');
         fielddisplay(self,'ar_timestep','time resolution of the autoregressive model [yr]');
         fielddisplay(self,'phi','basin-specific vectors of lag coefficients [unitless]');
		end % }}}
		function marshall(self,prefix,md,fid) % {{{
			yts=md.constants.yts;
			WriteData(fid,prefix,'name','md.frontalforcings.parameterization','data',55,'format','Integer');
			WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','num_basins','format','Integer');
			WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','subglacial_discharge','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'yts',yts);
         WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','ar_order','format','Integer');
         WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','ar_initialtime','format','Double','scale',yts);
         WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','ar_timestep','format','Double','scale',yts);
         WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','basin_id','data',self.basin_id-1,'name','md.frontalforcings.basin_id','format','IntMat','mattype',2); %0-indexed
         WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','beta0','format','DoubleMat','name','md.frontalforcings.beta0');
         WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','beta1','format','DoubleMat','name','md.frontalforcings.beta1','scale',1./yts,'yts',yts);
         WriteData(fid,prefix,'object',self,'class','frontalforcings','fieldname','phi','format','DoubleMat','name','md.frontalforcings.phi','yts',yts);
		end % }}}
	end
end
