/*!\file StochasticForcingx
 * \brief: compute noise terms for the StochasticForcing fields
 */

#include "./StochasticForcingx.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"
#include "../../shared/Random/random.h"

void StochasticForcingx(FemModel* femmodel){/*{{{*/

   /*Retrieve parameters*/
   bool randomflag;
   int M,N,numfields,my_rank;
   int* fields            = NULL;
   int* dimensions        = NULL;
   IssmDouble* covariance = NULL;
   femmodel->parameters->FindParam(&randomflag,StochasticForcingRandomflagEnum);
   femmodel->parameters->FindParam(&numfields,StochasticForcingNumFieldsEnum);
   femmodel->parameters->FindParam(&fields,&N,StochasticForcingFieldsEnum);    _assert_(N==numfields);
   femmodel->parameters->FindParam(&dimensions,&N,StochasticForcingDimensionsEnum);    _assert_(N==numfields);
   int dimtot=0;
   for(int i=0;i<numfields;i++) dimtot = dimtot+dimensions[i];
   femmodel->parameters->FindParam(&covariance,&M,&N,StochasticForcingCovarianceEnum); _assert_(M==dimtot); _assert_(N==dimtot);

   /*Compute noise terms*/
   IssmDouble* noiseterms = xNew<IssmDouble>(dimtot);
   my_rank=IssmComm::GetRank();
   if(my_rank==0){
      int fixedseed;
      IssmDouble time,dt,starttime;
      femmodel->parameters->FindParam(&time,TimeEnum);
      femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
      femmodel->parameters->FindParam(&starttime,TimesteppingStartTimeEnum);
		/*Determine whether random seed is fixed to time step (randomflag==false) or random seed truly random (randomflag==true)*/
      if(randomflag) fixedseed=-1;
      else fixedseed = reCast<int,IssmDouble>((time-starttime)/dt);
      /*multivariateNormal needs to be passed a NULL pointer to avoid memory leak issues*/
      IssmDouble* temparray = NULL;
      multivariateNormal(&temparray,dimtot,0.0,covariance,fixedseed);
      for(int i=0;i<dimtot;i++) noiseterms[i]=temparray[i];
      xDelete<IssmDouble>(temparray);
   }
   ISSM_MPI_Bcast(noiseterms,dimtot,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());
   
	int i=0;
   for(int j=0;j<numfields;j++){
      int enum_type;
      IssmDouble* noisefield = xNew<IssmDouble>(dimensions[j]);
      for(int k=0;k<dimensions[j];k++){
         noisefield[k]=noiseterms[i+k];
      }
      
      if(fields[j]==SMBautoregressionEnum)                        enum_type = SmbAutoregressionNoiseEnum;
		else if(fields[j]==FrontalForcingsRignotAutoregressionEnum) enum_type = ThermalforcingAutoregressionNoiseEnum;
		else _error_("Field "<<EnumToStringx(fields[j])<<" does not support stochasticity yet.\n"); 
      femmodel->parameters->SetParam(noisefield,dimensions[j],enum_type);
      i=i+dimensions[j];
      xDelete<IssmDouble>(noisefield);
   }

	/*Cleanup*/
   xDelete<int>(fields);
   xDelete<int>(dimensions);
   xDelete<IssmDouble>(covariance);
   xDelete<IssmDouble>(noiseterms);
}/*}}}*/
