%STOCHASTICFORCING class definition
%
%   Usage:
%      stochasticforcing=stochasticforcing();

classdef stochasticforcing
	properties (SetAccess=public) 
		isstochasticforcing  = 0;
		fields               = NaN;
		dimensions           = NaN;
		covariance           = NaN;
		randomflag           = 1;
	end
	methods
		function self = stochasticforcing(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = extrude(self,md) % {{{
			%Nothing for now
		end % }}}
		function self = setdefaultparameters(self) % {{{
			self.isstochasticforcing  = 0; %stochasticforcing is turned off by default
			self.randomflag           = 1; %true randomness is implemented by default
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			%Early return
			if ~self.isstochasticforcing, return; end

			num_fields = numel(self.fields);
			size_tot   = sum(self.dimensions);

			md = checkfield(md,'fieldname','stochasticforcing.isstochasticforcing','values',[0 1]);
			md = checkfield(md,'fieldname','stochasticforcing.fields','numel',num_fields,'cell',1,'values',supportedstochforcings()); %VV check here 'cell' (19Oct2021)
			md = checkfield(md,'fieldname','stochasticforcing.dimensions','NaN',1,'Inf',1,'>',0,'size',[1,num_fields]); %specific dimension for each field
			md = checkfield(md,'fieldname','stochasticforcing.covariance','NaN',1,'Inf',1,'size',[size_tot,size_tot]); %global covariance matrix
			md = checkfield(md,'fieldname','stochasticforcing.randomflag','numel',[1],'values',[0 1]);

			%Check that all fields agree with the corresponding md class	
			for field=self.fields
				if(contains(field,'SMB'))
					if~(isequal(class(md.smb),char(field)))
						error('md.smb does not agree with stochasticforcing field %s', char(field));
					end
				end
				if(contains(field,'frontalforcings'))
					if~(isequal(class(md.frontalforcings),char(field)))
						error('md.frontalforcings does not agree with stochasticforcing field %s', char(field));
					end
				end
			end
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   stochasticforcing parameters:'));
			fielddisplay(self,'isstochasticforcing','is stochasticity activated?');
			fielddisplay(self,'fields','fields with stochasticity applied, ex: {''SMBautoregression''}, or {''FrontalForcingsRignotAutoregression''}');
			fielddisplay(self,'dimensions','dimensionality of each field');
			fielddisplay(self,'covariance','covariance matrix for within- and between-fields covariance (units must be squared field units)');
			fielddisplay(self,'randomflag','whether to apply real randomness (true) or pseudo-randomness with fixed seed (false)');
			disp('Available fields:');
			disp('   SMBautoregression');
			disp('   FrontalForcingsRignotAutoregression (thermal forcing)');
		end % }}}
		function marshall(self,prefix,md,fid) % {{{

			yts=md.constants.yts;
			num_fields = numel(self.fields);
			%Scaling covariance matrix (scale column-by-column and row-by-row)
			scaledfields = {'SMBautoregression'}; %list of fields that need scaling *1/yts
			for i=1:num_fields
            if any(strcmp(scaledfields,self.fields(i)))
               inds = [1+sum(self.dimensions(1:i-1)):1:sum(self.dimensions(1:i))];
               for row=inds %scale rows corresponding to scaled field
                  self.covariance(row,:) = 1./yts*self.covariance(row,:);
               end
               for col=inds %scale columns corresponding to scaled field
                  self.covariance(:,col) = 1./yts*self.covariance(:,col);
               end
            end
         end

			WriteData(fid,prefix,'object',self,'fieldname','isstochasticforcing','format','Boolean');
			if ~self.isstochasticforcing 
				return
			else
				WriteData(fid,prefix,'data',num_fields,'name','md.stochasticforcing.num_fields','format','Integer');
				WriteData(fid,prefix,'object',self,'fieldname','fields','format','StringArray');
				WriteData(fid,prefix,'object',self,'fieldname','dimensions','format','IntMat');
				WriteData(fid,prefix,'object',self,'fieldname','covariance','format','DoubleMat');
				WriteData(fid,prefix,'object',self,'fieldname','randomflag','format','Boolean');
			end
		end % }}}
	end
end
