include("issm.jl")
include("issmenums.jl")

#define classes first
abstract type Parameter end
struct DoubleParam <: Parameter #{{{
	enum::IssmEnum
	value::Float64
end# }}}
struct IntParam <: Parameter #{{{
	enum::IssmEnum
	value::Int64
end# }}}
mutable struct Parameters #{{{
	lookup::Dict{IssmEnum,Parameter}
end# }}}
function AddParam(parameters::Parameters,value::Float64,enum::IssmEnum) #{{{

	parameters.lookup[enum] = DoubleParam(enum,value)

end#}}}
function AddParam(parameters::Parameters,value::Int64,enum::IssmEnum) #{{{

	parameters.lookup[enum] = IntParam(enum,value)

end#}}}

abstract type Input end
struct BoolInput <: Input#{{{
	enum::IssmEnum
	values::Matrix{Bool}
end# }}}
struct IntInput <: Input#{{{
	enum::IssmEnum
	values::Matrix{Int32}
end# }}}
struct ElementInput <: Input#{{{
	enum::IssmEnum
	interp::IssmEnum
	values::Vector{Float64}
end# }}}
mutable struct Inputs #{{{
	numberofelements::Int32
	numberofvertices::Int32
	lookup::Dict{IssmEnum,Input}
end# }}}
function SetInput(inputs::Inputs,enum::IssmEnum,index::Int32,value::Bool) #{{{

	#Does this input exist
	if !haskey(inputs.lookup,enum)
		#it does not exist yet, we need to create it...
		@assert inputs.numberofelements > 0
		input = BoolInput(enum,zeros(Bool,inputs.numberofelements))
		inputs.lookup[enum] = BoolInput(enum,zeros(Bool,inputs.numberofelements))
	end

	#Get this input and check type
	input = inputs.lookup[enum]
	if typeof(input)!=BoolInput error("not consistent") end

	#set value
	@assert index>0 && index<=inputs.numberofelements
	input.values[index] = value
end#}}}
function SetTriaInput(inputs::Inputs,enum::IssmEnum,interp::IssmEnum,index::Int32,value::Float64) #{{{

	#Does this input exist
	if !haskey(inputs.lookup,enum)
		#it does not exist yet, we need to create it...
		@assert inputs.numberofelements > 0
		if interp==P0Enum
			inputs.lookup[enum] = ElementInput(enum,interp,zeros(inputs.numberofelements))
		elseif interp==P1Enum
			inputs.lookup[enum] = ElementInput(enum,interp,zeros(inputs.numberofvertices))
		else
			error("not supported yet")
		end
	end

	#Get this input and check type
	input = inputs.lookup[enum]
	if typeof(input)!=ElementInput error("input type not consistent") end
	if interp!=input.interp        error("input interpolations not consistent") end

	#set value
	input.values[index] = value
end#}}}
function SetTriaInput(inputs::Inputs,enum::IssmEnum,interp::IssmEnum,indices::Vector{Int32},values::Vector{Float64}) #{{{

	#Does this input exist
	if !haskey(inputs.lookup,enum)
		#it does not exist yet, we need to create it...
		@assert inputs.numberofvertices>0
		if interp==P1Enum
			inputs.lookup[enum] = ElementInput(enum,interp,zeros(inputs.numberofvertices))
		else
			error("not supported yet")
		end
	end

	#Get this input and check type
	input = inputs.lookup[enum]
	if typeof(input)!=ElementInput error("input type not consistent") end
	if interp!=input.interp        error("input interpolations not consistent") end

	#set value
	input.values[indices] = values
end#}}}

mutable struct Node #{{{
	id::Int32
	sid::Int32
	indexingupdate::Bool
	gsize::Int32
	gdoflist::Vector{Int32}
	fdoflist::Vector{Int32}
	sdoflist::Vector{Int32}
	svalues::Vector{Float64}
end# }}}
struct Constraint #{{{
	id::Int32
	nodeid::Int32
	dof::Int8
	value::Float64
end# }}}
mutable struct Vertex#{{{
	sid::Int32
	x::Float64
	y::Float64
end# }}}
mutable struct Tria #{{{
	sid::Int32
	pid::Int32
	#vertexids::Int64[3]
	#vertices::Vertex[3]
	vertexids::Vector{Int32}
	vertices::Vector{Vertex}
	nodes::Vector{Node}
	parameters::Parameters
	inputs::Inputs
end# }}}
mutable struct FemModel #{{{
	elements::Vector{Tria}
	vertices::Vector{Vertex}
	nodes::Vector{Node}
	parameters::Vector{Parameter}
	inputs::Inputs
	constraints::Vector{Constraint}
	#loads::Vector{Loads}
end#}}}
struct GaussTria #{{{
	numgauss::Int32
	weights::Vector{Float64}
	coords1::Vector{Float64}
	coords2::Vector{Float64}
	coords3::Vector{Float64}
end #}}}
function GaussTria(order::Int32) #{{{

	#=Gauss quadrature points for the triangle.
	Higher-order points from D.A. Dunavant, "High Degree Efficient
	Symmetrical Gaussian Quadrature Rules for the Triangle", IJNME,
	Vol. 21, pp. 1129-1148 (1985), as transcribed for Probe rules3.=#

	if(order==1)
		npoints = 1
		weights = [1.732050807568877]
		coords1 = [0.333333333333333]
		coords2 = [0.333333333333333]
		coords3 = [0.333333333333333]
	elseif(order==2)
		weights = [0.577350269189625, 0.577350269189625, 0.577350269189625]
		coords1 = [0.666666666666667, 0.166666666666667, 0.166666666666667]
		coords2 = [0.166666666666667, 0.666666666666667, 0.166666666666667]
		coords3 = [0.166666666666667, 0.166666666666667, 0.666666666666667]
	elseif(order==3)
		npoints=4
		weights = [-0.974278579257493, 0.902109795608790, 0.902109795608790, 0.902109795608790]
		coords1 = [0.333333333333333, 0.600000000000000, 0.200000000000000, 0.200000000000000]
		coords2 = [0.333333333333333, 0.200000000000000, 0.600000000000000, 0.200000000000000]
		coords3 = [0.333333333333333, 0.200000000000000, 0.200000000000000, 0.600000000000000]
	else
		error("order ",order," not supported yet");
	end

	return GaussTria(npoints,weights,coords1,coords2,coords3)
end# }}}

#Modules
function FetchDataToInput(md::model,inputs::Inputs,elements::Vector{Tria},data::Vector{Float64},enum::IssmEnum) #{{{
	for i in 1:length(elements)
		InputCreate(elements[i],inputs,data,enum)
	end
end#}}}
function IssmCore(md::model) #{{{
	
	#Construct FemModel
	femmodel=ModelProcessor(md)

	#Solve
	Stressbalance(femmodel)

	#then what??

end# }}}
function ModelProcessor(md::model) #{{{

	#Initialize structures
	elements   = Vector{Tria}(undef,0)
	vertices   = Vector{Vertex}(undef,0)
	parameters = Parameters(Dict{IssmEnum,Parameter}())
	inputs     = Inputs(md.mesh.numberofelements,md.mesh.numberofvertices,Dict{IssmEnum,Input}())

	#Create  elements, vertices and materials (independent of the analysis)
	CreateElements(elements,md)
	CreateVertices(vertices,md)
	CreateParameters(parameters,md)
	CreateInputs(inputs,elements,md)

	#Now create analysis specific data structure
	StressbalanceUpdateParameters(parameters,md)
end# }}}
function CreateElements(elements::Vector{Tria},md::model) #{{{

	#Make sure elements is currently empty
	@assert length(elements)==0

	tempparams   = Parameters(Dict{IssmEnum,Parameter}())
	tempinputs   = Inputs(-1,-1,Dict{IssmEnum,Input}())
	tempvertices = Vector{Vertex}(undef,0)
	tempnodes    = Vector{Node}(undef,0)

	count = 0
	for i in 1:md.mesh.numberofelements
		push!(elements,Tria(i,count,md.mesh.elements[i,:],tempvertices,tempnodes,tempparams,tempinputs))
	end

end# }}}
function CreateVertices(vertices::Vector{Vertex},md::model) #{{{

	#Make sure vertices is currently empty
	@assert length(vertices)==0

	#Get data from md
	x = md.mesh.x
	y = md.mesh.y

	count = 0
	for i in 1:md.mesh.numberofvertices
		push!(vertices,Vertex(i,x[i],y[i]))
	end

end# }}}
function CreateParameters(parameters::Parameters,md::model) #{{{

	#Get data from md
	AddParam(parameters,md.materials.rho_ice,MaterialsRhoIceEnum)
	AddParam(parameters,md.materials.rho_water,MaterialsRhoSeawaterEnum)
	AddParam(parameters,md.materials.rho_freshwater,MaterialsRhoFreshwaterEnum)
end# }}}
function CreateInputs(inputs::Inputs,elements::Vector{Tria},md::model) #{{{

	#Only assume we have Matice for now
	FetchDataToInput(md,inputs,elements,md.materials.rheology_B,MaterialsRheologyBEnum);
	FetchDataToInput(md,inputs,elements,md.materials.rheology_n,MaterialsRheologyNEnum);
end# }}}

#Element functions
function InputCreate(element::Tria,inputs::Inputs,data::Vector{Float64},enum::IssmEnum) #{{{
	if size(data,1)==inputs.numberofelements
		SetTriaInput(inputs,enum,P0Enum,element.sid,data[element.sid])
	elseif size(data,1)==inputs.numberofvertices
		SetTriaInput(inputs,enum,P1Enum,element.vertexids,data[element.vertexids])
	else
		error("size ",size(data,1)," not supported yet");
	end
end #}}}
