using Printf

#Define classes used by model struct
abstract type AbstractMesh end
mutable struct Mesh2dTriangle <: AbstractMesh
	numberofvertices::Int32
	numberofelements::Int32
	x::Vector{Float64}
	y::Vector{Float64}
	elements::Matrix{Int32}
end
function Mesh2dTriangle() #{{{
	return Mesh2dTriangle( 0, 0, Vector{Float64}(undef,0), Vector{Float64}(undef, 0), Matrix{Int32}(undef, 0, 0))
end# }}}
function Base.show(io::IO, this::Mesh2dTriangle)# {{{

	println(io,typeof(this),":")
	for name in fieldnames(typeof(this))
		a=getfield(this,name)
		print(io,"   $(name) = ")
		if !isempty(a)
			println(io, typeof(a), " of size ", size(a))
		else
			println(io,"empty")
		end
	end
end# }}}
mutable struct Mesh3dPrism{T} <: AbstractMesh
	numberofvertices::Int32
	numberofelements::Int32
	numberoflayers::Int32
	x::Vector{Float64}
	y::Vector{Float64}
	z::Vector{Float64}
	elements::Matrix{Int32}
end
function Mesh3dPrism() #{{{
	return Mesh3dPrism( 0, 0, 0, Vector{Float64}(undef,0), Vector{Float64}(undef,0), Vector{Float64}(undef,0), Matrix{Int32}(undef, 0, 0))
end# }}}
mutable struct Geometry
	surface::Vector{Float64}
	base::Vector{Float64}
	thickness::Vector{Float64}
	bed::Vector{Float64}
end
function Geometry() #{{{
	return Geometry( Vector{Float64}(undef,0), Vector{Float64}(undef,0), Vector{Float64}(undef,0), Vector{Float64}(undef,0))
end# }}}
mutable struct Mask
	ocean_levelset::Vector{Float64}
	ice_levelset::Vector{Float64}
end
function Mask() #{{{
	return Mask( Vector{Float64}(undef,0), Vector{Float64}(undef,0))
end# }}}
mutable struct Initialization
	vx::Vector{Float64}
	vy::Vector{Float64}
end
function Initialization() #{{{
	return Initialization( Vector{Float64}(undef,0), Vector{Float64}(undef,0))
end# }}}
mutable struct Stressbalance
	spcvx::Vector{Float64}
	spcvy::Vector{Float64}
	restol::Float64
	reltol::Float64
	abstol::Float64
	maxiter::Int32
end
function Stressbalance() #{{{
	return Stressbalance( Vector{Float64}(undef,0), Vector{Float64}(undef,0), 0., 0., 0., 0)
end# }}}
mutable struct Materials
	rho_ice::Float64
	rho_water::Float64
	rho_freshwater::Float64
	mu_water::Float64
	heatcapacity::Float64
	latentheat::Float64
	thermalconductivity::Float64
	temperateiceconductivity::Float64
	effectiveconductivity_averaging::Int32
	meltingpoint::Float64
	beta::Float64
	mixed_layer_capacity::Float64
	thermal_exchange_velocity::Float64
	rheology_B::Vector{Float64}
	rheology_n::Vector{Float64}
	rheology_law::String
end
function Materials() #{{{
	return Materials(917., 1023., 1000., 0.001787, 2093., 3.34*10^5, 2.4, .24, 1, 273.15, 9.8*10^-8, 3974., 1.00*10^-4, Vector{Float64}(undef,0), Vector{Float64}(undef,0), "Cuffey")
end# }}}

mutable struct model
	mesh::AbstractMesh
	geometry::Geometry
	mask::Mask
	materials::Materials
	initialization::Initialization
	stressbalance::Stressbalance
end
function model() #{{{
	return model( Mesh2dTriangle(), Geometry(), Mask(), Materials(), Initialization(),Stressbalance())
end#}}}
function Base.show(io::IO, md::model)# {{{

	compact = get(io, :compact, false)

	println(io,"Model:")
	@printf "%19s: %-22s -- %s\n" "mesh" typeof(md.mesh) "mesh properties"
	@printf "%19s: %-22s -- %s\n" "geometry" typeof(md.geometry) "surface elevation, bedrock topography, ice thickness,..."
	@printf "%19s: %-22s -- %s\n" "mask" typeof(md.mask) "defines grounded and floating regions"
	@printf "%19s: %-22s -- %s\n" "materials" typeof(md.materials) "material properties"
	@printf "%19s: %-22s -- %s\n" "initialization" typeof(md.initialization) "initial state"
	@printf "%19s: %-22s -- %s\n" "stressbalance" typeof(md.stressbalance) "stress balance parameters"

end# }}}

#utils
function issmdir() #{{{
	issmdir = ENV["ISSM_DIR"]

	if isempty(issmdir)
		error("Could not determine the location of ISSM")
	else
		return issmdir
	end
end#}}}
function archread(filename::String,variablename::String) #{{{

	#initialize variables
	found = false

	#open file
	output = open(filename, "r") do f

		while !eof(f)
			reclen  = bswap(read(f, Int32))
			rectype = bswap(read(f, Int32))
			if rectype!=1
				error("Expected variable of type string")
			else
				fieldname_length = bswap(read(f, Int32))
				field_name = String(read(f, fieldname_length))
			end
			rec_length = bswap(read(f, Int32))
			field_type = bswap(read(f, Int32))
			if field_type==2
				data = bswap(read(f, Float64))
			elseif field_type==3
				rows = bswap(read(f, Int32))
				cols = bswap(read(f, Int32))
				data = reinterpret(Float64, read(f, sizeof(Float64)*rows*cols))
				data .= ntoh.(data)
				data = reshape(data, (rows,cols))
				data = collect(data)
				if cols == 1
					data = vec(data)
				end
			else
				error("Error: Encountered invalid field type when reading data.")
			end

			if field_name == variablename
				found = true
				return data
			end
		end
	end

	return output
end# }}}
function InterpFromMeshToMesh2d(index::Array,x::Vector,y::Vector,data::Vector,xout::Vector,yout::Vector) #{{{
	#prepare input arrays
	nods = Cint(length(x))
	nels = Cint(size(index,1))
	nods_interp = Cint(length(xout))
	Cindex=Array{Cint,1}(undef,length(index))
	for i in 1:size(index,1)
		for j in 1:3
			Cindex[(i-1)*3+j] = Int32(index[i,j])
		end
	end
	Cx    = Array{Cdouble,1}(undef,nods)
	Cy    = Array{Cdouble,1}(undef,nods)
	Cdata = Array{Cdouble,1}(undef,nods)
	for i in 1:nods
		Cx[i]    = x[i]
		Cy[i]    = y[i]
		Cdata[i] = data[i]
	end
	Cxout = Array{Cdouble,1}(undef,nods_interp)
	Cyout = Array{Cdouble,1}(undef,nods_interp)
	for i in 1:nods_interp
		Cxout[i] = xout[i]
		Cyout[i] = yout[i]
	end

	Cdataout = Vector{Float64}(undef,nods_interp)

	#This is not working....
	rc=ccall( (:InterpFromMeshToMesh2dx,"libISSMCore"),
				Cint, (Ptr{Ptr{Cdouble}},Ptr{Cint}, Ptr{Cdouble}, Ptr{Cdouble}, Cint, Cint, Ptr{Cdouble}, Cint, Cint, Ptr{Cdouble}, Ptr{Cdouble}, Cint),
				Ref(Ref(Cdataout)), Ref(Cindex), Ref(Cx), Ref(Cy), nods, nels,
				Ref(Cdata), nods, 1, Ref(Cxout), Ref(Cyout), nods_interp)

	#Process output
	dataout = Vector{Float64}(undef,nods_interp)
	for i in 1:nods_interp
		dataout[i] = Cdataout[i]
	end

	return dataout
end #}}}
function solve(md::model,solution::String) #{{{

	if solution=="sb" || solution=="Stressbalance"
		solutionstring = "StressbalanceSolution"
	else
		error("solutionstring "*solution*" not supported!");
	end

	IssmCore(md)

	return md
end #}}}
