/*!\file FrontalForcingsx
 * \brief: compute ice frontal melting rate
 */

#include "./FrontalForcingsx.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"
#include "../../shared/Random/random.h"

void FrontalForcingsx(FemModel* femmodel){/*{{{*/

	/*Recover melt_parameterization*/
	int melt_parameterization;
	femmodel->parameters->FindParam(&melt_parameterization,FrontalForcingsParamEnum);

	/*Calculate melting rate*/
	switch(melt_parameterization){
		case FrontalForcingsDefaultEnum:
			break;
		case FrontalForcingsRignotAutoregressionEnum:
			Thermalforcingautoregressionx(femmodel);
			/*Do not break here, call IcefrontAreax(),RignotMeltParameterizationx()*/
		case FrontalForcingsRignotEnum:
			femmodel->IcefrontAreax();
			femmodel->RignotMeltParameterizationx();
			break;
		default:
			_error_("Frontal forcings "<<EnumToStringx(melt_parameterization)<<" not supported yet");
	}
}/*}}}*/
void ThermalforcingautoregressionInitx(FemModel* femmodel){/*{{{*/

   /*Initialization step of Thermalforcingautoregressionx*/
   int M,N,Nphi,arorder,numbasins,my_rank;
   IssmDouble starttime,tstep_ar,tinit_ar;
   femmodel->parameters->FindParam(&numbasins,FrontalForcingsNumberofBasinsEnum);
   femmodel->parameters->FindParam(&arorder,FrontalForcingsAutoregressiveOrderEnum);
   IssmDouble* beta0    = NULL;
   IssmDouble* beta1    = NULL;
   IssmDouble* phi      = NULL;
   IssmDouble* covmat   = NULL;
   femmodel->parameters->FindParam(&starttime,TimesteppingStartTimeEnum);
   femmodel->parameters->FindParam(&tstep_ar,FrontalForcingsAutoregressionTimestepEnum);
   femmodel->parameters->FindParam(&tinit_ar,FrontalForcingsAutoregressionInitialTimeEnum);
   femmodel->parameters->FindParam(&beta0,&M,FrontalForcingsBeta0Enum);    _assert_(M==numbasins);
   femmodel->parameters->FindParam(&beta1,&M,FrontalForcingsBeta1Enum);    _assert_(M==numbasins);
   femmodel->parameters->FindParam(&phi,&M,&Nphi,FrontalForcingsPhiEnum);  _assert_(M==numbasins); _assert_(Nphi==arorder);
   femmodel->parameters->FindParam(&covmat,&M,&N,FrontalForcingsCovmatEnum); _assert_(M==numbasins); _assert_(N==numbasins);

   /*AR model spin-up*/
   int nspin{2*arorder+5};
   IssmDouble* noisespin = xNewZeroInit<IssmDouble>(numbasins*nspin);
   my_rank=IssmComm::GetRank();
   if(my_rank==0){
      bool randomflag{};
      femmodel->parameters->FindParam(&randomflag,FrontalForcingsRandomflagEnum);
      int fixedseed;
      for(int i=0;i<nspin;i++){
         IssmDouble* temparray = NULL;
         /*Determine whether random seed is fixed to for loop step (randomflag==false) or random seed truly random (randomflag==true)*/
         if(randomflag) fixedseed=-1;
         else fixedseed = i;
         multivariateNormal(&temparray,numbasins,0.0,covmat,fixedseed);
         for(int j=0;j<numbasins;j++){noisespin[i*numbasins+j]=temparray[j];}
         xDelete<IssmDouble>(temparray);
      }
   }
   ISSM_MPI_Bcast(noisespin,numbasins*nspin,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());

   /*Initialize ThermalforcingValuesAutoregressionEnum*/
   for(Object* &object:femmodel->elements->objects){
      Element* element      = xDynamicCast<Element*>(object); //generate element object
      element->AutoregressionInit(numbasins,arorder,nspin,starttime,tstep_ar,tinit_ar,beta0,beta1,phi,noisespin,FrontalForcingsRignotAutoregressionEnum);
   }

   /*Cleanup*/
   xDelete<IssmDouble>(beta0);
   xDelete<IssmDouble>(beta1);
   xDelete<IssmDouble>(phi);
   xDelete<IssmDouble>(noisespin);
   xDelete<IssmDouble>(covmat);
}/*}}}*/
void Thermalforcingautoregressionx(FemModel* femmodel){/*{{{*/

   /*Get time parameters*/
   IssmDouble time,dt,starttime,tstep_ar;
   femmodel->parameters->FindParam(&time,TimeEnum);
   femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
   femmodel->parameters->FindParam(&starttime,TimesteppingStartTimeEnum);
   femmodel->parameters->FindParam(&tstep_ar,FrontalForcingsAutoregressionTimestepEnum);

   /*Initialize module at first time step*/
   if(time<=starttime+dt){ThermalforcingautoregressionInitx(femmodel);}
   /*Determine if this is a time step for the AR model*/
   bool isstepforar = false;

   #ifndef _HAVE_AD_
   if((fmod(time,tstep_ar)<fmod((time-dt),tstep_ar)) || (time<=starttime+dt) || tstep_ar==dt) isstepforar = true;
   #else
   _error_("not implemented yet");
   #endif

   /*Load parameters*/
   int M,N,Nphi,arorder,numbasins,my_rank;
   femmodel->parameters->FindParam(&numbasins,FrontalForcingsNumberofBasinsEnum);
   femmodel->parameters->FindParam(&arorder,FrontalForcingsAutoregressiveOrderEnum);
   IssmDouble tinit_ar;
   IssmDouble* beta0      = NULL;
   IssmDouble* beta1      = NULL;
   IssmDouble* phi        = NULL;
   IssmDouble* covmat     = NULL;
   IssmDouble* noiseterms = xNew<IssmDouble>(numbasins);
   femmodel->parameters->FindParam(&tinit_ar,FrontalForcingsAutoregressionInitialTimeEnum);
   femmodel->parameters->FindParam(&beta0,&M,FrontalForcingsBeta0Enum);    _assert_(M==numbasins);
   femmodel->parameters->FindParam(&beta1,&M,FrontalForcingsBeta1Enum);    _assert_(M==numbasins);
   femmodel->parameters->FindParam(&phi,&M,&Nphi,FrontalForcingsPhiEnum);  _assert_(M==numbasins); _assert_(Nphi==arorder);
   femmodel->parameters->FindParam(&covmat,&M,&N,FrontalForcingsCovmatEnum); _assert_(M==numbasins); _assert_(N==numbasins);

   /*Time elapsed with respect to AR model initial time*/
   IssmDouble telapsed_ar = time-tinit_ar;

   /*Before looping through elements: compute noise term specific to each basin from covmat*/
   my_rank=IssmComm::GetRank();
   if(my_rank==0){
      bool randomflag{};
      femmodel->parameters->FindParam(&randomflag,FrontalForcingsRandomflagEnum);
      /*Determine whether random seed is fixed to time step (randomflag==false) or random seed truly random (randomflag==true)*/
      int fixedseed;
      if(randomflag) fixedseed=-1;
      else fixedseed = reCast<int,IssmDouble>((time-starttime)/dt);
		/*multivariateNormal needs to be passed a NULL pointer to avoid memory leak issues*/
      IssmDouble* temparray = NULL;
      multivariateNormal(&temparray,numbasins,0.0,covmat,fixedseed);
      for(int i=0;i<numbasins;i++) noiseterms[i]=temparray[i];
      xDelete<IssmDouble>(temparray);
   }
   ISSM_MPI_Bcast(noiseterms,numbasins,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());

   /*Loop over each element to compute SMB at vertices*/
   for(Object* &object:femmodel->elements->objects){
      Element* element = xDynamicCast<Element*>(object);
      element->Autoregression(isstepforar,arorder,telapsed_ar,beta0,beta1,phi,noiseterms,FrontalForcingsRignotAutoregressionEnum);
   }

   /*Cleanup*/
   xDelete<IssmDouble>(beta0);
   xDelete<IssmDouble>(beta1);
   xDelete<IssmDouble>(phi);
   xDelete<IssmDouble>(noiseterms);
   xDelete<IssmDouble>(covmat);
}/*}}}*/
