/*!\file: random
 * \brief random number generating functions
 */ 

/*Headers*/
/*{{{*/
#include <stdio.h>
#include <sys/types.h>
#include <math.h>
#include <float.h>    /*  DBL_EPSILON  */
#include <chrono>
#include <cstdarg>
#include <iostream>
#include <random>

#include "../Matrix/matrix.h"
#include "../Exceptions/exceptions.h"
#include "../MemOps/MemOps.h"
#include "../io/io.h"
/*}}}*/

void univariateNormal(IssmPDouble* prand, IssmPDouble mean, IssmPDouble sdev, int seedfixed=-1) { /*{{{*/

	unsigned seed;
	/*Random seed using time_since_epoch*/
	if(seedfixed==-1) seed = std::chrono::steady_clock::now().time_since_epoch().count(); 
	/*Seed fixed by input argument*/
	else seed = seedfixed;
	std::default_random_engine generator(seed);
	/*Normal Probability Distribution*/
   std::normal_distribution<IssmPDouble> normdistri(mean,sdev); 
	*prand = normdistri(generator);
} /*}}}*/
void multivariateNormal(IssmDouble** prand, int dim, IssmDouble mean, IssmDouble* covariancematrix, int seedfixed=-1) { /*{{{*/
   
	IssmPDouble* sampleStandardNormal    = xNew<IssmPDouble>(dim);
   IssmDouble* sampleMultivariateNormal = xNew<IssmDouble>(dim);
   IssmDouble* Lchol                    = xNewZeroInit<IssmDouble>(dim*dim);

	/*True randomness if seedfixed==-1, otherwise random seed is fixed at seedfixed*/
	for(int i=0;i<dim;i++) univariateNormal(&(sampleStandardNormal[i]),0.0,1.0,seedfixed); 
	CholeskyRealPositiveDefinite(Lchol,covariancematrix,dim);
   
	/*Matrix by vector multiplication*/
	for(int i=0;i<dim;i++){ 
      /*Entry-by-entry multiplication along matrix row*/
      IssmDouble sum=0.;
      for(int j=0;j<dim;j++) sum += sampleStandardNormal[j]*Lchol[i*dim+j]; 
      sampleMultivariateNormal[i] = mean+sum;
   }

   /*Assign output pointer and cleanup*/
   *prand = sampleMultivariateNormal;
   xDelete<IssmPDouble>(sampleStandardNormal);
   xDelete<IssmDouble>(Lchol);
} /*}}}*/
void multivariateNormal(IssmDouble** prand, int dim, IssmDouble* mean, IssmDouble* covariancematrix, int seedfixed=-1) { /*{{{*/
	
	IssmPDouble* sampleStandardNormal    = xNew<IssmPDouble>(dim);
	IssmDouble* sampleMultivariateNormal = xNew<IssmDouble>(dim);
	IssmDouble* Lchol                    = xNewZeroInit<IssmDouble>(dim*dim);

	/*True randomness if seedfixed==-1, otherwise random seed is fixed at seedfixed*/
	for(int i=0;i<dim;i++) univariateNormal(&(sampleStandardNormal[i]),0.0,1.0,seedfixed); 
	CholeskyRealPositiveDefinite(Lchol,covariancematrix,dim);

	/*Matrix by vector multiplication*/
	for(int i=0;i<dim;i++){
		IssmDouble sum = 0.;
      for(int j=0;j<dim;j++) sum += sampleStandardNormal[j]*Lchol[i*dim+j]; 
      sampleMultivariateNormal[i] = mean[i]+sum;
	}
   
	/*Assign output pointer and cleanup*/
	*prand = sampleMultivariateNormal;
	xDelete<IssmPDouble>(sampleStandardNormal);
	xDelete<IssmDouble>(Lchol);
} /*}}}*/



