
#exp object definition, constructor, and disp
mutable struct ExpStruct #{{{
	name::String
	nods::Int32
	density::Float64
	x::Vector{Float64}
   y::Vector{Float64}
	closed::Bool
end  #}}}
function ExpStruct() #{{{
	return ExpStruct("",0, 0., Vector{Float64}(undef,0), Vector{Float64}(undef,0), false)
end# }}}
function Base.show(io::IO, exp::ExpStruct)# {{{

	compact = get(io, :compact, false)

	println(io,"ExpStruct:")
	for name in fieldnames(typeof(exp))
		a=getfield(exp,name)
		print(io,"   $(name) = ")
		if !isempty(a)
			if compact && eltype(a)<:Number && length(a)>3
				println(io, typeof(a), " of size ", size(a))
			else
				println(io,a)
			end
		else
			println(io,"empty")
		end
	end
end# }}}

#methods
#expread {{{
"""
	EXPREAD - read a file exp and build a Structure

	This function reads an *.exp* and builds a structure containing the fields x
	and y corresponding to the coordinates, one for the filename of the exp
	file, for the density, for the nodes, and a field 'closed' to indicate if the
	domain is closed.

	Usage:
		exp=expread(filename)

# Examples:
```julia-repl
julia> exp=expread('domainoutline.exp')
```

# Arguments:
- filename: the ARGUS file to read
"""
function expread(filename::String)

	#initialize output
	contours = Vector{ExpStruct}(undef, 0)

	#open file
	f = open(filename, "r") do f

		#initialize some variables
		nprof = 0
		line = 1

		while !eof(f)

			#read first line
			A = readline(f); line += 1

			#if isempty, go to the next line and try again
			if isempty(A)
				continue
			else
				#initialize new profile
				nprof += 1 
				exp = ExpStruct();
			end

			#extract profile name
			if A[1:8]!="## Name:"
				println("line $(line): $(A)")
				error("Unexpected exp file formatting") 
			end
			if length(A)>8
				exp.name = A[9:end]
			end

			#read Icon
			A = readline(f); line += 1
			if A[1:8]!="## Icon:" error("Unexpected exp file formatting") end

			#read Info
			A = readline(f); line += 1
			if A[1:14]!="# Points Count"
				println("line $(line): $(A)")
				error("Unexpected exp file formatting") 
			end

			#Reads number of nods and density
			A = readline(f); line += 1
			A = parse.(Float64, split(A))
			if length(A) != 2 error("Unexpected exp file formatting") end
			exp.nods = A[1]; exp.density = A[2]

			#Allocate arrays
			if exp.nods<=0 error("Unexpected exp file formatting") end
			exp.x = Vector{Float64}(undef,exp.nods)
			exp.y = Vector{Float64}(undef,exp.nods)

			#Read coordinates
			A = readline(f); line += 1
			if A[1:13]!="# X pos Y pos" error("Unexpected exp file formatting") end
			for i in 1:exp.nods
				A = readline(f); line += 1
				A = parse.(Float64, split(A))
				if length(A) != 2 error("Unexpected exp file formatting") end
				if any(isnan.(A)) error("NaNs found in coordinate") end
				exp.x[i] = A[1]; exp.y[i] = A[2]
			end

			#check if closed
			if exp.nods>1 && exp.x[1]==exp.x[end] && exp.y[1]==exp.y[end]
				exp.closed = true
			else
				exp.closed = false
			end

			#add profile to list
			push!(contours, exp)
		end
	end

	return contours
end# }}}
