%SOLIDEARTHSETTINGS class definition
%
%   Usage:
%      solidearthsettings=solidearthsettings();

classdef solidearthsettings
	properties (SetAccess=public) 
		reltol                 = 0;
		abstol                 = 0;
		maxiter                = 0;
		rigid                  = 0;
		elastic                = 0;
		viscous                = 0;
		rotation               = 0;
		ocean_area_scaling     = 0;
		runfrequency           = 1; %how many time steps we skip before we run grd_core
		computesealevelchange  = 1; %will sea-level be coputed? 
		isgrd                  = 0; %will GRD patterns be computed? 
		compute_bp_grd         = 0; %will GRD patterns for bottom pressures be computed? 
		degacc                 = 0; %degree increment for resolution of Green tables.
		timeacc                = 0; %time step accurary required to compute Green tables
		horiz                  = 0; %compute horizontal deformation
		grdmodel               = 0; %grd model (0 by default, 1 for elastic, 2 for Ivins)
		cross_section_shape    = 0; %cross section only used when grd model is Ivins
	end
	methods
		function self = solidearthsettings(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{
		
		%Convergence criterion: absolute, relative and residual
		self.reltol=0.01; % 1 percent
		self.abstol=NaN;  % default

		%maximum of non-linear iterations.
		self.maxiter=5;

		%computational flags: 
		self.rigid=1;
		self.elastic=1;
		self.viscous=1;
		self.rotation=1;
		self.ocean_area_scaling=0;
		self.compute_bp_grd=0;
		self.isgrd=0;
		self.computesealevelchange=1;

		%numerical discretization accuracy
		self.degacc=.01;
		self.timeacc=1; 

		%how many time steps we skip before we run solidearthsettings solver during transient
		self.runfrequency=1;
		
		%horizontal displacement?  (not by default)
		self.horiz=0;
		
		%cross section for Ivins model 
		self.cross_section_shape=1; %square as default (see iedge in GiaDeflectionCorex)

		%no grd model by default: 
		self.grdmodel=0; 

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			if ~ismember('SealevelchangeAnalysis',analyses) | (strcmp(solution,'TransientSolution') & md.transient.isslc==0), 
				return; 
			end
			md = checkfield(md,'fieldname','solidearth.settings.reltol','size',[1 1]);
			md = checkfield(md,'fieldname','solidearth.settings.abstol','size',[1 1]);
			md = checkfield(md,'fieldname','solidearth.settings.maxiter','size',[1 1],'>=',1);
			md = checkfield(md,'fieldname','solidearth.settings.runfrequency','size',[1 1],'>=',1);
			md = checkfield(md,'fieldname','solidearth.settings.degacc','size',[1 1],'>=',1e-10);
			md = checkfield(md,'fieldname','solidearth.settings.timeacc','size',[1 1],'>',0);
			md = checkfield(md,'fieldname','solidearth.settings.horiz','NaN',1,'Inf',1,'values',[0 1]);
			md = checkfield(md,'fieldname','solidearth.settings.grdmodel','>=',0,'<=',2);
			md = checkfield(md,'fieldname','solidearth.settings.cross_section_shape','numel',[1],'values',[1,2]);

			%checks on computational flags
			if self.elastic==1 & self.rigid==0,
				error('solidearthsettings checkconsistency error message: need rigid on if elastic flag is set');
			end
			if self.viscous==1 & self.elastic==0,
				error('solidearthsettings checkconsistency error message: need elastic on if viscous flag is set');
			end

			%a GRD computation has been requested, make some checks on the nature of the meshes provided. 
			if self.isgrd,
				if strcmpi(class(md.mesh),'mesh3dsurface'),
					if self.grdmodel==2,
						error('model requires a 2D mesh to run gia Ivins computations (change mesh from mesh3dsurface to mesh2d)');
					end
				else
					if self.grdmodel==1,
						error('model requires a 3D surface mesh to run GRD computations (change mesh from mesh2d to mesh3dsurface)');
					end
				end
			end

			if self.compute_bp_grd==1 & md.solidearth.settings.isgrd==0,
					error('solidearthsettings checkconsistency error message; if bottom pressure grd patterns are requested, solidearth settings class should have isgrd flag on');
			end

		end % }}}
		function disp(self) % {{{
			disp(sprintf('   solidearth settings:'));

			fielddisplay(self,'reltol','sea level change relative convergence criterion, (default, NaN: not applied)');
			fielddisplay(self,'abstol','sea level change absolute convergence criterion, NaN: not applied');
			fielddisplay(self,'maxiter','maximum number of nonlinear iterations');
			fielddisplay(self,'ocean_area_scaling','correction for model representation of ocean area [default: No correction]'); 
			fielddisplay(self,'computesealevelchange','compute sealevel change (default 1)');
			fielddisplay(self,'isgrd','compute GRD patterns (default 1)');
			fielddisplay(self,'compute_bp_grd','compute GRD patterns for bottom pressure loads (default 1)');
			fielddisplay(self,'runfrequency','how many time steps we skip before we run solidearthsettings solver during transient (default: 1)');
			fielddisplay(self,'rigid','rigid earth graviational potential perturbation');
			fielddisplay(self,'elastic','elastic earth graviational potential perturbation');
			fielddisplay(self,'viscous','viscous earth graviational potential perturbation');
			fielddisplay(self,'rotation','earth rotational potential perturbation');
			fielddisplay(self,'degacc','accuracy (default .01 deg) for numerical discretization of the Green''s functions');
			fielddisplay(self,'timeacc','time accuracy (default 1 yr) for numerical discretization of the Green''s functions');
			fielddisplay(self,'grdmodel','type of deformation model, 1 for elastic, 2 for visco-elastic from Ivins');
			fielddisplay(self,'cross_section_shape','1: square-edged (default). 2: elliptical. See iedge in GiaDeflectionCore');
		end % }}}
		function marshall(self,prefix,md,fid) % {{{
			WriteData(fid,prefix,'object',self,'fieldname','reltol','name','md.solidearth.settings.reltol','format','Double');
			WriteData(fid,prefix,'object',self,'fieldname','abstol','name','md.solidearth.settings.abstol','format','Double');
			WriteData(fid,prefix,'object',self,'fieldname','maxiter','name','md.solidearth.settings.maxiter','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','rigid','name','md.solidearth.settings.rigid','format','Boolean');
			WriteData(fid,prefix,'object',self,'fieldname','elastic','name','md.solidearth.settings.elastic','format','Boolean');
			WriteData(fid,prefix,'object',self,'fieldname','viscous','name','md.solidearth.settings.viscous','format','Boolean');
			WriteData(fid,prefix,'object',self,'fieldname','rotation','name','md.solidearth.settings.rotation','format','Boolean');
			WriteData(fid,prefix,'object',self,'fieldname','ocean_area_scaling','name','md.solidearth.settings.ocean_area_scaling','format','Boolean');
			WriteData(fid,prefix,'object',self,'fieldname','runfrequency','name','md.solidearth.settings.runfrequency','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','degacc','name','md.solidearth.settings.degacc','format','Double');
			WriteData(fid,prefix,'object',self,'fieldname','timeacc','name','md.solidearth.settings.timeacc','format','Double','scale',md.constants.yts);
			WriteData(fid,prefix,'object',self,'fieldname','horiz','name','md.solidearth.settings.horiz','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','computesealevelchange','name','md.solidearth.settings.computesealevelchange','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','isgrd','name','md.solidearth.settings.isgrd','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','compute_bp_grd','name','md.solidearth.settings.compute_bp_grd','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','grdmodel','name','md.solidearth.settings.grdmodel','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','cross_section_shape','name','md.solidearth.settings.cross_section_shape','format','Integer');
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejsdouble(fid,[modelname '.solidearth.settings.maxiter'],self.maxiter);
			writejsdouble(fid,[modelname '.solidearth.settings.reltol'],self.reltol);
			writejsdouble(fid,[modelname '.solidearth.settings.abstol'],self.abstol);
			writejsdouble(fid,[modelname '.solidearth.settings.rigid'],self.rigid);
			writejsdouble(fid,[modelname '.solidearth.settings.elastic'],self.elastic);
			writejsdouble(fid,[modelname '.solidearth.settings.viscous'],self.viscous);
			writejsdouble(fid,[modelname '.solidearth.settings.rotation'],self.rotation);
			writejsdouble(fid,[modelname '.solidearth.settings.ocean_area_scaling'],self.ocean_area_scaling);
			writejsdouble(fid,[modelname '.solidearth.settings.run_frequency'],self.run_frequency);
			writejsdouble(fid,[modelname '.solidearth.settings.degacc'],self.degacc);
			writejsdouble(fid,[modelname '.solidearth.settings.timeacc'],self.timeacc);
			writejsdouble(fid,[modelname '.solidearth.settings.cross_section_shape'],self.cross_section_shape);
		end % }}}
		function self = extrude(self,md) % {{{
		end % }}}
	end
end
