/*!\file: sealevelchange_core.cpp
 * \brief: core of the sea-level change solution 
 */ 

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../classes/Inputs/TriaInput.h"
#include "../classes/Inputs/TransientInput.h"
#include "../classes/Inputs/DatasetInput.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

/*support routines local definitions:{{{*/
void TransferForcing(FemModel* femmodel,int forcingenum);
void TransferSealevel(FemModel* femmodel,int forcingenum);
void slcconvergence(bool* pconverged, Vector<IssmDouble>* RSLg,Vector<IssmDouble>* RSLg_old,IssmDouble eps_rel,IssmDouble eps_abs);
IssmDouble SealevelloadsOceanAverage(Vector<IssmDouble>* sealevelloads,Vector<IssmDouble>* oceanareas, IssmDouble oceanarea);
SealevelMasks* sealevel_masks(FemModel* femmodel);
void RotationAxisMotion(IssmDouble* m, FemModel* femmodel,IssmDouble* loads, IssmDouble* sealeveloads);
void ConserveOceanMass(FemModel* femmodel,Vector<IssmDouble>* sealevelloads, IssmDouble offset, SealevelMasks* masks);
/*}}}*/

/*main cores:*/
void sealevelchange_core(FemModel* femmodel){ /*{{{*/

	/*Start profiler*/
	femmodel->profiler->Start(SLRCORE);

	/*Parameters, variables:*/
	bool save_results;
	int  optim=0; 

	/*Retrieve parameters:*/
	femmodel->parameters->FindParam(&save_results,SaveResultsEnum);
	femmodel->parameters->FindParam(&optim,SolidearthSettingsOptimEnum);
	
	/*Verbose: */
	if(VerboseSolution()) _printf0_("   computing sea level change\n");

	/*set SLR configuration: */
	femmodel->SetCurrentConfiguration(SealevelchangeAnalysisEnum);

	/*run geometry core: */
	sealevelchange_geometry(femmodel);
	
	/*any external forcings?:*/
	solidearthexternal_core(femmodel);

	/*Run coupler input transfer:*/
	couplerinput_core(femmodel);

	/*Run geodetic:*/
	if(!optim) grd_core(femmodel);
	else grd_core_optim(femmodel);

	/*Run steric core for sure:*/
	dynstr_core(femmodel);

	/*Run coupler output transfer: */
	coupleroutput_core(femmodel);

	/*Save results: */
	if(save_results){
		int     numoutputs;
		char **requested_outputs = NULL;
		if(VerboseSolution()) _printf0_("   saving results\n");
		femmodel->parameters->FindParam(&requested_outputs,&numoutputs,SealevelchangeRequestedOutputsEnum);
		femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs);
		if(numoutputs){for(int i=0;i<numoutputs;i++){xDelete<char>(requested_outputs[i]);} xDelete<char*>(requested_outputs);}
	}

	/*End profiler*/
	femmodel->profiler->Stop(SLRCORE);
}
/*}}}*/
void solidearthexternal_core(FemModel* femmodel){ /*{{{*/

	/*variables:*/
	Vector<IssmDouble> *bedrock  = NULL; 
	Vector<IssmDouble> *bedrock_rate = NULL;
	Vector<IssmDouble> *bedrockeast  = NULL; 
	Vector<IssmDouble> *bedrockeast_rate = NULL;
	Vector<IssmDouble> *bedrocknorth  = NULL; 
	Vector<IssmDouble> *bedrocknorth_rate = NULL;
	Vector<IssmDouble> *geoid= NULL; 
	Vector<IssmDouble> *geoid_rate= NULL; 
	int horiz=0;
	int modelid=-1;
	int  isexternal=0;
 
	/*parameters: */
	IssmDouble          dt;

	/*Retrieve parameters:*/
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	femmodel->parameters->FindParam(&isexternal,SolidearthIsExternalEnum); 
	
	/*Early return:*/
	if (!isexternal)return;

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing external solid earth contributions\n");

	/*Retrieve geoid viscous and elastic rates, bedrock uplift viscous and elastic rates + steric rate, as vectors:*/
	GetVectorFromInputsx(&bedrock,femmodel,BedEnum,VertexSIdEnum);
	GetVectorFromInputsx(&geoid,femmodel,SealevelEnum,VertexSIdEnum); //In ISSM, Sealevel is absolute.
	if(horiz){
		GetVectorFromInputsx(&bedrockeast,femmodel,BedEastEnum,VertexSIdEnum);
		GetVectorFromInputsx(&bedrocknorth,femmodel,BedNorthEnum,VertexSIdEnum);
	}
	
	GetVectorFromInputsx(&geoid_rate,femmodel,SolidearthExternalGeoidRateEnum,VertexSIdEnum);
	GetVectorFromInputsx(&bedrock_rate,femmodel,SolidearthExternalDisplacementUpRateEnum,VertexSIdEnum);
	if(horiz){
		GetVectorFromInputsx(&bedrockeast_rate,femmodel,SolidearthExternalDisplacementEastRateEnum,VertexSIdEnum);
		GetVectorFromInputsx(&bedrocknorth_rate,femmodel,SolidearthExternalDisplacementNorthRateEnum,VertexSIdEnum);
	}

	/*compute: sea level change = initial sea level + (N_gia_rate+N_esa_rate)  * dt + steric_rate + dynamic_rate dt*/
	geoid->AXPY(geoid_rate,dt);
	bedrock->AXPY(bedrock_rate,dt);
	if(horiz){
		bedrockeast->AXPY(bedrockeast_rate,dt);
		bedrocknorth->AXPY(bedrocknorth_rate,dt);
	}

	/*update element inputs:*/
	InputUpdateFromVectorx(femmodel,bedrock,BedEnum,VertexSIdEnum);	
	InputUpdateFromVectorx(femmodel,geoid,SealevelEnum,VertexSIdEnum);	
	if(horiz){
		InputUpdateFromVectorx(femmodel,bedrockeast,BedEastEnum,VertexSIdEnum);	
		InputUpdateFromVectorx(femmodel,bedrocknorth,BedNorthEnum,VertexSIdEnum);	
	}

	/*Free ressources:*/	
	delete bedrock; delete bedrock_rate;
	delete geoid; delete geoid_rate;
	if(horiz){
		delete bedrockeast; delete bedrockeast_rate;
		delete bedrocknorth; delete bedrocknorth_rate;
	}
}
/*}}}*/
void couplerinput_core(FemModel* femmodel){  /*{{{*/

	/*Be very careful here, everything is well thought through, do not remove 
	 * without taking big risks:*/
	
	/*parameters:*/
	int  iscoupling;
	int  modelid,earthid;
	int  count,frequency;
	int  horiz;

	/*retrieve more parameters:*/
	femmodel->parameters->FindParam(&iscoupling,IsSlcCouplingEnum);
	femmodel->parameters->FindParam(&frequency,SolidearthSettingsRunFrequencyEnum);
	femmodel->parameters->FindParam(&count,SealevelchangeRunCountEnum);

	if(iscoupling){
		femmodel->parameters->FindParam(&modelid,ModelIdEnum);
		femmodel->parameters->FindParam(&earthid,EarthIdEnum);
	}
	else{
		/* we are here, we are not running in a coupler, so we will indeed compute SLR,
		 * so make sure we are identified as being the Earth.:*/
		modelid=1; earthid=1; 
	}

	/*if we are carrying loads but are not yet computing grd core, accumulate them and skip 
	 * the rest: */
	if (count<frequency){
		count++; 
		femmodel->parameters->SetParam(count,SealevelchangeRunCountEnum); 
		return;
	}

	/*Basins are supposed to accumulate loads and hand them over to the Earth
	  for slr computations every "frequency" time steps. If we are here, we
	  have reached the "frequency"'th time step, and we are going to pick up
	  the old loads, the current loads, and send them to the Earth for slr
	  computations.  So the Earth is never supposed to compute loads. Except,
	  when we are running the Earth as a giant basin (ex: running a
	  Peltier-style GIA model, or running a "Snow Ball" Earth) which only
	  happens when we are not coupled, at which point we are sending these
	  loads to ourselves (hence the convoluted condition).
	  */

	/*transer loads from basins to Earth for grd core computations. :*/
	if(iscoupling){
	
		/*transfer ice thickness change load from basins to earth: */
		TransferForcing(femmodel,DeltaIceThicknessEnum);
		TransferForcing(femmodel,DeltaBottomPressureEnum);
		TransferForcing(femmodel,DeltaTwsEnum);

		/*transfer external forcings back to Earth:*/
		TransferSealevel(femmodel,BedEnum);
		TransferSealevel(femmodel,SealevelEnum);
		if(horiz){
			TransferSealevel(femmodel,BedEastEnum);
			TransferSealevel(femmodel,BedNorthEnum);
		}
	}
	
}; /*}}}*/
void grd_core(FemModel* femmodel){ /*{{{*/

	/*Gravity rotation deformation core GRD: */

	/*variables:*/
	Vector<IssmDouble> *RSLg    = NULL;
	Vector<IssmDouble> *RSLg_barystatic  = NULL; 
	Vector<IssmDouble> *U_grd  = NULL; 
	Vector<IssmDouble> *N_grd  = NULL; 
	Vector<IssmDouble> *U_north_grd   = NULL; 
	Vector<IssmDouble> *U_east_grd    = NULL; 
	Vector<IssmDouble> *bedrock  = NULL; 
	Vector<IssmDouble> *bedrockeast  = NULL; 
	Vector<IssmDouble> *bedrocknorth  = NULL; 
	Vector<IssmDouble> *geoid= NULL; 
	SealevelMasks* masks=NULL;

	/*parameters:*/
	int  modelid,earthid;
	int  horiz;
	IssmDouble oceanarea;
	int  count,frequency,iscoupling;
	int  grd=0;

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing GRD patterns\n");
	
	/*retrieve parameters:*/
	femmodel->parameters->FindParam(&grd,SolidearthSettingsGRDEnum); 
	femmodel->parameters->FindParam(&frequency,SolidearthSettingsRunFrequencyEnum);
	femmodel->parameters->FindParam(&count,SealevelchangeRunCountEnum);

	/*only run if grd was requested, if we are the earth, and we have reached
	 * the necessary number of time steps dictated by :*/
	if(!grd)            return;
	if(count!=frequency)return;
	femmodel->parameters->FindParam(&iscoupling,IsSlcCouplingEnum);
	if(iscoupling){
		femmodel->parameters->FindParam(&modelid,ModelIdEnum);
		femmodel->parameters->FindParam(&earthid,EarthIdEnum);
		if(modelid!=earthid)return;
	}

	/*call masks core: */
	masks=sealevel_masks(femmodel);

	/*call barystatic core  (generalized eustatic - Farrel and Clark, Eq 4, 1st, 3rd and 4rd terms on the RHS) */
	RSLg_barystatic=sealevelchange_core_barystatic(femmodel,masks,&oceanarea); 

	/*call self attraction and loading module (ocean loading tems  - 2nd and 5th terms on the RHS of Farrel and Clark) */
	RSLg=sealevelchange_core_sal(femmodel,masks,RSLg_barystatic,oceanarea); 

	/*compute bedrock motion and derive geoid: */
	sealevelchange_core_deformation(&N_grd,&U_grd,&U_north_grd,&U_east_grd,femmodel,RSLg,masks);

	/*Update bedrock motion and geoid:*/
	GetVectorFromInputsx(&geoid,femmodel,SealevelEnum,VertexSIdEnum);
	GetVectorFromInputsx(&bedrock,femmodel,BedEnum,VertexSIdEnum);
	if(horiz){
		GetVectorFromInputsx(&bedrockeast,femmodel,BedEastEnum,VertexSIdEnum);
		GetVectorFromInputsx(&bedrocknorth,femmodel,BedNorthEnum,VertexSIdEnum);
	}

	geoid->AXPY(N_grd,1);
	bedrock->AXPY(U_grd,1);
	if(horiz){
		bedrockeast->AXPY(U_east_grd,1);
		bedrocknorth->AXPY(U_north_grd,1);
	}

	/*get some of the updates into elements:*/
	InputUpdateFromVectorx(femmodel,U_grd,SealevelUGrdEnum,VertexSIdEnum); 
	InputUpdateFromVectorx(femmodel,N_grd,SealevelEnum,VertexSIdEnum); 
	InputUpdateFromVectorx(femmodel,N_grd,SealevelNGrdEnum,VertexSIdEnum); 
	if(RSLg)InputUpdateFromVectorx(femmodel,RSLg,SealevelRSLEnum,VertexSIdEnum); 
	if(RSLg_barystatic)InputUpdateFromVectorx(femmodel,RSLg_barystatic,SealevelRSLBarystaticEnum,VertexSIdEnum); 
	if (horiz){
		InputUpdateFromVectorx(femmodel,U_north_grd,SealevelUNorthEsaEnum,VertexSIdEnum);	
		InputUpdateFromVectorx(femmodel,U_east_grd,SealevelUEastEsaEnum,VertexSIdEnum);	
	} 

	/*reset counter to 1:*/
	femmodel->parameters->SetParam(1,SealevelchangeRunCountEnum); //reset counter.

	/*free ressources:{{{*/
	delete RSLg;
	delete RSLg_barystatic;
	delete U_grd;
	delete N_grd;
	delete bedrock; 
	delete geoid; 
	if(horiz){
		delete U_north_grd;
		delete U_east_grd;
		delete bedrockeast; 
		delete bedrocknorth; 
	}
	delete masks;
	/*}}}*/

} 
/*}}}*/
void dynstr_core(FemModel* femmodel){ /*{{{*/

	/*variables:*/
	Vector<IssmDouble> *sealevel  = NULL; 
	Vector<IssmDouble> *deltadsl  = NULL; 
	Vector<IssmDouble> *deltastr = NULL;

	/*parameters: */
	int  step;
	int computesealevel=0;
	bool isocean=false;
	IssmDouble time;

	IssmDouble cumgmtslc=0;
	IssmDouble cumbslc=0;
	IssmDouble cumgmslc=0;
	IssmDouble gmtslc=0;

	/*early return if we are not computing sea level, but rather deformation: */
	femmodel->parameters->FindParam(&computesealevel,SolidearthSettingsComputesealevelchangeEnum);
	if (!computesealevel)return;

	/*early return if we have no ocean transport:*/
	femmodel->parameters->FindParam(&isocean,TransientIsoceantransportEnum);
	if(!isocean)return;
	
	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing steric and dynamic sea level change\n");

	/*Retrieve sealevel and add steric + dynamic rates:*/
	GetVectorFromInputsx(&sealevel,femmodel,SealevelEnum,VertexSIdEnum);
	GetVectorFromInputsx(&deltadsl,femmodel,DeltaDslEnum,VertexSIdEnum);
	GetVectorFromInputsx(&deltastr,femmodel,DeltaStrEnum,VertexSIdEnum);

	/*compute: sea level change = initial sea level + steric + dynamic*/
	sealevel->AXPY(deltadsl,1);
	sealevel->AXPY(deltastr,1);

	/*cumulate thermal steric rate:*/
	femmodel->parameters->FindParam(&cumgmtslc,CumGmtslcEnum); 
	femmodel->parameters->FindParam(&cumbslc,CumBslcEnum); 

	gmtslc=deltastr->Norm(NORM_TWO);
	cumgmtslc+=gmtslc;
	cumgmslc=cumbslc+cumgmtslc;

	femmodel->parameters->SetParam(cumgmtslc,CumGmtslcEnum);
	femmodel->parameters->SetParam(cumgmslc,CumGmslcEnum);
	
	/*Outputs some metrics:*/
	femmodel->parameters->FindParam(&step,StepEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);

	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,GmtslcEnum,gmtslc,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumGmtslcEnum,cumgmtslc,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumGmslcEnum,cumgmslc,step,time));

	/*update element inputs:*/
	InputUpdateFromVectorx(femmodel,sealevel,SealevelEnum,VertexSIdEnum);	

	/*Free ressources:*/	
	delete sealevel;
	delete deltadsl;
	delete deltastr;
}
/*}}}*/
void coupleroutput_core(FemModel* femmodel){  /*{{{*/
	
	/*parameters:*/
	int iscoupling;
	int horiz=0;

	/*retrieve more parameters:*/
	femmodel->parameters->FindParam(&iscoupling,IsSlcCouplingEnum);
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);
		
	if(iscoupling){
		/*transfer sea level back to ice caps:*/
		TransferSealevel(femmodel,SealevelEnum);
		TransferSealevel(femmodel,BedEnum);
		if(horiz){
			TransferSealevel(femmodel,BedNorthEnum);
			TransferSealevel(femmodel,BedEastEnum);
		}
	}
}; /*}}}*/

void grd_core_optim(FemModel* femmodel) { /*{{{*/

	/*variables:{{{*/
	int nel;
	BarystaticContributions* barycontrib=NULL;
	SealevelMasks* masks=NULL;
	GenericParam<BarystaticContributions*>* barycontribparam=NULL;
	IssmDouble rotationaxismotionvector[3];
	
	Vector<IssmDouble>*    loads=NULL;
	IssmDouble*            allloads=NULL; 
	Vector<IssmDouble>*    sealevelloads=NULL;
	Vector<IssmDouble>*    oldsealevelloads=NULL;
	IssmDouble*            allsealevelloads=NULL;
	Vector<IssmDouble>*    oceanareas=NULL;
	IssmDouble             oceanarea;
	IssmDouble             oceanaverage;
	bool                   scaleoceanarea=false;
	IssmDouble             rho_water;

	IssmDouble           eps_rel;
	IssmDouble           eps_abs;
	int                  max_nonlinear_iterations;
	int                  iterations=0;
	int                  step;
	IssmDouble           time; 
	bool converged=false;

	int  modelid,earthid;
	int  horiz;
	int  count,frequency,iscoupling;
	int  grd=0;
	int  computesealevel=0;

	/*}}}*/

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing GRD patterns\n");

	/*retrieve parameters:{{{*/
	femmodel->parameters->FindParam(&grd,SolidearthSettingsGRDEnum); 
	femmodel->parameters->FindParam(&frequency,SolidearthSettingsRunFrequencyEnum);
	femmodel->parameters->FindParam(&count,SealevelchangeRunCountEnum);
	femmodel->parameters->FindParam(&computesealevel,SolidearthSettingsComputesealevelchangeEnum);
	femmodel->parameters->FindParam(&max_nonlinear_iterations,SolidearthSettingsMaxiterEnum);
	/*}}}*/

	/*only run if grd was requested, if we are the earth, and we have reached
	 * the necessary number of time steps dictated by :*/
	if(!grd)            return;
	if(count!=frequency)return;
	femmodel->parameters->FindParam(&iscoupling,IsSlcCouplingEnum);
	if(iscoupling){
		femmodel->parameters->FindParam(&modelid,ModelIdEnum);
		femmodel->parameters->FindParam(&earthid,EarthIdEnum);
		if(modelid!=earthid)return;
	}
	/*retrieve parameters: {{{*/ 
	femmodel->parameters->FindParam(&scaleoceanarea,SolidearthSettingsOceanAreaScalingEnum);
	barycontribparam = xDynamicCast<GenericParam<BarystaticContributions*>*>(femmodel->parameters->FindParamObject(BarystaticContributionsEnum));
	barycontrib=barycontribparam->GetParameterValue();
	femmodel->parameters->FindParam(&rho_water,MaterialsRhoSeawaterEnum);
	femmodel->parameters->FindParam(&eps_rel,SolidearthSettingsReltolEnum);
	femmodel->parameters->FindParam(&eps_abs,SolidearthSettingsAbstolEnum);
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);
	/*}}}*/

	/*initialize matrices and vectors:*/
	femmodel->parameters->FindParam(&nel,MeshNumberofelementsEnum);
	loads=new Vector<IssmDouble>(nel);
	sealevelloads=new Vector<IssmDouble>(nel);
	oceanareas=new Vector<IssmDouble>(nel);

	/*call masks core: */
	masks=sealevel_masks(femmodel);
	if(VerboseSolution()) _printf0_("	  starting  GRD convolutions\n");
	
	/*buildup loads: */
	for(Object* & object : femmodel->elements->objects){
		Element* element = xDynamicCast<Element*>(object);
		element->SealevelchangeBarystaticLoads(loads, barycontrib,masks); 
	}
	loads->Assemble(); 

	//broadcast loads 
	allloads=loads->ToMPISerial();

	//compute rotation axis motion:
	RotationAxisMotion(&rotationaxismotionvector[0],femmodel,allloads,NULL);

	/*skip computation of sea level if requested, which means sea level loads should be zeroed */
	if(!computesealevel){
		allsealevelloads=xNewZeroInit<IssmDouble>(nel);
		goto deformation;
	}

	/*convolve loads:*/
	for(Object* & object : femmodel->elements->objects){
		Element* element = xDynamicCast<Element*>(object);
		element->SealevelchangeInitialConvolution(sealevelloads,oceanareas,allloads,rotationaxismotionvector,masks);
	}
	sealevelloads->Assemble();
	oceanareas->Assemble();
	
	//Get ocean area: 
	oceanareas->Sum(&oceanarea); _assert_(oceanarea>0.);
	if(scaleoceanarea) oceanarea=3.619e+14; // use true ocean area, m^2

	//conserve ocean mass: 
	oceanaverage=SealevelloadsOceanAverage(sealevelloads,oceanareas,oceanarea);
	ConserveOceanMass(femmodel,sealevelloads,barycontrib->Total()/oceanarea - oceanaverage,masks);

	//broadcast sea level loads 
	allsealevelloads=sealevelloads->ToMPISerial();

	femmodel->results->AddObject(new GenericExternalResult<IssmDouble*>(femmodel->results->Size()+1,DummyEnum,allsealevelloads,nel,1,1,1));

	//compute rotation axis motion:
	RotationAxisMotion(&rotationaxismotionvector[0],femmodel,allloads,allsealevelloads);
	
	if(VerboseSolution()) _printf0_("	  converging ocean GRD convolutions\n");
	for(;;){
			
		oldsealevelloads=sealevelloads->Duplicate(); sealevelloads->Copy(oldsealevelloads);

		/*convolve load and sealevel loads on oceans:*/
		for(Object* & object : femmodel->elements->objects){
			Element* element = xDynamicCast<Element*>(object);
			element->SealevelchangeOceanConvolution(sealevelloads, allsealevelloads, allloads,rotationaxismotionvector,masks);
		}
		sealevelloads->Assemble();
			
		//Conserve ocean mass: 
		oceanaverage=SealevelloadsOceanAverage(sealevelloads,oceanareas,oceanarea);
		ConserveOceanMass(femmodel,sealevelloads,barycontrib->Total()/oceanarea - oceanaverage,masks);

		//broadcast sea level loads 
		allsealevelloads=sealevelloads->ToMPISerial();

		//compute rotation axis motion:
		RotationAxisMotion(&rotationaxismotionvector[0],femmodel,allloads,allsealevelloads);

		//convergence?
		slcconvergence(&converged,sealevelloads,oldsealevelloads,eps_rel,eps_abs);
		if (converged)break;

		//early return?
		if(iterations>max_nonlinear_iterations)break;
		else iterations++;
	}

	deformation:
	
	if(VerboseSolution()) _printf0_("	  deformation GRD convolutions\n");

	/*convolve loads and sea level loads to get the deformation:*/
	for(Object* & object : femmodel->elements->objects){
		Element* element = xDynamicCast<Element*>(object);
		element->SealevelchangeDeformationConvolution(allsealevelloads, allloads, rotationaxismotionvector,masks);
	}
	
	if(VerboseSolution()) _printf0_("	  updating GRD fields\n");

	/*Update bedrock motion and geoid:*/
	if(computesealevel){
		femmodel->inputs->Shift(SealevelGRDEnum,barycontrib->Total()/rho_water/oceanarea- oceanaverage/rho_water);

		/*cumulate barystatic contributions and save to results: */
		barycontrib->Cumulate(femmodel->parameters);
		barycontrib->Save(femmodel->results,femmodel->parameters,oceanarea);
	}

	femmodel->inputs->AXPY(1,SealevelGRDEnum,SealevelEnum);
	femmodel->inputs->AXPY(1,BedGRDEnum,BedEnum);
	if(horiz){
		femmodel->inputs->AXPY(1,BedEastGRDEnum,BedEastEnum);
		femmodel->inputs->AXPY(1,BedNorthGRDEnum, BedNorthEnum);
	}

}
/*}}}*/

//Geometry:
SealevelMasks* sealevel_masks(FemModel* femmodel) {  /*{{{*/

	int grdmodel=0; 

	/*early return?:*/
	femmodel->parameters->FindParam(&grdmodel,GrdModelEnum);
	if(grdmodel==IvinsEnum) return NULL;

	if(VerboseSolution()) _printf0_("	  computing sea level masks\n");
	
	/*initialize SealevelMasks structure: */
	SealevelMasks* masks=new SealevelMasks(femmodel->elements->Size());

	/*go through elements and fill the masks: */
	for(Object* & object : femmodel->elements->objects){
		Element*   element=xDynamicCast<Element*>(object);
		element->SetSealevelMasks(masks);
	}

	return masks;
}/*}}}*/
void sealevelchange_geometry(FemModel* femmodel) {  /*{{{*/

	/*Geometry core where we compute indices into tables pre computed in the SealevelRiseAnalysis: */

	/*parameters: */
	bool spherical=true;
	IssmDouble *latitude  = NULL;
	IssmDouble *longitude = NULL;
	IssmDouble *radius    = NULL;
	IssmDouble *xx    = NULL;
	IssmDouble *yy    = NULL;
	IssmDouble *zz    = NULL;
	IssmDouble *xxe    = NULL;
	IssmDouble *yye    = NULL;
	IssmDouble *zze    = NULL;
	IssmDouble* areae  = NULL;

	int  horiz;
	bool geometrydone = false;
	int  optim;
	int  grdmodel=0;

		
	/*retrieve parameters:*/
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);
	femmodel->parameters->FindParam(&geometrydone,SealevelchangeGeometryDoneEnum);
	femmodel->parameters->FindParam(&grdmodel,GrdModelEnum);
	femmodel->parameters->FindParam(&optim,SolidearthSettingsOptimEnum);
	
	/*early return?:*/
	if(grdmodel==IvinsEnum) return;

	if(geometrydone){
		if(VerboseSolution()) _printf0_("	  geometrical offsets have already been computed, skipping \n");
		return; //don't need to run this again.
	}

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing geometrical offsets into precomputed Green tables \n");

	/*first, recover lat,long and radius vectors from vertices: */
	VertexCoordinatesx(&latitude,&longitude,&radius,femmodel->vertices,spherical); 
	if(horiz) VertexCoordinatesx(&xx,&yy,&zz,femmodel->vertices); 
	
	/*first, recover x,y,z and areas from elements: */
	ElementCoordinatesx(&xxe,&yye,&zze,&areae,femmodel->elements);

	/*Run sealevel geometry routine in elements:*/
	for(Object* & object : femmodel->elements->objects){
		Element*   element=xDynamicCast<Element*>(object);
		if(optim) element->SealevelchangeGeometryOptim(latitude,longitude,radius,xx,yy,zz,xxe,yye,zze,areae);
		else      element->SealevelchangeGeometry(latitude,longitude,radius,xx,yy,zz,xxe,yye,zze);
	}

	/*Free ressources:*/
	if(horiz){
		xDelete<IssmDouble>(xx);
		xDelete<IssmDouble>(yy);
		xDelete<IssmDouble>(zz);
	}
	xDelete<IssmDouble>(xxe);
	xDelete<IssmDouble>(yye);
	xDelete<IssmDouble>(zze);
	xDelete<IssmDouble>(areae);
	xDelete<IssmDouble>(latitude);
	xDelete<IssmDouble>(longitude);
	xDelete<IssmDouble>(radius);

	/*Record the fact that we ran this module already: */
	femmodel->parameters->SetParam(true,SealevelchangeGeometryDoneEnum); 


}/*}}}*/

//GRD: 
Vector<IssmDouble>* sealevelchange_core_barystatic(FemModel* femmodel,SealevelMasks* masks, IssmDouble* poceanarea){ /*{{{*/

	/*Barystatic core of the SLR solution (terms that are constant with respect to sea-level)*/

	Vector<IssmDouble> *RSLgi    = NULL;
	IssmDouble          RSLgi_oceanaverage   = 0;

	/*parameters: */
	int  gsize;
	IssmDouble oceanarea;
	int        step;
	IssmDouble time;

	/*barystatic contribution:*/
	IssmDouble bslc;
	IssmDouble bslcice;
	IssmDouble* bslcice_partition=NULL;
	IssmDouble bslchydro;
	IssmDouble* bslchydro_partition=NULL;
	IssmDouble cumbslc;
	IssmDouble cumbslcice;
	IssmDouble cumbslchydro;
	IssmDouble* cumbslcice_partition=NULL;
	int npartice;
	IssmDouble* cumbslchydro_partition=NULL;
	int nparthydro;
	int computesealevel=0;

	/*early return if we are not computing sea level, but rather deformation: */
	femmodel->parameters->FindParam(&computesealevel,SolidearthSettingsComputesealevelchangeEnum);
	if (!computesealevel)return NULL;
	
	if(VerboseSolution()) _printf0_("	  computing bslc components on ice\n");

	/*Figure out size of g-set deflection vector and allocate solution vector: */
	gsize = femmodel->nodes->NumberOfDofs(GsetEnum);

	/*some parameters:*/
	femmodel->parameters->FindParam(&step,StepEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);
	femmodel->parameters->FindParam(&cumbslc,CumBslcEnum);
	femmodel->parameters->FindParam(&cumbslcice,CumBslcIceEnum);
	femmodel->parameters->FindParam(&cumbslchydro,CumBslcHydroEnum);
	femmodel->parameters->FindParam(&npartice,SolidearthNpartIceEnum);
	femmodel->parameters->FindParam(&nparthydro,SolidearthNpartHydroEnum);
	if(npartice) femmodel->parameters->FindParam(&cumbslcice_partition,&npartice,NULL,CumBslcIcePartitionEnum);
	if(nparthydro) femmodel->parameters->FindParam(&cumbslchydro_partition,&nparthydro,NULL,CumBslcHydroPartitionEnum);

	/*Initialize:*/
	RSLgi = new Vector<IssmDouble>(gsize);

	/*call the bslc main module: */
	femmodel->SealevelchangeBarystatic(RSLgi,&oceanarea,&bslc, &bslcice, &bslchydro, &bslcice_partition, &bslchydro_partition,masks); //this computes 

	/*we need to average RSLgi over the ocean: RHS term  4 in Eq.4 of Farrel and clarke. Only the elements can do that: */
	RSLgi_oceanaverage=femmodel->SealevelchangeOceanAverage(RSLgi,masks, oceanarea);

	/*RSLg is the sum of the pure bslc component (term 3) and the contribution from the perturbation to the graviation potential due to the 
	 * presence of ice (terms 1 and 4 in Eq.4 of Farrel and Clarke):*/
	RSLgi->Shift(-bslc-RSLgi_oceanaverage);

	/*save bslc and cumulated bslc value for results:{{{ */
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,BslcEnum,-bslc,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,BslcIceEnum,-bslcice,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,BslcHydroEnum,-bslchydro,step,time));

	//cumulative barystatic contribution: 
	cumbslc=cumbslc-bslc;
	cumbslcice=cumbslcice-bslcice;
	cumbslchydro=cumbslchydro-bslchydro;

	femmodel->parameters->SetParam(cumbslc,CumBslcEnum);
	femmodel->parameters->SetParam(cumbslcice,CumBslcIceEnum);
	femmodel->parameters->SetParam(cumbslchydro,CumBslcHydroEnum);

	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumBslcEnum,cumbslc,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumBslcIceEnum,cumbslcice,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumBslcHydroEnum,cumbslchydro,step,time));

	//cumulative barystatic contributions by partition:
	if(npartice){
		for(int i=0;i<npartice;i++) cumbslcice_partition[i] -= bslcice_partition[i];
		femmodel->parameters->SetParam(cumbslcice_partition,npartice,1,CumBslcIcePartitionEnum);
		femmodel->results->AddResult(new GenericExternalResult<IssmDouble*>(femmodel->results->Size()+1,CumBslcIcePartitionEnum,cumbslcice_partition,npartice,1,step,time));
	}

	if(nparthydro){
		for(int i=0;i<nparthydro;i++) cumbslchydro_partition[i] -= bslchydro_partition[i];
		femmodel->parameters->SetParam(cumbslchydro_partition,nparthydro,1,CumBslcHydroPartitionEnum);
		femmodel->results->AddResult(new GenericExternalResult<IssmDouble*>(femmodel->results->Size()+1,CumBslcHydroPartitionEnum,cumbslchydro_partition,nparthydro,1,step,time));
	}
	/*}}}*/
	
	/*Assign output pointers and return: */
	*poceanarea=oceanarea;
	return RSLgi;
}/*}}}*/
Vector<IssmDouble>* sealevelchange_core_sal(FemModel* femmodel, SealevelMasks* masks, Vector<IssmDouble>* RSLg_barystatic,IssmDouble oceanarea){ /*{{{*/

	/*this core computes the contributions from Eq.4 of Farrel and Clarke, rhs terms 2 and 5.
	  sal core of the SLR solution */

	Vector<IssmDouble> *RSLg    = NULL;
	Vector<IssmDouble> *RSLg_old    = NULL;

	Vector<IssmDouble> *RSLgo    = NULL; //ocean convolution of the perturbation to gravity potential.
	Vector<IssmDouble> *RSLgo_rot= NULL; // rotational feedback 
	IssmDouble          RSLgo_oceanaverage = 0;  //average of RSLgo over the ocean.

	/*parameters: */
	int count;
	bool save_results;
	int  gsize;
	bool converged=true;
	bool rotation=true;
	bool verboseconvolution=true;
	int max_nonlinear_iterations;
	IssmDouble           eps_rel;
	IssmDouble           eps_abs;
	IssmDouble			Ixz, Iyz, Izz; 
	int computesealevel=0;

	/*early return if we are not computing sea level, but rather deformation: */
	femmodel->parameters->FindParam(&computesealevel,SolidearthSettingsComputesealevelchangeEnum);
	if (!computesealevel)return NULL;
	
	if(VerboseSolution()) _printf0_("	  converging on ocean components\n");

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&max_nonlinear_iterations,SolidearthSettingsMaxiterEnum);
	femmodel->parameters->FindParam(&eps_rel,SolidearthSettingsReltolEnum);
	femmodel->parameters->FindParam(&eps_abs,SolidearthSettingsAbstolEnum);

	/*computational flag: */
	femmodel->parameters->FindParam(&rotation,SolidearthSettingsRotationEnum);

	/*Figure out size of g-set deflection vector and allocate solution vector: */
	gsize = femmodel->nodes->NumberOfDofs(GsetEnum);

	/*Initialize:*/
	RSLg = new Vector<IssmDouble>(gsize);
	RSLg->Assemble();
	RSLg_barystatic->Copy(RSLg);  //first initialize RSLg with the barystatic component computed in sealevelchange_core_barystatic.

	RSLg_old = new Vector<IssmDouble>(gsize);
	RSLg_old->Assemble();

	count=1;
	converged=false;

	/*Start loop: */
	for(;;){

		//save pointer to old sea level
		delete RSLg_old; RSLg_old=RSLg; 

		/*Initialize solution vector: */
		RSLg  = new Vector<IssmDouble>(gsize); RSLg->Assemble();
		RSLgo = new Vector<IssmDouble>(gsize); RSLgo->Assemble();

		/*call the sal module: */
		femmodel->SealevelchangeSal(RSLgo, RSLg_old,  masks, verboseconvolution);

		/*assemble solution vector: */
		RSLgo->Assemble(); 

		if(rotation){

			/*call rotational feedback  module: */
			RSLgo_rot = new Vector<IssmDouble>(gsize); RSLgo_rot->Assemble();
			femmodel->SealevelchangeRotationalFeedback(RSLgo_rot,RSLg_old,&Ixz,&Iyz,&Izz, masks); 
			RSLgo_rot->Assemble(); 

			/*save changes in inertia tensor as results: */
			femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,SealevelInertiaTensorXZEnum,Ixz));
			femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,SealevelInertiaTensorYZEnum,Iyz));
			femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,SealevelInertiaTensorZZEnum,Izz));

			RSLgo->AXPY(RSLgo_rot,1); 
		}

		/*we need to average RSLgo over the ocean: RHS term  5 in Eq.4 of Farrel and clarke. Only the elements can do that: */
		RSLgo_oceanaverage=femmodel->SealevelchangeOceanAverage(RSLgo,masks, oceanarea);

		/*RSLg is the sum of the barystatic term, and the ocean terms: */
		RSLg_barystatic->Copy(RSLg); RSLg->AXPY(RSLgo,1); 
		RSLg->Shift(-RSLgo_oceanaverage);

		/*convergence criterion:*/
		slcconvergence(&converged,RSLg,RSLg_old,eps_rel,eps_abs);
		
	
		/*free ressources: */
		delete RSLgo;
		delete RSLgo_rot;

		/*Increase count: */
		count++;
		if(converged==true){
			break;
		}
		if(count>=max_nonlinear_iterations){
			_printf0_("   maximum number of nonlinear iterations (" << max_nonlinear_iterations << ") exceeded\n"); 
			converged=true;
			break;
		}	

		/*some minor verbosing adjustment:*/
		if(count>1)verboseconvolution=false;

	}
	if(VerboseConvergence()) _printf0_("\n              total number of iterations: " << count-1 << "\n");
	
	
	delete RSLg_old;

	return RSLg;
} /*}}}*/
void sealevelchange_core_deformation(Vector<IssmDouble>** pN_grd, Vector<IssmDouble>** pU_grd, Vector<IssmDouble>** pU_north_grd,Vector<IssmDouble>** pU_east_grd,FemModel* femmodel,Vector<IssmDouble>* RSLg, SealevelMasks* masks){ /*{{{*/

	Vector<IssmDouble> *U_grd  = NULL; 
	Vector<IssmDouble> *dUdt_grd  = NULL; 
	Vector<IssmDouble> *N_grd  = NULL; 
	Vector<IssmDouble> *U_north_grd   = NULL; 
	Vector<IssmDouble> *U_east_grd    = NULL; 

	/*parameters: */
	int  gsize;
	bool spherical=true;

	IssmDouble          *latitude   = NULL;
	IssmDouble          *longitude  = NULL;
	IssmDouble          *radius     = NULL;
	IssmDouble          *xx     = NULL;
	IssmDouble          *yy     = NULL;
	IssmDouble          *zz     = NULL;
	int  horiz;
	int  grdmodel; 
	
	if(VerboseSolution()) _printf0_("	  computing vertical and horizontal geodetic signatures\n");

	/*retrieve some parameters:*/
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);
	femmodel->parameters->FindParam(&grdmodel,GrdModelEnum);

	/*find size of vectors:*/
	gsize      = femmodel->nodes->NumberOfDofs(GsetEnum);

	/*intialize vectors:*/
	U_grd = new Vector<IssmDouble>(gsize);
	if(grdmodel==IvinsEnum) dUdt_grd = new Vector<IssmDouble>(gsize);
	N_grd = new Vector<IssmDouble>(gsize);
	if (horiz){
		U_north_grd = new Vector<IssmDouble>(gsize);
		U_east_grd = new Vector<IssmDouble>(gsize);
	}
	
	/*retrieve geometric information: */
	VertexCoordinatesx(&latitude,&longitude,&radius,femmodel->vertices,spherical); 
	VertexCoordinatesx(&xx,&yy,&zz,femmodel->vertices); 

	/*call the deformation module: */
	switch(grdmodel){
		case NoneEnum: 
			//do nothing: 
			break;
		case IvinsEnum:
			femmodel->IvinsDeformation(U_grd,dUdt_grd,xx,yy);
			break;
		case ElasticEnum:
			femmodel->SealevelchangeDeformation(N_grd, U_grd,U_north_grd,U_east_grd,RSLg, masks);
			break;
		default:
			_error_("Grd model " << EnumToStringx(grdmodel) << " not supported yet");
	}

	/*Assign output pointers:*/
	*pU_grd=U_grd;
	*pN_grd=N_grd;
	if(horiz){
		*pU_east_grd=U_east_grd;
		*pU_north_grd=U_north_grd;
	}

	/*Free ressources: */
	xDelete<IssmDouble>(longitude);
	xDelete<IssmDouble>(latitude);
	xDelete<IssmDouble>(xx);
	xDelete<IssmDouble>(yy);
	xDelete<IssmDouble>(zz);
	xDelete<IssmDouble>(radius);
	if(grdmodel==IvinsEnum)delete dUdt_grd;
}
/*}}}*/

/*Support routines:*/
void TransferForcing(FemModel* femmodel,int forcingenum){ /*{{{*/

	/*forcing being transferred from models to earth: */
	IssmDouble** forcings=NULL;
	IssmDouble*  forcing=NULL; 
	Vector<IssmDouble>* forcingglobal=NULL; 
	int*         nvs=NULL;

	/*transition vectors:*/
	IssmDouble** transitions=NULL;
	int          ntransitions; 
	int*         transitions_m=NULL;
	int*         transitions_n=NULL;
	int          nv;
	int          existforcing=0;         

	/*communicators:*/
	ISSM_MPI_Comm tocomm;
	ISSM_MPI_Comm* fromcomms=NULL;
	ISSM_MPI_Status status;
	int         my_rank;
	int         modelid,earthid;
	int         nummodels;

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&modelid,ModelIdEnum);
	femmodel->parameters->FindParam(&earthid,EarthIdEnum);
	femmodel->parameters->FindParam(&nummodels,NumModelsEnum);
	my_rank=IssmComm::GetRank();

	/*retrieve the inter communicators that will be used to send data from each ice cap to the earth: */
	if(modelid==earthid){
		GenericParam<ISSM_MPI_Comm*>* parcoms = dynamic_cast<GenericParam<ISSM_MPI_Comm*>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcoms)_error_("TransferForcing error message: could not find IcecapToEarthComm communicator");
		fromcomms=parcoms->GetParameterValue();
	}
	else {
		GenericParam<ISSM_MPI_Comm>* parcom = dynamic_cast<GenericParam<ISSM_MPI_Comm>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcom)_error_("TransferForcing error message: could not find IcecapToEarthComm communicator");
		tocomm=parcom->GetParameterValue();
	}

	/*For each icecap, retrieve the forcing vector that will be sent to the earth model: */
	if(modelid!=earthid){
		nv=femmodel->vertices->NumberOfVertices();
		existforcing=reCast<int>(femmodel->inputs->Exist(forcingenum));
		if(existforcing)GetVectorFromInputsx(&forcing,femmodel,forcingenum,VertexSIdEnum);
	}

	/*Send the forcing to the earth model:{{{*/
	if(my_rank==0){
		if(modelid==earthid){
			forcings=xNew<IssmDouble*>(nummodels-1);
			nvs=xNew<int>(nummodels-1);
			for(int i=0;i<earthid;i++){
				ISSM_MPI_Recv(&existforcing, 1, ISSM_MPI_INT, 0,i, fromcomms[i], &status);
				if(existforcing){
					ISSM_MPI_Recv(nvs+i, 1, ISSM_MPI_INT, 0,i, fromcomms[i], &status);
					forcings[i]=xNew<IssmDouble>(nvs[i]);
					ISSM_MPI_Recv(forcings[i], nvs[i], ISSM_MPI_DOUBLE, 0,i, fromcomms[i], &status);
				}
				else{
					forcings[i]=NULL;
				}
			}

		}
		else{
			ISSM_MPI_Send(&existforcing, 1, ISSM_MPI_INT, 0, modelid, tocomm);
			if(existforcing){
				ISSM_MPI_Send(&nv, 1, ISSM_MPI_INT, 0, modelid, tocomm);
				ISSM_MPI_Send(forcing, nv, ISSM_MPI_DOUBLE, 0, modelid, tocomm);
			}
		}
	}
	/*}}}*/

	/*On the earth model, consolidate all the forcings into one, and update the elements dataset accordingly: {{{*/
	if(modelid==earthid){

		/*Out of all the delta thicknesses, build one delta thickness vector made of all the ice cap contributions. 
		 *First, build the global delta thickness vector in the earth model: */
		nv=femmodel->vertices->NumberOfVertices();
		GetVectorFromInputsx(&forcingglobal,femmodel,forcingenum,VertexSIdEnum);

		/*Retrieve transition vectors, used to plug from each ice cap into the global forcing:*/
		femmodel->parameters->FindParam(&transitions,&ntransitions,&transitions_m,&transitions_n,SealevelchangeTransitionsEnum);

		if(ntransitions!=earthid)_error_("TransferForcing error message: number of transition vectors is not equal to the number of icecaps!");

		/*Go through all the delta thicknesses coming from each ice cap: */
		if(my_rank==0){
			for(int i=0;i<earthid;i++){

				IssmDouble* forcingfromcap= forcings[i]; //careful, this only exists on rank 0 of the earth model!
				if(forcingfromcap){
					IssmDouble* transition=transitions[i];
					int         M=transitions_m[i];

					/*build index to plug values: */
					int*        index=xNew<int>(M); for(int i=0;i<M;i++)index[i]=reCast<int>(transition[i])-1; //matlab indexing!

					/*We are going to plug this vector into the earth model, at the right vertices corresponding to this particular 
					 * ice cap: */
					forcingglobal->SetValues(M,index,forcingfromcap,ADD_VAL);
					xDelete<int>(index);
				}
			}
		}

		/*Assemble vector:*/
		forcingglobal->Assemble();

		/*Plug into elements:*/
		InputUpdateFromVectorx(femmodel,forcingglobal,forcingenum,VertexSIdEnum);
	} 
	/*}}}*/

	/*Free ressources:{{{*/
	if(forcings){
		for(int i=0;i<nummodels-1;i++){
			IssmDouble* temp=forcings[i]; 
			if(temp)xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(forcings);
	}
	if(forcing)xDelete<IssmDouble>(forcing);
	if(forcingglobal)delete forcingglobal;
	if(transitions){
		for(int i=0;i<earthid;i++){
			IssmDouble* temp=transitions[i];
			xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(transitions);
		xDelete<int>(transitions_m);
		xDelete<int>(transitions_n);
	}
	if(nvs)xDelete<int>(nvs);
	/*}}}*/

} /*}}}*/
void TransferSealevel(FemModel* femmodel,int forcingenum){ /*{{{*/

	/*forcing being transferred from earth to ice caps: */
	IssmDouble*  forcing=NULL; 
	IssmDouble*  forcingglobal=NULL; 

	/*transition vectors:*/
	IssmDouble** transitions=NULL;
	int          ntransitions; 
	int*         transitions_m=NULL;
	int*         transitions_n=NULL;
	int          nv;

	/*communicators:*/
	ISSM_MPI_Comm fromcomm;
	ISSM_MPI_Comm* tocomms=NULL;
	ISSM_MPI_Status status;
	int         my_rank;
	int         modelid,earthid;
	int         nummodels;
	int         numcoms;

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&modelid,ModelIdEnum);
	femmodel->parameters->FindParam(&earthid,EarthIdEnum);
	femmodel->parameters->FindParam(&nummodels,NumModelsEnum);
	my_rank=IssmComm::GetRank();

	/*retrieve the inter communicators that will be used to send data from earth to ice caps:*/
	if(modelid==earthid){
		GenericParam<ISSM_MPI_Comm*>* parcoms = dynamic_cast<GenericParam<ISSM_MPI_Comm*>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcoms)_error_("TransferSealevel error message: could not find IcecapToEarthComm communicator");
		tocomms=parcoms->GetParameterValue();
		//femmodel->parameters->FindParam((int**)(&tocomms),&numcoms,IcecapToEarthCommEnum);
	}
	else{
		GenericParam<ISSM_MPI_Comm>* parcom = dynamic_cast<GenericParam<ISSM_MPI_Comm>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcom)_error_("TransferSealevel error message: could not find IcecapToEarthComm communicator");
		fromcomm=parcom->GetParameterValue();
		//femmodel->parameters->FindParam((int*)(&fromcomm), IcecapToEarthCommEnum);
	}

	/*Retrieve sea-level on earth model: */
	if(modelid==earthid){
		nv=femmodel->vertices->NumberOfVertices();
		GetVectorFromInputsx(&forcingglobal,femmodel,forcingenum,VertexSIdEnum);
	}

	/*Send the forcing to the ice caps:{{{*/
	if(my_rank==0){

		if(modelid==earthid){

			/*Retrieve transition vectors, used to figure out global forcing contribution to each ice cap's own elements: */
			femmodel->parameters->FindParam(&transitions,&ntransitions,&transitions_m,&transitions_n,SealevelchangeTransitionsEnum);

			if(ntransitions!=earthid)_error_("TransferSealevel error message: number of transition vectors is not equal to the number of icecaps!");

			for(int i=0;i<earthid;i++){
				nv=transitions_m[i];
				forcing=xNew<IssmDouble>(nv);
				IssmDouble* transition=transitions[i];
				for(int j=0;j<nv;j++){
					forcing[j]=forcingglobal[reCast<int>(transition[j])-1];
				}
				ISSM_MPI_Send(&nv, 1, ISSM_MPI_INT, 0, i, tocomms[i]);
				ISSM_MPI_Send(forcing, nv, ISSM_MPI_DOUBLE, 0, i, tocomms[i]);
			}
		}
		else{
			ISSM_MPI_Recv(&nv, 1, ISSM_MPI_INT, 0, modelid, fromcomm, &status);
			forcing=xNew<IssmDouble>(nv);
			ISSM_MPI_Recv(forcing, nv, ISSM_MPI_DOUBLE, 0, modelid, fromcomm, &status);
		}
	}
	/*}}}*/

	/*On each ice cap, spread the forcing across cpus, and update the elements dataset accordingly: {{{*/
	if(modelid!=earthid){

		ISSM_MPI_Bcast(&nv,1,ISSM_MPI_INT,0,IssmComm::GetComm());
		if(my_rank!=0)forcing=xNew<IssmDouble>(nv);
		ISSM_MPI_Bcast(forcing,nv,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());

		/*Plug into elements:*/
		InputUpdateFromVectorx(femmodel,forcing,forcingenum,VertexSIdEnum);
	} 
	/*}}}*/

	/*Free ressources:{{{*/
	if(forcingglobal)xDelete<IssmDouble>(forcingglobal);
	if(forcing)xDelete<IssmDouble>(forcing);
	if(transitions){
		for(int i=0;i<ntransitions;i++){
			IssmDouble* temp=transitions[i];
			xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(transitions);
		xDelete<int>(transitions_m);
		xDelete<int>(transitions_n);
	}
	/*}}}*/

} /*}}}*/
void slcconvergence(bool* pconverged, Vector<IssmDouble>* RSLg,Vector<IssmDouble>* RSLg_old,IssmDouble eps_rel,IssmDouble eps_abs){ /*{{{*/

	bool converged=true;
	IssmDouble ndS,nS; 
	Vector<IssmDouble> *dRSLg    = NULL;

	//compute norm(du) and norm(u) if requested
	dRSLg=RSLg_old->Duplicate(); RSLg_old->Copy(dRSLg); dRSLg->AYPX(RSLg,-1.0);
	ndS=dRSLg->Norm(NORM_TWO); 

	if (xIsNan<IssmDouble>(ndS)) _error_("convergence criterion is NaN!");

	if(!xIsNan<IssmDouble>(eps_rel)){
		nS=RSLg_old->Norm(NORM_TWO);
		if (xIsNan<IssmDouble>(nS)) _error_("convergence criterion is NaN!");
	}

	//clean up
	delete dRSLg;

	//print
	if(!xIsNan<IssmDouble>(eps_rel)){
		if((ndS/nS)<eps_rel){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "              convergence criterion: norm(dS)/norm(S)" << ndS/nS*100 << " < " << eps_rel*100 << " %\n");
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "              convergence criterion: norm(dS)/norm(S)" << ndS/nS*100 << " > " << eps_rel*100 << " %\n");
			converged=false;
		}
	}
	if(!xIsNan<IssmDouble>(eps_abs)){
		if(ndS<eps_abs){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "              convergence criterion: norm(dS)" << ndS << " < " << eps_abs << " \n");
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "              convergence criterion: norm(dS)" << ndS << " > " << eps_abs << " \n");
			converged=false;
		}
	}

	/*assign output*/
	*pconverged=converged;

} /*}}}*/
IssmDouble SealevelloadsOceanAverage(Vector<IssmDouble>* sealevelloads,Vector<IssmDouble>* oceanareas, IssmDouble oceanarea){ /*{{{*/

	IssmDouble sealevelloadsaverage;	

	Vector<IssmDouble>* sealevelloadsvolume=sealevelloads->Duplicate();
	sealevelloadsvolume->PointwiseMult(sealevelloads,oceanareas);
	sealevelloadsvolume->Sum(&sealevelloadsaverage);
	delete sealevelloadsvolume; 
	
	return sealevelloadsaverage/oceanarea;
} /*}}}*/
void RotationAxisMotion(IssmDouble* m, FemModel* femmodel,IssmDouble* loads, IssmDouble* sealevelloads){ /*{{{*/

	IssmDouble  moi_list[3]={0,0,0};
	IssmDouble  moi_list_cpu[3]={0,0,0};
	IssmDouble*	tide_love_h  = NULL;
	IssmDouble*	tide_love_k  = NULL;
	IssmDouble*	load_love_k  = NULL;
	IssmDouble  tide_love_k2secular;
	IssmDouble  moi_e, moi_p;
	IssmDouble	m1, m2, m3;
	bool rotation=false;
	
	/*early return?:*/
	femmodel->parameters->FindParam(&rotation,SolidearthSettingsRotationEnum);
	if(!rotation)return;

	/*retrieve parameters: */
	femmodel->parameters->FindParam(&load_love_k,NULL,NULL,LoadLoveKEnum);
	femmodel->parameters->FindParam(&tide_love_h,NULL,NULL,TidalLoveHEnum);
	femmodel->parameters->FindParam(&tide_love_k,NULL,NULL,TidalLoveKEnum);
	femmodel->parameters->FindParam(&tide_love_k2secular,TidalLoveK2SecularEnum);
	femmodel->parameters->FindParam(&moi_e,RotationalEquatorialMoiEnum);
	femmodel->parameters->FindParam(&moi_p,RotationalPolarMoiEnum);

	for(Object* & object : femmodel->elements->objects){
		Element* element = xDynamicCast<Element*>(object);
		element->SealevelchangeMomentOfInertiaElement(&moi_list[0],loads,sealevelloads);
		moi_list_cpu[0] += moi_list[0];
		moi_list_cpu[1] += moi_list[1];
		moi_list_cpu[2] += moi_list[2];
	}

	ISSM_MPI_Reduce (&moi_list_cpu[0],&moi_list[0],1,ISSM_MPI_DOUBLE,ISSM_MPI_SUM,0,IssmComm::GetComm() );
	ISSM_MPI_Bcast(&moi_list[0],1,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());
	
	ISSM_MPI_Reduce (&moi_list_cpu[1],&moi_list[1],1,ISSM_MPI_DOUBLE,ISSM_MPI_SUM,0,IssmComm::GetComm() );
	ISSM_MPI_Bcast(&moi_list[1],1,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());
	
	ISSM_MPI_Reduce (&moi_list_cpu[2],&moi_list[2],1,ISSM_MPI_DOUBLE,ISSM_MPI_SUM,0,IssmComm::GetComm() );
	ISSM_MPI_Bcast(&moi_list[2],1,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());

	/*compute perturbation terms for angular velocity vector: */
	m1 = 1/(1-tide_love_k[2]/tide_love_k2secular) * (1+load_love_k[2])/(moi_p-moi_e) * moi_list[0];
	m2 = 1/(1-tide_love_k[2]/tide_love_k2secular) * (1+load_love_k[2])/(moi_p-moi_e) * moi_list[1];
	m3 = -(1+load_love_k[2])/moi_p * moi_list[2];	// term associated with fluid number (3-order-of-magnitude smaller) is negelected

	/*Assign output pointers:*/
	m[0]=m1;
	m[1]=m2;
	m[2]=m3;
} /*}}}*/
void ConserveOceanMass(FemModel* femmodel,Vector<IssmDouble>* sealevelloads, IssmDouble offset, SealevelMasks* masks){ /*{{{*/

	/*Shift sealevel loads by ocean average, only on ocean! :*/
	for(Object* & object : femmodel->elements->objects){
		Element* element = xDynamicCast<Element*>(object);
		element->SealevelchangeShift(sealevelloads,offset,masks);
	}
	sealevelloads->Assemble();

} /*}}}*/
