%LOVENUMBERS numbers class definition
%
%   Usage:
%      lovenumbers=lovenumbers() 
%      lovenumbers=lovenumbers('maxdeg',10000,'referenceframe','CF'); 
% 
%   choose numbers of degrees required (1000 by default) and reference frame (between CF and CM; CM by default) 

classdef lovenumbers
	properties (SetAccess=public) 
		
		%regular love numbers:
		h           = []; %provided by PREM model
		k           = []; %idem
		l           = []; %idem
		
		%tidal love numbers for computing rotational feedback:
		th          = []; 
		tk          = []; 
		tl          = []; 
		tk2secular  = 0;  %deg 2 secular number.

	end
	methods
		function self = lovenumbers(varargin) % {{{
			options=pairoptions(varargin{:});
			maxdeg=getfieldvalue(options,'maxdeg',1000);
			referenceframe=getfieldvalue(options,'referenceframe','CM');
			self=setdefaultparameters(self,maxdeg,referenceframe);
		end % }}}
		function self = setdefaultparameters(self,maxdeg,referenceframe) % {{{
		
			%initialize love numbers:
			self.h=getlovenumbers('type','loadingverticaldisplacement','referenceframe',referenceframe,'maxdeg',maxdeg);
			self.k=getlovenumbers('type','loadinggravitationalpotential','referenceframe',referenceframe,'maxdeg',maxdeg);
			self.l=getlovenumbers('type','loadinghorizontaldisplacement','referenceframe',referenceframe,'maxdeg',maxdeg);
			self.th=getlovenumbers('type','tidalverticaldisplacement','referenceframe',referenceframe,'maxdeg',maxdeg);
			self.tk=getlovenumbers('type','tidalgravitationalpotential','referenceframe',referenceframe,'maxdeg',maxdeg);
			self.tl=getlovenumbers('type','tidalhorizontaldisplacement','referenceframe',referenceframe,'maxdeg',maxdeg);

			%secular fluid love number: 
			self.tk2secular=0.942; 


		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			if ~ismember('SealevelchangeAnalysis',analyses) | (strcmp(solution,'TransientSolution') & md.transient.isslc==0), 
				return; 
			end

			md = checkfield(md,'fieldname','solidearth.lovenumbers.h','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','solidearth.lovenumbers.k','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','solidearth.lovenumbers.l','NaN',1,'Inf',1);
			
			
			md = checkfield(md,'fieldname','solidearth.lovenumbers.th','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','solidearth.lovenumbers.tk','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','solidearth.lovenumbers.tk2secular','NaN',1,'Inf',1);

			%check that love numbers are provided at the same level of accuracy: 
			if (size(self.h,1)~=size(self.k,1) | size(self.h,1)~=size(self.l,1)),
				error('lovenumbers error message: love numbers should be provided at the same level of accuracy');
			end
			

		end % }}}
		function list=defaultoutputs(self,md) % {{{
			list = {};
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   lovenumbers parameters:'));

			fielddisplay(self,'h','load Love number for radial displacement');
			fielddisplay(self,'k','load Love number for gravitational potential perturbation');
			fielddisplay(self,'l','load Love number for horizontal displacements');

			fielddisplay(self,'th','tidal load Love number (deg 2)');
			fielddisplay(self,'tk','tidal load Love number (deg 2)');
			fielddisplay(self,'tk2secular','secular fluid Love number');

		end % }}}
		function marshall(self,prefix,md,fid) % {{{

			WriteData(fid,prefix,'object',self,'fieldname','h','name','md.solidearth.lovenumbers.h','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','k','name','md.solidearth.lovenumbers.k','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','l','name','md.solidearth.lovenumbers.l','format','DoubleMat','mattype',1);

			WriteData(fid,prefix,'object',self,'fieldname','th','name','md.solidearth.lovenumbers.th','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','tk','name','md.solidearth.lovenumbers.tk','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','tl','name','md.solidearth.lovenumbers.tl','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'data',self.tk2secular,'fieldname','lovenumbers.tk2secular','format','Double');

		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
			writejs1Darray(fid,[modelname '.lovenumbers.h'],self.h);
			writejs1Darray(fid,[modelname '.lovenumbers.k'],self.k);
			writejs1Darray(fid,[modelname '.lovenumbers.l'],self.l);
		end % }}}
		function self = extrude(self,md) % {{{
		end % }}}
	end
end
