/*---------------------------------------------------------
  hilbert.c contains all sorting or renumeration functions 
   - Hilbert or edge based vertex renumeration
   - Implicit ordered hashing renumeration for edges, 
  triangles and tetrahedra
---------------------------------------------------------*/
#include "hilbert.h"


/* Classify in an increasing order depending on the rank array
  The idea of the heap approach is to build a flat binary tree */
static int heapRank(int siz,int *rank,int *vid) {
  int    i,j,r,l;
  int    crit[2];

  if ( siz <= 1 )  return(0);
  
  l = siz / 2 + 1;
  r = siz;

  while ( r != 1 ) {  
    if ( l > 1 ) {
      /* save state l */
      l       = l-1;
      crit[0] = rank[l];
      crit[1] = vid[l];
    }
    else {
      /* save state r and put 1 in r */
      crit[0] = rank[r];
      crit[1] = vid[r];
      rank[r] = rank[1];
      vid[r]  = vid[1];
      r = r-1;
      if ( r == 1 ) 
        break;
    }
    j = l;
    i = j;
    j = 2*j;

    while ( j <= r ) {
      if ( j != r ) {   
        if ( rank[j] < rank[j+1] ) 
	  j = j+1;
      }
      
      if ( crit[0] < rank[j] ) {
        /* put j in i (i is save) */ 
        rank[i] = rank[j];
        vid[i]  = vid[j];
        i = j;
        j = 2*j;
      }
      else
        break;    
    }
    
    /* put save state in i */
    rank[i] = crit[0];
    vid[i]  = crit[1];
    continue;

  }
  rank[1] = crit[0];
  vid[1]  = crit[1];

  return(1);
}


static int hilbert2d(double *p,double *cub,int code,int depth,int *power) {
  double  a[2],b[2],c[2],d[2],cube[8];
  int     pcode,res,ncode;
  double  w[2],v1[2],v2[2];
  double  v1nrm,v2nrm,alpha,beta;
  
  if ( depth <= 0 ) {
    pcode = code;
    return(pcode);
  }

  else {
    depth = depth-1;

    /* set square */
    a[0] = cub[0];  a[1] = cub[1];
    b[0] = cub[2];  b[1] = cub[3];
    c[0] = cub[4];  c[1] = cub[5];
    d[0] = cub[6];  d[1] = cub[7];
  
    w[0]  = p[0]-a[0];	w[1]  = p[1]-a[1];
    v1[0] = d[0]-a[0];	v1[1] = d[1]-a[1];
    v2[0] = b[0]-a[0];	v2[1] = b[1]-a[1];
    
    v1nrm = v1[0]*v1[0] + v1[1]*v1[1];
    v2nrm = v2[0]*v2[0] + v2[1]*v2[1];
    
    alpha = (w[0]*v1[0] + w[1]*v1[1]) / v1nrm;
    beta  = (w[0]*v2[0] + w[1]*v2[1]) / v2nrm;
    
    if ( alpha < 0.5 ) {
      if ( beta < 0.5 ) 
        res = 1;
      else 
        res = 2;
    }
    else {
      if ( beta > 0.5 ) 
        res = 3;
      else 
        res = 4;
    }
    
    switch (res) {
    case 1: /* p closer to a */
      ncode = code;
      cube[0] = a[0];             cube[1] = a[1];
      cube[2] = 0.5*(a[0]+d[0]);  cube[3] = 0.5*(a[1]+d[1]);
      cube[4] = 0.5*(a[0]+c[0]);  cube[5] = 0.5*(a[1]+c[1]);
      cube[6] = 0.5*(a[0]+b[0]);  cube[7] = 0.5*(a[1]+b[1]);
      pcode = hilbert2d(p,cube,ncode,depth,power);
      break;
    
    case 2: /* p closer to b */
      ncode = code + power[depth];
      cube[0] = 0.5*(a[0]+b[0]);  cube[1] = 0.5*(a[1]+b[1]);
      cube[2] = b[0];             cube[3] = b[1];
      cube[4] = 0.5*(b[0]+c[0]);  cube[5] = 0.5*(b[1]+c[1]);
      cube[6] = 0.5*(b[0]+d[0]);  cube[7] = 0.5*(b[1]+d[1]);
      pcode = hilbert2d(p,cube,ncode,depth,power);
      break;
    
    case 3: /* p closer to c */
      ncode = code + 2*power[depth];
      cube[0] = 0.5*(a[0]+c[0]);  cube[1] = 0.5*(a[1]+c[1]);
      cube[2] = 0.5*(b[0]+c[0]);  cube[3] = 0.5*(b[1]+c[1]);
      cube[4] = c[0];             cube[5] = c[1];
      cube[6] = 0.5*(c[0]+d[0]);  cube[7] = 0.5*(c[1]+d[1]);
      pcode = hilbert2d(p,cube,ncode,depth,power);
      break;

    case 4: /* p is closer to d */
      ncode = code + 3*power[depth];
      cube[0] = 0.5*(d[0]+c[0]);  cube[1] = 0.5*(d[1]+c[1]);
      cube[2] = 0.5*(b[0]+d[0]);  cube[3] = 0.5*(b[1]+d[1]);
      cube[4] = 0.5*(a[0]+d[0]);  cube[5] = 0.5*(a[1]+d[1]);
      cube[6] = d[0];             cube[7] = d[1];
      pcode = hilbert2d(p,cube,ncode,depth,power);
      break;
    }      
  }
  
  return(pcode);
}


/* RQ1: pseudo optimal prof:
  double lon  = xmax-xmin;
  double prof = log(lon/hMin) / log(2.);
  RQ2: a sort on the rank of each coordinate is not good at all 
       if we don't consider any equalities, ie when rank goes from 1 to nbv. */
int hilbertSort2d(double *coor,int *old,int *new,int np,int depth) {
  int      *rank,k,iad,power[15];
  double    cub[8],xmin,xmax,ymin,ymax;

  depth = H_MAX(10,(H_MIN(depth,15)));
  rank  = (int*)calloc(np+1,sizeof(int));
  assert(rank);

  for (k=0; k<=depth; k++)  power[k] = pow(4,k);

  xmin = xmax = (double)coor[2];
  ymin = ymax = (double)coor[3];
  old[1] = 1;
  iad  = 4;
  for (k=2; k<=np; k++) {    
    xmin = H_MIN(xmin,coor[iad]);
    xmax = H_MAX(xmax,coor[iad]);
    ymin = H_MIN(ymin,coor[iad+1]);
    ymax = H_MAX(ymax,coor[iad+1]);
    iad += 2;
    old[k] = new[k] = k;
  }

  /* init */
  cub[0] = xmin;  cub[1] = ymin;
  cub[2] = xmin;  cub[3] = ymax;
  cub[4] = xmax;  cub[5] = ymax;
  cub[6] = xmax;  cub[7] = ymin;

  /* hilbert set the rank of each vertex, ie the case where it is located */
  new[0]  = old[0] = 0;
  rank[0] = -1;
  iad     = 2;
  for (k=1; k<=np; k++) {
    rank[k] = hilbert2d(&coor[iad],cub,0,depth,power);
    if ( rank[k] < 0 ) {
      fprintf(stderr,"\n  ## VERTEX RENUMERATION: RANK NEGATIVE.\n");
      return(0);
    }
    iad += 2;
  }

  heapRank(np,rank,old);

  for (k=0; k<=np; k++)  new[old[k]] = k;

  free(rank);
  return(1);
}












/*

   HILBERT RENUMERATION TO IMPROVE CACHE MISS AND INDIRECT ADRESSING

*/
int Hilbert3d(double *p, double* a, double* b, double* c, double* d, 
              double* e, double* f, double* g, double* h, int code, int prof)
{
//  double p[3],a[3],b[3],c[3],d[3],e[3],f[3],g[3],h[3];
  double aa[3],bb[3],cc[3],dd[3],ee[3],ff[3],gg[3],hh[3];
  int    i,pcode,res,ncode;
  double w[3],v1[3],v2[3],v3[3];
  double v1nrm,v2nrm,v3nrm,alpha,beta,gamma;
  
  //printf("hilbert: %d  %d\n",prof,code);

  if ( prof <= 0 ) {
    pcode = code;
    //printf(" pcode = %d\n",pcode);
    return pcode;
  }

  else {
    
    prof = prof-1;

    for (i=0; i<3; ++i) {
      w[i]  = p[i]-a[i];
      v1[i] = d[i]-a[i];
      v2[i] = b[i]-a[i];
      v3[i] = h[i]-a[i];
    }
    
    v1nrm = v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2];
    v2nrm = v2[0]*v2[0] + v2[1]*v2[1] + v2[2]*v2[2];
    v3nrm = v3[0]*v3[0] + v3[1]*v3[1] + v3[2]*v3[2];
    
    alpha = ( w[0]*v1[0] + w[1]*v1[1] + w[2]*v1[2] ) / v1nrm;
    beta  = ( w[0]*v2[0] + w[1]*v2[1] + w[2]*v2[2] ) / v2nrm;
    gamma = ( w[0]*v3[0] + w[1]*v3[1] + w[2]*v3[2] ) / v3nrm;
    
    if ( alpha < 0.5 ) {
      if ( beta < 0.5 ) {
        if ( gamma < 0.5 ) 
          res = 1;
        else 
          res = 8;
      }
      else {
        if ( gamma < 0.5 ) 
          res = 2;
        else 
          res = 7;
      }
    }
    else {
      if ( beta > 0.5 ) {
        if ( gamma < 0.5 ) 
          res = 3;
        else 
          res = 6;
      }
      else {
        if ( gamma < 0.5 ) 
          res = 4;
        else 
          res = 5;
      }
    }
    
    //printf("hilbert: %d  %d  %d\n",prof,res,code);
    
    switch (res) {
    case 1:	// p is closer to a
      for (i=0; i<3; ++i) {
        aa[i] = a[i];
	bb[i] = 0.5*(a[i]+h[i]);
        cc[i] = 0.5*(a[i]+e[i]);
        dd[i] = 0.5*(a[i]+d[i]);
	ee[i] = 0.5*(a[i]+c[i]);
        ff[i] = 0.5*(a[i]+f[i]);
        gg[i] = 0.5*(a[i]+g[i]);
        hh[i] = 0.5*(a[i]+b[i]);
      }
      ncode = code + 0*(pow(8,prof));
      pcode = Hilbert3d(p,aa,bb,cc,dd,ee,ff,gg,hh,ncode,prof);
      break;
    
    case 2:	// p is closer to b
      for (i=0; i<3; ++i) {
        aa[i] = 0.5*(b[i]+a[i]);
	bb[i] = b[i];
        cc[i] = 0.5*(b[i]+g[i]);
        dd[i] = 0.5*(a[i]+g[i]);
	ee[i] = 0.5*(a[i]+f[i]);
        ff[i] = 0.5*(b[i]+f[i]);
        gg[i] = 0.5*(b[i]+c[i]);
        hh[i] = 0.5*(a[i]+c[i]);
      }
      ncode = code + 1*(pow(8,prof));
      pcode = Hilbert3d(p,aa,bb,cc,dd,ee,ff,gg,hh,ncode,prof);
      break;
    
    case 3:	// p is closer to c
        aa[i] = 0.5*(c[i]+a[i]);
	bb[i] = 0.5*(c[i]+b[i]);
        cc[i] = c[i];
        dd[i] = 0.5*(c[i]+d[i]);
	ee[i] = 0.5*(c[i]+e[i]);
        ff[i] = 0.5*(c[i]+f[i]);
        gg[i] = 0.5*(b[i]+f[i]);
        hh[i] = 0.5*(a[i]+f[i]);
      ncode = code + 2*(pow(8,prof));
      pcode = Hilbert3d(p,aa,bb,cc,dd,ee,ff,gg,hh,ncode,prof);
      break;

    case 4:	// p is closer to d
      for (i=0; i<3; ++i) {
        aa[i] = 0.5*(a[i]+f[i]);
	bb[i] = 0.5*(a[i]+c[i]);
        cc[i] = 0.5*(a[i]+d[i]);
        dd[i] = 0.5*(a[i]+e[i]);
	ee[i] = 0.5*(d[i]+e[i]);
        ff[i] = d[i];
        gg[i] = 0.5*(d[i]+c[i]);
        hh[i] = 0.5*(c[i]+e[i]);
      }
      ncode = code + 3*(pow(8,prof));
      pcode = Hilbert3d(p,aa,bb,cc,dd,ee,ff,gg,hh,ncode,prof);
      break;
    
    case 5:	// p is closer to e
      for (i=0; i<3; ++i) {
        aa[i] = 0.5*(e[i]+c[i]);
	bb[i] = 0.5*(e[i]+f[i]);
        cc[i] = e[i];
        dd[i] = 0.5*(e[i]+d[i]);
	ee[i] = 0.5*(e[i]+a[i]);
        ff[i] = 0.5*(e[i]+h[i]);
        gg[i] = 0.5*(e[i]+g[i]);
        hh[i] = 0.5*(a[i]+f[i]);
      }
      ncode = code + 4*(pow(8,prof));
      pcode = Hilbert3d(p,aa,bb,cc,dd,ee,ff,gg,hh,ncode,prof);
      break;
    
    case 6:	// p is closer to f
      for (i=0; i<3; ++i) {
        aa[i] = 0.5*(f[i]+a[i]);
	bb[i] = 0.5*(f[i]+b[i]);
        cc[i] = 0.5*(f[i]+c[i]);
        dd[i] = 0.5*(e[i]+c[i]);
	ee[i] = 0.5*(f[i]+e[i]);
        ff[i] = f[i];
        gg[i] = 0.5*(f[i]+g[i]);
        hh[i] = 0.5*(e[i]+g[i]);
      }
      ncode = code + 5*(pow(8,prof));
      pcode = Hilbert3d(p,aa,bb,cc,dd,ee,ff,gg,hh,ncode,prof);
      break;
    
    case 7:	// p is closer to g
      for (i=0; i<3; ++i) {
        aa[i] = 0.5*(g[i]+e[i]);
	bb[i] = 0.5*(g[i]+f[i]);
        cc[i] = 0.5*(f[i]+b[i]);
        dd[i] = 0.5*(f[i]+a[i]);
	ee[i] = 0.5*(g[i]+a[i]);
        ff[i] = 0.5*(g[i]+b[i]);
        gg[i] = g[i];
        hh[i] = 0.5*(g[i]+h[i]);
      }
      ncode = code + 6*(pow(8,prof));
      pcode = Hilbert3d(p,aa,bb,cc,dd,ee,ff,gg,hh,ncode,prof);
      break;
    
    case 8:	// p is closer to h
      for (i=0; i<3; ++i) {
        aa[i] = 0.5*(h[i]+g[i]);
	bb[i] = 0.5*(a[i]+g[i]);
        cc[i] = 0.5*(a[i]+f[i]);
        dd[i] = 0.5*(e[i]+g[i]);
	ee[i] = 0.5*(h[i]+e[i]);
        ff[i] = 0.5*(a[i]+e[i]);
        gg[i] = 0.5*(h[i]+a[i]);
        hh[i] = h[i];
      }
      ncode = code + 7*(pow(8,prof));
      pcode = Hilbert3d(p,aa,bb,cc,dd,ee,ff,gg,hh,ncode,prof);
      break;
    }      

  }
  
  return pcode;

}





/*

   HILBERT RENUMERATION TO IMPROVE CACHE MISS AND INDIRECT ADRESSING

*/
int Hilbert3d2(double* cub, int code, int prof, int *power)
{
//  double cub[27];	// p is (cub[24],cub[25],cub[26])
  double cube[27];
  int    i,pcode,res,ncode;
  double w[3],v1[3],v2[3],v3[3];
  double v1nrm,v2nrm,v3nrm,alpha,beta,gamma;
  
  //printf("hilbert: %d  %d\n",prof,code);

  if ( prof <= 0 ) {
    pcode = code;
    //printf(" pcode = %d\n",pcode);
    return pcode;
  }

  else {
    
    prof = prof-1;

    for (i=0; i<3; ++i) {
      w[i]  = cub[24+i] - cub[i];
      v1[i] = cub[9+i]  - cub[i];
      v2[i] = cub[3+i]  - cub[i];
      v3[i] = cub[21+i] - cub[i];
    }
    
    v1nrm = v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2];
    v2nrm = v2[0]*v2[0] + v2[1]*v2[1] + v2[2]*v2[2];
    v3nrm = v3[0]*v3[0] + v3[1]*v3[1] + v3[2]*v3[2];
    
    alpha = ( w[0]*v1[0] + w[1]*v1[1] + w[2]*v1[2] ) / v1nrm;
    beta  = ( w[0]*v2[0] + w[1]*v2[1] + w[2]*v2[2] ) / v2nrm;
    gamma = ( w[0]*v3[0] + w[1]*v3[1] + w[2]*v3[2] ) / v3nrm;
    
    if ( alpha < 0.5 ) {
      if ( beta < 0.5 ) {
        if ( gamma < 0.5 ) 
          res = 1;
        else 
          res = 8;
      }
      else {
        if ( gamma < 0.5 ) 
          res = 2;
        else 
          res = 7;
      }
    }
    else {
      if ( beta > 0.5 ) {
        if ( gamma < 0.5 ) 
          res = 3;
        else 
          res = 6;
      }
      else {
        if ( gamma < 0.5 ) 
          res = 4;
        else 
          res = 5;
      }
    }
    
    //printf("hilbert: %d  %d  %d\n",prof,res,code);
    
    switch (res) {
    case 1:	// p is closer to a
      for (i=0; i<3; ++i) {
        cube[i]    = cub[i];
	cube[3+i]  = 0.5*(cub[i]+cub[21+i]);
        cube[6+i]  = 0.5*(cub[i]+cub[12+i]);
        cube[9+i]  = 0.5*(cub[i]+cub[9+i]);
	cube[12+i] = 0.5*(cub[i]+cub[6+i]);
        cube[15+i] = 0.5*(cub[i]+cub[15+i]);
        cube[18+i] = 0.5*(cub[i]+cub[18+i]);
        cube[21+i] = 0.5*(cub[i]+cub[3+i]);
	cube[24+i] = cub[24+i];
      }
      ncode = code; // + 0*power[prof];
      pcode = Hilbert3d2(cube,ncode,prof,power);
      break;
    
    case 2:	// p is closer to b
      for (i=0; i<3; ++i) {
        cube[i]    = 0.5*(cub[3+i]+cub[i]);
	cube[3+i]  = cub[3+i];
        cube[6+i]  = 0.5*(cub[3+i]+cub[18+i]);
        cube[9+i]  = 0.5*(cub[i]+cub[18+i]);
	cube[12+i] = 0.5*(cub[i]+cub[15+i]);
        cube[15+i] = 0.5*(cub[3+i]+cub[15+i]);
        cube[18+i] = 0.5*(cub[3+i]+cub[6+i]);
        cube[21+i] = 0.5*(cub[i]+cub[6+i]);
	cube[24+i] = cub[24+i];
      }
      ncode = code + power[prof]; //1*power[prof];
      pcode = Hilbert3d2(cube,ncode,prof,power);
      break;
    
    case 3:	// p is closer to c
      for (i=0; i<3; ++i) {
        cube[i]    = 0.5*(cub[6+i]+cub[i]);
	cube[3+i]  = 0.5*(cub[6+i]+cub[3+i]);
        cube[6+i]  = cub[6+i];
        cube[9+i]  = 0.5*(cub[6+i]+cub[9+i]);
	cube[12+i] = 0.5*(cub[6+i]+cub[12+i]);
        cube[15+i] = 0.5*(cub[6+i]+cub[15+i]);
        cube[18+i] = 0.5*(cub[3+i]+cub[15+i]);
        cube[21+i] = 0.5*(cub[i]+cub[15+i]);
	cube[24+i] = cub[24+i];
      }
      ncode = code + 2*power[prof];
      pcode = Hilbert3d2(cube,ncode,prof,power);
      break;

    case 4:	// p is closer to d
      for (i=0; i<3; ++i) {
        cube[i]    = 0.5*(cub[i]+cub[15+i]);
	cube[3+i]  = 0.5*(cub[i]+cub[6+i]);
        cube[6+i]  = 0.5*(cub[i]+cub[9+i]);
        cube[9+i]  = 0.5*(cub[i]+cub[12+i]);
	cube[12+i] = 0.5*(cub[9+i]+cub[12+i]);
        cube[15+i] = cub[9+i];
        cube[18+i] = 0.5*(cub[9+i]+cub[6+i]);
        cube[21+i] = 0.5*(cub[6+i]+cub[12+i]);
	cube[24+i] = cub[24+i];
      }
      ncode = code + 3*power[prof];
      pcode = Hilbert3d2(cube,ncode,prof,power);
      break;
    
    case 5:	// p is closer to e
      for (i=0; i<3; ++i) {
        cube[i]    = 0.5*(cub[12+i]+cub[6+i]);
	cube[3+i]  = 0.5*(cub[12+i]+cub[15+i]);
        cube[6+i]  = cub[12+i];
        cube[9+i]  = 0.5*(cub[12+i]+cub[9+i]);
	cube[12+i] = 0.5*(cub[12+i]+cub[i]);
        cube[15+i] = 0.5*(cub[12+i]+cub[21+i]);
        cube[18+i] = 0.5*(cub[12+i]+cub[18+i]);
        cube[21+i] = 0.5*(cub[i]+cub[15+i]);
	cube[24+i] = cub[24+i];
      }
      ncode = code + 4*power[prof];
      pcode = Hilbert3d2(cube,ncode,prof,power);
      break;
    
    case 6:	// p is closer to f
      for (i=0; i<3; ++i) {
        cube[i]    = 0.5*(cub[15+i]+cub[i]);
	cube[3+i]  = 0.5*(cub[15+i]+cub[3+i]);
        cube[6+i]  = 0.5*(cub[15+i]+cub[6+i]);
        cube[9+i]  = 0.5*(cub[12+i]+cub[6+i]);
	cube[12+i] = 0.5*(cub[15+i]+cub[12+i]);
        cube[15+i] = cub[15+i];
        cube[18+i] = 0.5*(cub[15+i]+cub[18+i]);
        cube[21+i] = 0.5*(cub[12+i]+cub[18+i]);
	cube[24+i] = cub[24+i];
      }
      ncode = code + 5*power[prof];
      pcode = Hilbert3d2(cube,ncode,prof,power);
      break;
    
    case 7:	// p is closer to g
      for (i=0; i<3; ++i) {
        cube[i]    = 0.5*(cub[18+i]+cub[12+i]);
	cube[3+i]  = 0.5*(cub[18+i]+cub[15+i]);
        cube[6+i]  = 0.5*(cub[15+i]+cub[3+i]);
        cube[9+i]  = 0.5*(cub[15+i]+cub[i]);
	cube[12+i] = 0.5*(cub[18+i]+cub[i]);
        cube[15+i] = 0.5*(cub[18+i]+cub[3+i]);
        cube[18+i] = cub[18+i];
        cube[21+i] = 0.5*(cub[18+i]+cub[21+i]);
	cube[24+i] = cub[24+i];
      }
      ncode = code + 6*power[prof];
      pcode = Hilbert3d2(cube,ncode,prof,power);
      break;
    
    case 8:	// p is closer to h
      for (i=0; i<3; ++i) {
        cube[i]    = 0.5*(cub[21+i]+cub[18+i]);
	cube[3+i]  = 0.5*(cub[i]+cub[18+i]);
        cube[6+i]  = 0.5*(cub[i]+cub[15+i]);
        cube[9+i]  = 0.5*(cub[12+i]+cub[18+i]);
	cube[12+i] = 0.5*(cub[21+i]+cub[12+i]);
        cube[15+i] = 0.5*(cub[i]+cub[12+i]);
        cube[18+i] = 0.5*(cub[21+i]+cub[i]);
        cube[21+i] = cub[21+i];
	cube[24+i] = cub[24+i];
      }
      ncode = code + 7*power[prof];
      pcode = Hilbert3d2(cube,ncode,prof,power);
      break;
    }      

  }
  
  return pcode;

}










/* Classify in an increasing order depending on the rank array
  The idea of the heap approach is to build a flat binary tree */
int heapCrd(int siz, float *crd, int *vid) {
  int     i,j,r,l;
  float   frit;
  int     crit;

  if ( siz <= 1 )  return(0);
  
  l = siz / 2 + 1;
  r = siz;

  while ( r != 1 ) {
    if ( l > 1 ) {
      /* save state l */
      l       = l-1;
      frit = crd[l];
      crit = vid[l];
    }
    else {
      /* save state r and put 1 in r */
      frit   = crd[r];
      crit   = vid[r];
      crd[r] = crd[1];
      vid[r] = vid[1];
      r = r-1;
      if ( r == 1 ) 
        break;
    }
    j = l;
    i = j;
    j = 2*j;

    while ( j <= r ) {
      if ( j != r ) {   
        if ( crd[j] < crd[j+1] ) 
	  j = j+1;
      }
      if ( frit < crd[j] ) {
        /* put j in i (i is save) */
        crd[i] = crd[j];
        vid[i] = vid[j];
        i = j;
        j = 2*j;
      }
      else
        break;    
    }
    /* put save state in i */
    crd[i] = frit;
    vid[i] = crit;
    continue;
  }
  crd[1] = frit;
  vid[1] = crit;

  return(1);
}










