/*
 *  Uniform subdivision of triangles
 *
 *  Written by Pascal J. Frey, Inria-Rocquencourt
 *  Copyright (c) Inria, 2001.  All rights reserved. 
*/
#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"


int subdvd(pSurfMesh sm,int n) {  
  pTriangle    pt,pt1;
  pPoint       p0,p1,p2,pc;
  pHashtable   pht;
  double       u,v,w,u1;
  int          i,k,k1,k2,k3,l,nbele,nbv,nt,lvtx;
  int          ab1,ab2,ab3;
  int         *locvtx,*linei,*linei1,*t1,*t2,*t3,edg[3];
  ubyte        tag[3];

  if ( n < 1 )  return(1);
  E_put("subdvd");
  ++sm->mark;

  /* memory alloc */
  nbv  = n*(n+1) / 2 + n-2;
  locvtx = (int*)M_calloc(3*sm->ne*(3+nbv),sizeof(int),"subdvd");
  if ( !locvtx ) {
    prierr(ERR,0031);
    return(0);
  }
  linei  = (int*)M_malloc((n+3)*sizeof(int),"subdvd");
  linei1 = (int*)M_malloc((n+3)*sizeof(int),"subdvd");
  if ( !linei || !linei1) {
    prierr(ERR,0031);
    return(0);
  }

  /* reset hash table */
  for (k=1; k<nhmax; k++)
    hash[k].ind = hash[k].elt = 0;

  for (k=sm->np+1; k<=sm->npmax; k++)
    (&sm->point[k])->tag = M_UNUSED;

  /* triangle subdivision */
  nbele = sm->ne;
  lvtx  = 1;
  for (k=1; k<=nbele; k++) {
    pt = &sm->tria[k];
    if ( !pt->v[0] )  continue;

    /* keep tags */
    tag[0] = pt->tag[0];
    tag[1] = pt->tag[1];
    tag[2] = pt->tag[2];
    edg[0] = pt->edg[0];
    edg[1] = pt->edg[1];
    edg[2] = pt->edg[2];
	
    /* store edge vertices */
    k1 = k2 = k3 = 0;
    pht = hedge(pt->v[0],pt->v[1],&k1,&i);
    if ( !k1 ) {
      pht->elt = lvtx;
      t1 = &locvtx[lvtx];
      for (l=2; l<=n; l++)  t1[l] = ++sm->np;
      if ( pt->v[0] < pt->v[1] ) {
        t1[1]   = pt->v[0];
        t1[n+1] = pt->v[1];
        p0 = &sm->point[pt->v[0]];
        p1 = &sm->point[pt->v[1]];
      }
      else {
        t1[1]   = pt->v[1];
        t1[n+1] = pt->v[0];
        p0 = &sm->point[pt->v[1]];
        p1 = &sm->point[pt->v[0]];
      }
      for (l=2; l<=n; l++) {
        pc = &sm->point[t1[l]];
        u = (double)(l-1) / n;
        pc->c[0] = p0->c[0] + u * (p1->c[0]-p0->c[0]);
        pc->c[1] = p0->c[1] + u * (p1->c[1]-p0->c[1]);
        pc->c[2] = p0->c[2] + u * (p1->c[2]-p0->c[2]);
        pc->ref  = pt->edg[2];
        pc->tag  = pt->tag[2];
        pc->color = sm->mark;
        pc->size  = p0->size + u * (p1->size-p0->size);
      }
      lvtx += n+1;
    }
    else
      t1 = &locvtx[k1];

    pht = hedge(pt->v[0],pt->v[2],&k2,&i);
    if ( !k2 ) {
      pht->elt = lvtx;
      t2 = &locvtx[lvtx];
      for (l=2; l<=n; l++)  t2[l] = ++sm->np;
      if ( pt->v[0] < pt->v[2] ) {
        t2[1]   = pt->v[0];
        t2[n+1] = pt->v[2];
        p0 = &sm->point[pt->v[0]];
        p2 = &sm->point[pt->v[2]];
      }
      else {
        t2[1]   = pt->v[2];
        t2[n+1] = pt->v[0];
        p0 = &sm->point[pt->v[2]];
        p2 = &sm->point[pt->v[0]];
      }
      for (l=2; l<=n; l++) {
        pc = &sm->point[t2[l]];
        u = (double)(l-1) / n;
        pc->c[0] = p0->c[0] + u * (p2->c[0]-p0->c[0]);
        pc->c[1] = p0->c[1] + u * (p2->c[1]-p0->c[1]);
        pc->c[2] = p0->c[2] + u * (p2->c[2]-p0->c[2]);
        pc->ref  = pt->edg[1];
        pc->tag  = pt->tag[1];
        pc->color = sm->mark;
        pc->size  = p0->size + u * (p2->size-p0->size);
      }
      lvtx += n+1;
    }
    else
      t2 = &locvtx[k2];

    pht = hedge(pt->v[1],pt->v[2],&k3,&i);
    if ( !k3 ) {
      pht->elt = lvtx;
      t3 = &locvtx[lvtx];
      for (l=2; l<=n; l++)  t3[l] = ++sm->np;
      if ( pt->v[1] < pt->v[2] ) {
        t3[1]   = pt->v[1];
        t3[n+1] = pt->v[2];
        p1 = &sm->point[pt->v[1]];
        p2 = &sm->point[pt->v[2]];
      }
      else {
        t3[1]   = pt->v[2];
        t3[n+1] = pt->v[1];
        p1 = &sm->point[pt->v[2]];
        p2 = &sm->point[pt->v[1]];
      }
      for (l=2; l<=n; l++) {
        pc = &sm->point[t3[l]];
        u = (double)(l-1) / n;
        pc->c[0] = p1->c[0] + u * (p2->c[0]-p1->c[0]);
        pc->c[1] = p1->c[1] + u * (p2->c[1]-p1->c[1]);
        pc->c[2] = p1->c[2] + u * (p2->c[2]-p1->c[2]);
        pc->ref  = pt->edg[0];
        pc->tag  = pt->tag[0];
        pc->color = sm->mark;
        pc->size  = p1->size + u * (p2->size-p1->size);
      }
      lvtx += n+1;
    }
    else
      t3 = &locvtx[k3];

    p0 = &sm->point[pt->v[0]];
    p1 = &sm->point[pt->v[1]];
    p2 = &sm->point[pt->v[2]];
    
    ab1 = pt->v[0] == t1[1];
    ab2 = pt->v[0] == t2[1];
    ab3 = pt->v[1] == t3[1];

    /* scan line */
    linei[1] = pt->v[0];
    nt = 0;
    for (l=1; l<=n; l++) {
      linei1[1]   = ab1 ? t1[l+1] : t1[n+1-l];
      linei1[l+1] = ab2 ? t2[l+1] : t2[n+1-l];
      if ( l < n )
        for (i=2; i<=l; i++) linei1[i] = ++sm->np;
      else if ( ab3 ) 
        for (i=2; i<=n; i++) linei1[i] = t3[i];
      else
        for (i=2; i<=n; i++) linei1[i] = t3[n+2-i];

      /* barycentric */
      u1 = (double)l / n;
      u  = 1.0f - u1;
      for (i=1; i<=l; i++) {
        v = (double)(l-i+1) / n;
        w = (double)(i-1) / n;

        /* new point */
        pc = &sm->point[linei1[i]];
        if ( pc->tag == M_UNUSED ) {
          pc->c[0] = u*p0->c[0] + v*p1->c[0] + w*p2->c[0];
          pc->c[1] = u*p0->c[1] + v*p1->c[1] + w*p2->c[1];
          pc->c[2] = u*p0->c[2] + v*p1->c[2] + w*p2->c[2];
          pc->tag = M_NOTAG;
        }

        /* new triangle(s) */
        ++nt;
        pt1 = nt == 1 ? pt : &sm->tria[++sm->ne];
        pt1->v[0] = linei1[i];
        pt1->v[1] = linei1[i+1];
        pt1->v[2] = linei[i];
        pt1->ref  = pt->ref;
        
        pt1->tag[0] = M_NOTAG;
        pt1->tag[1] = M_NOTAG;
        pt1->tag[2] = M_NOTAG;
        pt1->edg[0] = 0;
        pt1->edg[1] = 0;
        pt1->edg[2] = 0;
        if ( i == 1 ) {
          pt1->tag[1] = tag[2];
          pt1->edg[1] = edg[2];
        }
        if ( i == l ) {
          pt1->tag[0] = tag[1];
          pt1->edg[0] = edg[1];
        }
        if ( l == n ) {
          pt1->tag[2] = tag[0];
          pt1->edg[2] = edg[0];
        }

        if ( i > 1 ) {
          nt++;
          pt1 = &sm->tria[++sm->ne];
          pt1->v[0] = linei1[i];
          pt1->v[1] = linei[i];
          pt1->v[2] = linei[i-1];
          pt1->ref  = pt->ref;
          pt1->tag[0] = pt1->tag[1] = pt1->tag[2] = M_NOTAG;
          pt1->edg[0] = pt1->edg[1] = pt1->edg[2] = 0;
        }
      }
      /*linei = linei1;*/
      memcpy(linei,linei1,(n+2)*sizeof(int));
    }

    if ( sm->ne > sm->nemax-2 ) {
      yerr.inderr[0] = sm->ne;
      yerr.inderr[1] = sm->nemax;
      prierr(ERR,4001);
      return(0);
    }
  }

  /* free memory */
  M_free(locvtx);
  M_free(linei);
  M_free(linei1);

  E_pop();
  return(sm->np-sm->npfixe);
}
