/*
 *  compute approximation of the principal curvatures
 *  and directions at a vertex (using a paraboloid)
 *
 *  Written by Pascal J. Frey, Inria-Rocquencourt
 *  Copyright (c) Inria, 1999-2001.  All rights reserved. 
*/

#ifdef __cplusplus
extern "C" {
#endif

#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"

#define  EPS3    1.e-15
#define  EPSZ    1.e-03


/* solve 3x3 definite symmetric positive */
int sol3x3(double ma[6],double mb[3],double mx[3]) {
  double  aa,bb,cc,dd,ee,ff,det;
  double  vmin,vmax,maxx;
  int     k;

  /* check ill-conditionned matrix */
  vmin = vmax = fabs(ma[0]);
  for (k=1; k<6; k++) {
    maxx = fabs(ma[k]);
    if ( maxx < vmin )  vmin = maxx;
    if ( maxx > vmax )  vmax = maxx;
  }
  if ( vmax == 0.0 )  return(0);
  else if ( vmin > EPS3 && vmin < vmax*EPS3 )  return(0);

  /* theory says: epsr = EPS3 * vmax;  */
  /*epsr = vmin == 0.0 ? EPS3 * vmax : vmax / vmin * EPS3;*/

  /* compute sub-dets */
  aa = ma[3]*ma[5] - ma[4]*ma[4];
  bb = ma[4]*ma[2] - ma[1]*ma[5];
  cc = ma[1]*ma[4] - ma[2]*ma[3];
  det = ma[0]*aa + ma[1]*bb + ma[2]*cc;
  if ( det == 0.0 )  return(0);
  det = 1.0f / det;

  mx[0] = (mb[0]*aa + mb[1]*bb + mb[2]*cc) * det;
  dd = ma[0]*ma[5] - ma[2]*ma[2];
  ee = ma[1]*ma[2] - ma[0]*ma[4];
  ff = ma[0]*ma[3] - ma[1]*ma[1];
  mx[1] = (mb[0]*bb + mb[1]*dd + mb[2]*ee) * det;
  mx[2] = (mb[0]*cc + mb[1]*ee + mb[2]*ff) * det;

  return(1);
}


/* local curvatures */
int radpoi_a(pSurfMesh sm) {
  pGeomSupp   geo;
  pMetric     pm1;
  pTriangle   pt,pt1,pta;
  pPoint      p1,p2;
  Ball        bb;
  double      gap,dd,ux,uy,uz,ac;
  double      x,y,z,x2,xy,y2,z2,maxd,sum,prd,alpha,zflat;
  double      ma[6],mb[3],mx[3],e1[2],e2[2];
  float      *np,b1[3],b2[3],k1,k2,kappa;
  float       v1[3],v2[3],c[3],nn[3],hmax2;
  int         i,k,kk,nk,l,a,b,i1,i2,nb;

  /* default */
  E_put("radpoi_a");
  if ( imprim < -4 )  primsg(1015);
  opts.kmin = FLT_MAX;
  opts.kmax = 0.0;

  /* set max size, if not specified */
  if ( opts.hmax < 0.0 )
    opts.hmax = 1.4 * fabs(info.delta);
  if ( opts.hmin > opts.hmax )
    opts.hmin = opts.hmax;
  hmax2 = 1.0 / (opts.hmax*opts.hmax);

  zflat = EPSZ * min(opts.bande,info.delta/1024.0);

  /* allocate memory */
  if ( !sm->metric && !zaldy3(sm,3) )  return(0);

  /* dim 2 */
  if ( sm->dim == 2 ) {
    if ( sm->nmfixe )
      for (k=1; k<=sm->np; k++) {
        p1  = &sm->point[k];
        pm1 = &sm->metric[k];
        pm1->m[5] = hmax2;
        pm1->m[2] = pm1->m[4] = 0.0;
        pm1->k1   = pm1->k2   = 0.0;
      }
    else
      for (k=1; k<=sm->np; k++) {
        p1  = &sm->point[k];
        pm1 = &sm->metric[k];
        pm1->m[0] = pm1->m[3] = pm1->m[5] = hmax2;
        pm1->m[1] = pm1->m[2] = pm1->m[4] = 0.0;
        pm1->k1   = pm1->k2 = 0.0;
        p1->size  = opts.hmax;
      }
    E_pop();
    return(1);
  }

  /* analyze triangle vertices */
  ++sm->mark;
  for (k=1; k<=sm->ne; k++) {
    pt = &sm->tria[k];
    if ( !pt->v[0] )  continue;

    for (i=0; i<3; i++) {
      i1 = idir[i+1];
      i2 = idir[i+2];
      a  = pt->v[i];
      p1 = &sm->point[a];
      if ( p1->size < 0.0 )  p1->size = opts.hmax;
  
      /* start along ridge */
      if ( p1->tag > M_NOTAG && pt->tag[i2] == M_NOTAG )
        continue;
      else if ( p1->color == sm->mark )
        continue;
      pm1 = &sm->metric[a];
      geo = &sm->geom[pt->vn[i]];
      np  =  sm->geom[pt->vn[i]].vn;

      /* referentiel (b1,b2,normale) */
      if ( fabs(np[0]) > EPS ) {
        b1[0] = -(np[1]+np[2]) / np[0];
        b1[1] = b1[2] = 1.0f;
      }
      else if ( fabs(np[1]) > EPS ) {
        b1[1] = -(np[0]+np[2]) / np[1];
        b1[0] = b1[2] = 1.0f;
      }
      else if ( fabs(np[2]) > EPS ) {
        b1[2] = -(np[0]+np[1]) / np[2];
        b1[0] = b1[1] = 1.0f;
      }
      else {
        b  = pt->v[i1];
        p2 = &sm->point[b];
        b1[0] = p2->c[0] - p1->c[0];
        b1[1] = p2->c[1] - p1->c[1];
        b1[2] = p2->c[2] - p1->c[2];
        dd = b1[0]*np[0] + b1[1]*np[1] + b1[2]*np[2];
        b1[0] = b1[0] - dd*np[0];
        b1[1] = b1[1] - dd*np[1];
        b1[2] = b1[2] - dd*np[2];
      }
      dd = b1[0]*b1[0] + b1[1]*b1[1] + b1[2]*b1[2];
      if ( dd == 0.0 ) continue;
      dd = 1.0f / sqrt(dd);
      b1[0] *= dd;
      b1[1] *= dd;
      b1[2] *= dd;

      b2[0] = np[1]*b1[2] - np[2]*b1[1];
      b2[1] = np[2]*b1[0] - np[0]*b1[2];
      b2[2] = np[0]*b1[1] - np[1]*b1[0];

      /* corner = hmax */
      if ( p1->tag & M_CORNER ) {
        setmet_a(p1,pm1,0.0,0.0,b1,b2);
        continue;
      }

      /* get ball of vertex a */
      bb.ilist = boulep(sm,k,i,&bb);
      if ( bb.ilist < 1 )  continue; 

      for (l=0; l<6; l++)  ma[l] = 0.0;
      for (l=0; l<3; l++)  mb[l] = mx[l] = 0.0;
      maxd = 0.0;
      nb   = 0;
      for (l=1; l<=bb.ilist; l++) {
        kk = bb.list[l];
        nk = bb.nump[l];
        i1 = idir[nk+1];
        pt1 = &sm->tria[kk];
        p2  = &sm->point[pt1->v[i1]];
        ux = p2->c[0] - p1->c[0];
        uy = p2->c[1] - p1->c[1];
        uz = p2->c[2] - p1->c[2];

        /* compute gap = angle between normals */
        gap = np[0]*pt1->n[0] + np[1]*pt1->n[1] + np[2]*pt1->n[2];
        if ( gap < geo->gap )  geo->gap = gap;

        /* coordinates of u in local frame */
        x = ux*b1[0] + uy*b1[1] + uz*b1[2];
        y = ux*b2[0] + uy*b2[1] + uz*b2[2];
        z = ux*np[0] + uy*np[1] + uz*np[2];
        x2 = x*x;
        xy = 2*x*y;
        y2 = y*y;
        z2 = z*z;
        ma[0] += x2*x2;
        ma[1] += x2*xy;
        ma[2] += x2*y2;
        ma[3] += 4*x2*y2;
        ma[4] += xy*y2;
        ma[5] += y2*y2;
        mb[0] += x2*z;
        mb[1] += xy*z;
        mb[2] += y2*z;
        if ( maxd < z2 )  maxd = z2;
        nb++;
        
        /* use mid-point */
        if ( coorpo(sm,kk,nk,c,0.5,nn) ) {
          ux = c[0] - p1->c[0];
          uy = c[1] - p1->c[1];
          uz = c[2] - p1->c[2];
          x  = ux*b1[0] + uy*b1[1] + uz*b1[2];
          y  = ux*b2[0] + uy*b2[1] + uz*b2[2];
          z  = ux*np[0] + uy*np[1] + uz*np[2];
          x2 = x*x;
          xy = 2*x*y;
          y2 = y*y;
          z2 = z*z;
          ma[0] += x2*x2;
          ma[1] += x2*xy;
          ma[2] += x2*y2;
          ma[3] += 4*x2*y2;
          ma[4] += xy*y2;
          ma[5] += y2*y2;
          mb[0] += x2*z;
          mb[1] += xy*z;
          mb[2] += y2*z;
          if ( maxd < z2 )  maxd = z2;
          nb++;
        }
        
        /* use point voyeur */
        if ( pt1->adj[nk] && pt1->tag[nk] == M_NOTAG ) {
          pta = &sm->tria[pt1->adj[nk]];
          dd  = pt1->n[0]*pta->n[0] + pt1->n[1]*pta->n[1]
              + pt1->n[2]*pta->n[2];
          if ( fabs(dd) > COS1DEG ) {
            i2 = pt1->voy[nk];
            p2 = &sm->point[pta->v[i2]];
            ux = p2->c[0] - p1->c[0];
            uy = p2->c[1] - p1->c[1];
            uz = p2->c[2] - p1->c[2];
            x  = ux*b1[0]  + uy*b1[1]  + uz*b1[2];
            y  = ux*b2[0]  + uy*b2[1]  + uz*b2[2];
            z  = ux*np[0] + uy*np[1] + uz*np[2];
            x2 = x*x;
            y2 = y*y;
            xy = 2.0*x*y;
            z2 = z*z;
            ma[0] += x2*x2;
            ma[1] += x2*xy;
            ma[2] += x2*y2;
            ma[3] += 4*x2*y2;
            ma[4] += xy*y2;
            ma[5] += y2*y2;
            mb[0] += x2*z;
            mb[1] += xy*z;
            mb[2] += y2*z;
            nb++;
            if ( maxd < z2 )  maxd = z2;
          }
        }
      }

      /* last point = last equation */
      if ( bb.closed )
        p1->color = sm->mark;
      else {
        kk = bb.list[bb.ilist];
        nk = bb.nump[bb.ilist];
        i2 = idir[nk+2];
        pt1 = &sm->tria[kk];
        p2  = &sm->point[pt1->v[i2]];
        ux = p2->c[0] - p1->c[0];
        uy = p2->c[1] - p1->c[1];
        uz = p2->c[2] - p1->c[2];

        /* coordinates of u in local frame */
        x = ux*b1[0] + uy*b1[1] + uz*b1[2];
        y = ux*b2[0] + uy*b2[1] + uz*b2[2];
        z = ux*np[0] + uy*np[1] + uz*np[2];
        x2 = x*x;
        xy = 2*x*y;
        y2 = y*y;
        z2 = z*z;
        ma[0] += x2*x2;
        ma[1] += x2*xy;
        ma[2] += x2*y2;
        ma[3] += 4*x2*y2;
        ma[4] += xy*y2;
        ma[5] += y2*y2;
        mb[0] += x2*z;
        mb[1] += xy*z;
        mb[2] += y2*z;
        if ( maxd < z2 )  maxd = z2;
        nb++;

        /* use mid-point */
        if ( coorpo(sm,kk,nk,c,0.5,nn) ) {
          ux = c[0] - p1->c[0];
          uy = c[1] - p1->c[1];
          uz = c[2] - p1->c[2];
          x  = ux*b1[0] + uy*b1[1] + uz*b1[2];
          y  = ux*b2[0] + uy*b2[1] + uz*b2[2];
          z  = ux*np[0] + uy*np[1] + uz*np[2];
          x2 = x*x;
          xy = 2*x*y;
          y2 = y*y;
          z2 = z*z;
          ma[0] += x2*x2;
          ma[1] += x2*xy;
          ma[2] += x2*y2;
          ma[3] += 4*x2*y2;
          ma[4] += xy*y2;
          ma[5] += y2*y2;
          mb[0] += x2*z;
          mb[1] += xy*z;
          mb[2] += y2*z;
          if ( maxd < z2 )  maxd = z2;
          nb++;
        }
      }

      /* solve linear system */
      if ( maxd < zflat || nb < 3 || !sol3x3(ma,mb,mx) ) {
        setmet_a(p1,pm1,0.0,0.0,b1,b2);
        continue;
      }
      p1->geom = M_CURVE;

      /* compute curvatures */
      sum = 2.0f*(mx[0]+mx[2]);
      prd = 4.0f*(mx[0]*mx[2]-mx[1]*mx[1]);
      dd  = sum*sum - 4.0*prd;
      if ( dd > 0.0 )  dd = sqrt(dd);
      k1 = 0.5 * (sum + dd);
      k2 = 0.5 * (sum - dd);
      kappa = max(fabs(k1),fabs(k2));
      if ( kappa < opts.kmin )  opts.kmin = kappa;
      if ( kappa > opts.kmax )  opts.kmax = kappa;

      /* compute curvatures and directions */
      if ( fabs(mx[1]) > EPSD ) {
        ac = mx[0]-mx[2];
        dd = ac*ac + mx[1]*mx[1];
        dd = sqrt(dd);
        alpha = (ac + dd) / mx[1];
        alpha = atan(-alpha);
        e2[0] = cos(alpha);   e2[1] = sin(alpha);
        e1[0] = -e2[1];       e1[1] = e2[0];

        /* compute principal directions in global frame */
        v1[0] = e1[0]*b1[0] + e1[1]*b2[0];
        v1[1] = e1[0]*b1[1] + e1[1]*b2[1];
        v1[2] = e1[0]*b1[2] + e1[1]*b2[2];
        v2[0] = e2[0]*b1[0] + e2[1]*b2[0];
        v2[1] = e2[0]*b1[1] + e2[1]*b2[1];
        v2[2] = e2[0]*b1[2] + e2[1]*b2[2];
        if ( !setmet_a(p1,pm1,k1,k2,v1,v2) )  return(0);
      }
      else if ( !setmet_a(p1,pm1,k1,k2,b1,b2) )  return(0);
    }
  }

  E_pop();
  return(1);
}


#ifdef __cplusplus
}
#endif
