/* 
 *  print mesh quality
 *
 *  Written by Pascal J. Frey
 *  email: Pascal.Frey@inria.fr, 1999
*/


#ifdef __cplusplus
extern "C" {
#endif

#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"

#define RINMIN  1.0e-08
#define MAXERR  5

/* print histogram of mesh quality */
int priqua(pSurfMesh sm) {
  pPoint      p0,p1,p2;
  pTriangle   pt;
  double      ux,uy,uz,delta,d1,d2,d3,cotmax,peri,surfa,rins;
  double      rap4,rapmin,rapmax,rapavg;
  int         k,nerr,rapnum,iel,ne;
  int         histo1[32],histo10[16];

  /* default values */
  yerr.lerror = FALSE;
  E_put("priqua");
  delta  = max(info.xmax-info.xmin,
           max(info.ymax-info.ymin,info.zmax-info.zmin));
  rapmin = (double)FLT_MAX;
  rapmax = (double)-FLT_MAX;
  rapavg = 0.0f; 
  rapnum = iel = ne = 0;
  nerr   = 0;

  for (k=0; k<=31; k++)  histo1[k]  = 0;
  for (k=0; k<=15; k++)  histo10[k] = 0.0f;

  for (k=1; k<=sm->ne; k++) {
    pt = &sm->tria[k];
    if ( !pt->v[0] )  continue;
    ne++;
    p0 = &sm->point[pt->v[0]];
    p1 = &sm->point[pt->v[1]];
    p2 = &sm->point[pt->v[2]];

    /* compute edge lengths */
    ux = p1->c[0] - p0->c[0];
    uy = p1->c[1] - p0->c[1];
    uz = p1->c[2] - p0->c[2];
    d1 = ux*ux + uy*uy + uz*uz;
    /*if ( d1 == 0.0f )  continue;*/
    d1 = sqrt(d1);
    
    ux = p2->c[0] - p0->c[0];
    uy = p2->c[1] - p0->c[1];
    uz = p2->c[2] - p0->c[2];
    d2 = ux*ux + uy*uy + uz*uz;
    /*if ( d2 == 0.0f )  continue;*/
    d2 = sqrt(d2);

    ux = p2->c[0] - p1->c[0];
    uy = p2->c[1] - p1->c[1];
    uz = p2->c[2] - p1->c[2];
    d3 = ux*ux + uy*uy + uz*uz;
    /*if ( d3 == 0.0f )  continue;*/
    d3 = sqrt(d3);

    /* triangle quality, Heron's formula */
    peri  = 0.5 * (d1 + d2 + d3);
    if ( peri == 0.0f )  continue;
    surfa = peri * (peri-d1) * (peri-d2) * (peri-d3);
    if ( surfa > 0.0f )
      surfa = sqrt(surfa);
    else
      surfa = 0.0f;
    if ( surfa != 0.0f && peri != 0.0f )
      rins  = surfa / peri;
    else
      rins = 0.0f;
    if ( rins > RINMIN ) {
      cotmax = d1;
      if (d2 > cotmax) cotmax = d2;
      if (d3 > cotmax) cotmax = d3;
      rap4 = max(ALPHAC * cotmax / rins,1.0);
      if ( rap4 < 17.e+10 ) {
        rapmin = min(rapmin,rap4);
        if ( rapmax <= rap4 ) {
          iel = k;
          rapmax = rap4;
        }
        rapnum++;
        rapavg += rap4;
        histo1[0]++;
        if ( rap4 < 10. )
          histo10[(int)rap4]++;
        else
          histo1[(int)log10(rap4)]++;
      }
      else {
        yerr.cooerr[0] = rap4;
        yerr.inderr[0] = k;
        nerr++;
        prierr(WAR,4007);
      }
    }
    else {
      yerr.cooerr[0] = rins;
      yerr.inderr[0] = k;
      prierr(WAR,4008);
      nerr++;
    }
  }

  if ( abs(imprim) > 1 && rapnum ) {
    if ( nerr > MAXERR ) 
      fprintf(stdout,"  -- TOTAL: %d ERRORS FOUND.\n",nerr);
    yerr.inderr[0] = sm->nefixe;
    pt = &sm->tria[iel];
    if ( rapnum == ne )
      fprintf(stdout,"\n  -- SURFACE MESH QUALITY   %d\n",rapnum);
    else
      fprintf(stdout,"\n  -- SURFACE MESH QUALITY   %d / %d\n",rapnum,ne);
    fprintf(stdout,"     AVERAGE QUALITY        %12.4f\n",rapavg / rapnum);
    fprintf(stdout,"     BEST  ELEMENT QUALITY  %12.4f\n",rapmin);
    if ( rapmax < 1.e7 )
      fprintf(stdout,"     WORST ELEMENT QUALITY  %12.4f\n",rapmax);
    else
      fprintf(stdout,"     WORST ELEMENT QUALITY  %12.4E\n",rapmax);
    fprintf(stdout,"     WORST ELEMENT   %d     %d %d %d\n",
	    iel,pt->v[0],pt->v[1],pt->v[2]);

    if ( abs(imprim) < 5 ) {
      E_pop();
      return(1);
    }
    if ((int)rapmin == (int)rapmax) {
      E_pop();
      return(1);
    }
    prihis(histo10,histo1,rapmin,rapmax);
  }
  
  E_pop();
  return(1);
}


#ifdef __cplusplus
}
#endif
