/*
 *  print mesh edge lengths
 *
 *  Written by Pascal J. Frey
 *  email: Pascal.Frey@inria.fr, 1999
*/

#ifdef __cplusplus
extern "C" {
#endif

#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"


int prilen(pSurfMesh sm) {
  pPoint     p0,p1;
  pTriangle  pt;
  pMetric    ma,mb;
  double     lmin,lmax,tail,ecart,lavg,som;
  double     dd,dd1,dd2,ux,uy,uz,rap;
  int        k,i,i1,i2,np,ne,navg,hl[10],iamin,iamax,ibmin,ibmax;
  static double bd[9] = {0., 0.2, 0.5, ISQRT2, 0.9, 1.111, SQRT2, 2., 5. };

  /* default values */
  E_put("prilen");
  lmin = (double)FLT_MAX;
  lmax = 0.0f;
  lavg = 0.0f; 
  som  = 0.0f;   
  navg = iamin = ibmin = iamax = ibmax = 0;

  for (k=1; k<10; k++)  hl[k] = 0;

  /* update vertex indices */
  np = ne = 0;
  for (k=1; k<=sm->np; k++) {
    p0 = &sm->point[k];
    if ( p0->tag & M_UNUSED ) continue;
    p0->tmp = ++np;
  }

  for (k=1; k<=sm->ne; k++) {
    pt  = &sm->tria[k];
    if ( !pt->v[0] )  continue;
    ne++;

    /* analyze triangle edges */
    for (i=0; i<3; i++) {
      if ( pt->adj[i] > k )  continue;
      navg++;
      i1 = idir[i+1];
      i2 = idir[i+2];
      p0 = &sm->point[pt->v[i1]];
      p1 = &sm->point[pt->v[i2]];

      /* compute edge length */
      ux = p1->c[0] - p0->c[0];
      uy = p1->c[1] - p0->c[1];
      uz = p1->c[2] - p0->c[2];
      dd = ux*ux + uy*uy + uz*uz;
      if ( opts.ctrl & ISO ) {
        rap = (p1->size - p0->size) / p0->size;
        if ( fabs(rap) < EPS )
          tail = sqrt(dd) * (2.-EPS) / (2.*p0->size);
        else {
          tail = 1.0/p0->size + 1.0/p1->size + 8.0/(p0->size+p1->size);
          tail = sqrt(dd) / 6.0 * tail;
        }
      }
      else {
        ma = &sm->metric[pt->v[i1]];
        mb = &sm->metric[pt->v[i2]];

        dd1 =      ma->m[0]*ux*ux + ma->m[3]*uy*uy + ma->m[5]*uz*uz \
            + 2.0*(ma->m[1]*ux*uy + ma->m[2]*ux*uz + ma->m[4]*uy*uz);
        if ( dd1 <= 0.0f )  dd1 = 0.0f;
        dd2 =      mb->m[0]*ux*ux + mb->m[3]*uy*uy + mb->m[5]*uz*uz \
            + 2.0*(mb->m[1]*ux*uy + mb->m[2]*ux*uz + mb->m[4]*uy*uz);
        if ( dd2 <= 0.0f )  dd2 = 0.0f;

        tail = (sqrt(dd1)+sqrt(dd2)+4.0*sqrt(0.5*(dd1+dd2))) / 6.0;
      }
      ecart = tail;
      lavg += tail;

      /* update efficiency index */
      if ( ecart > 1.0 )  ecart = 1.0 / ecart;
      som  += ecart - 1.0;

      /* find largest, smallest edge */
      if (tail < lmin) {
	lmin  = tail;
	iamin = pt->v[i1];
	ibmin = pt->v[i2];
      }
      else if (tail > lmax) {
	lmax  = tail;
        iamax = pt->v[i1];
        ibmax = pt->v[i2];
      }

      /* update histogram */
      if (tail < bd[3]) {
	if (tail > bd[2])       hl[3]++;
	else if (tail > bd[1])  hl[2]++;
	else                    hl[1]++;
      }
      else if (tail < bd[5]) {
	if (tail > bd[4])       hl[5]++;
	else if (tail > bd[3])  hl[4]++;
      }
      else if (tail < bd[6])    hl[6]++;
      else if (tail < bd[7])    hl[7]++;
      else if (tail < bd[8])    hl[8]++;
      else                      hl[9]++;
    }
  }

  /* print statistics */
  if ( abs(imprim) > 1  && navg ) {
    fprintf(stdout,"\n  -- RESULTING EDGE LENGTHS  %d\n",navg);
    fprintf(stdout,"     AVERAGE LENGTH         %12.4f\n",lavg / (float)navg);
    fprintf(stdout,"     SMALLEST EDGE LENGTH   %12.4f   %6d %6d\n",
            lmin,(&sm->point[iamin])->tmp,(&sm->point[ibmin])->tmp);
    fprintf(stdout,"     LARGEST  EDGE LENGTH   %12.4f   %6d %6d \n",
            lmax,(&sm->point[iamax])->tmp,(&sm->point[ibmax])->tmp);
    fprintf(stdout,"     EFFICIENCY INDEX       %12.4f\n",exp(som/(double)navg));
    if ( hl[4]+hl[5]+hl[6] )
      fprintf(stdout,"   %6.2f < L <%5.2f  %8d   %5.2f %%  \n",
	ISQRT2,SQRT2,hl[4]+hl[5]+hl[6],100.*(hl[4]+hl[5]+hl[6])/(double)navg);

    if ( floor(10.*lmin) == floor(10.*lmax) ) {
      E_pop();
      return(1);
    }
    if ( abs(imprim) < 5 ) {
      E_pop();
      return(1);
    }
    
    /* print histogram */
    fprintf(stdout,"\n     HISTOGRAMM\n");
    if ( hl[1] )
      fprintf(stdout,"     0.00 < L < 0.20  %8d   %5.2f %%  \n",
	      hl[1],100.*(hl[1]/(float)navg));
    if ( lmax > 0.2 ) {
      for (k=2; k<9; k++) {
        if ( hl[k] > 0 )
  	  fprintf(stdout,"   %6.2f < L <%5.2f  %8d   %5.2f %%  \n",
		  bd[k-1],bd[k],hl[k],100.*(hl[k]/(float)navg));
      }
      if ( hl[9] )
        fprintf(stdout,"     5.   < L         %8d   %5.2f %%  \n",
	        hl[9],100.*(hl[9]/(float)navg));
    }
  }

  E_pop();
  return(1);
}


#ifdef __cplusplus
}
#endif
