/*
 *  print mesh quality (after normalization back)
 *
 *  Written by Pascal J. Frey, Inria-Rocquencourt
 *  Copyright (c) Inria, 1999.  All rights reserved. 
*/

#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"

#define RINMIN   1.0e-08
#define MAXERR   5

int outqua(pSurfMesh sm) {
  pPoint      p0,p1,p2;
  pTriangle   pt;
  double      ux,uy,uz,delta,d1,d2,d3,cotmax,peri,surfa,rins;
  double      rap4,rapmin,rapmax,rapavg;
  float       qworst;
  int         k,rapnum,iel,ielne,ne,np,nbwors;
  int         histo1[32],histo10[16];
  int         nerr,iskwmax,skw[12];
  double      co,skew,aire,skwavg=0.,skewmin=10.,skewmax=0.;

  /* default values */
  E_put("outqua");
  delta  = max(info.xmax-info.xmin,
           max(info.ymax-info.ymin,info.zmax-info.zmin));
  rapmin = (double)FLT_MAX;
  rapmax = (double)-FLT_MAX;
  rapavg = 0.0f;
  rapnum = 0;
  qworst = ALFACOEFF / info.qworst;
  nbwors = iel = ielne = ne = iskwmax = nerr = 0;

  /* update vertex indices */
  np = 0;
  for (k=1; k<=sm->np; k++) {
    p0 = &sm->point[k];
    if ( p0->tag & M_UNUSED ) continue;
    p0->tmp = ++np;
  }

  /* print stats */
  for (k=0; k<=31; k++)  histo1[k] = 0;
  for (k=0; k<=15; k++)  histo10[k] = 0.0f;
  for (k=0; k<=11; k++)  skw[k] = 0.0f;

  for (k=1; k<=sm->ne; k++) {
    pt = &sm->tria[k];
    if ( !pt->v[0] )  continue;
    ne++;
    p0 = &sm->point[pt->v[0]];
    p1 = &sm->point[pt->v[1]];
    p2 = &sm->point[pt->v[2]];

    /* compute edge lengths */
    ux = p1->c[0] - p0->c[0];
    uy = p1->c[1] - p0->c[1];
    uz = p1->c[2] - p0->c[2];
    d1 = sqrt(ux*ux + uy*uy + uz*uz);

    ux = p2->c[0] - p0->c[0];
    uy = p2->c[1] - p0->c[1];
    uz = p2->c[2] - p0->c[2];
    d2 = sqrt(ux*ux + uy*uy + uz*uz);

    ux = p2->c[0] - p1->c[0];
    uy = p2->c[1] - p1->c[1];
    uz = p2->c[2] - p1->c[2];
    d3 = sqrt(ux*ux + uy*uy + uz*uz);

    /* triangle quality, Heron's formula */
    peri = 0.5 * (d1 + d2 + d3);
    if ( peri == 0.0f )  continue;
    surfa = peri * (peri-d1) * (peri-d2) * (peri-d3);
    if ( surfa > 0.0f )
      surfa = sqrt(surfa);
    else
      surfa = 0.0;
    if ( surfa != 0.0 && peri != 0.0 )
      rins  = surfa / peri;
    else
      rins = 0.0;
    if ( rins > RINMIN ) {
      cotmax = d1;
      if (d2 > cotmax) cotmax = d2;
      if (d3 > cotmax) cotmax = d3;
      rap4 = max(ALPHAC * cotmax / rins,1.0);
      if ( rap4 < 17.e+10 ) {
        rapmin = min(rapmin,rap4);
        if ( rapmax <= rap4 ) {
          iel    = k;
          rapmax = rap4;
          ielne  = ne;
        }
        rapnum++;
        rapavg += rap4;
        if ( rap4 > qworst )
          nbwors++;
        histo1[0]++;
        if ( rap4 < 10. )
          histo10[(int)rap4]++;
        else
          histo1[(int)log10(rap4)]++;

        /* compute skewness */
        co = 2.0*peri*(d2+d3-d1)*(d1+d3-d2)*(d1+d2-d3);
        co = d1*d2*d3 / sqrt(co);
        aire = co * co * 3. * SQRT3DIV4;
        skew = (aire - surfa) / aire;
        skwavg += skew;
        skw[(int)(10.*skew)]++;
        if ( skew < skewmin ) {
          skewmin = skew;
        }
        if ( skew > skewmax ) {
          iskwmax = k;
          skewmax = skew;
        }
      }
      else {
        nbwors++;
        yerr.cooerr[0] = rap4;
        yerr.inderr[0] = ne;
        nerr++;
        prierr(WAR,4007);
      }
    }
    else {
      nbwors++;
      yerr.cooerr[0] = (peri-d1) * (peri-d2) * (peri-d3);
      yerr.inderr[0] = ne;
      nerr++;
      prierr(WAR,4008);
    }
  }

  if ( abs(imprim) > 1 && rapnum ) {
    yerr.inderr[0] = sm->nefixe;
    /* print surface skewness */
    if ( abs(imprim) == 20 ) {
      pt = &sm->tria[iskwmax];
      fprintf(stdout,"\n  -- RESULTING TRIANGLE SKEWNESS   %d\n",rapnum);
      fprintf(stdout,"     AVERAGE SKEWNESS        %12.4f\n",skwavg / rapnum);
      fprintf(stdout,"     BEST SKEWNESS VALUE     %14.6f\n",skewmin);
      fprintf(stdout,"     WORST SKEWNESS VALUE    %14.6f\n",skewmax);
      fprintf(stdout,"     WORST ELEMENT   %d     %d %d %d\n",
	      iskwmax,(&sm->point[pt->v[0]])->tmp,(&sm->point[pt->v[1]])->tmp,
	      (&sm->point[pt->v[2]])->tmp);
      fprintf(stdout,"     0 < SKEW < 0.5    %8d   %5.2f %%\n",
	      skw[0]+skw[1]+skw[2]+skw[3]+skw[4],
              100.*(skw[0]+skw[1]+skw[2]+skw[3]+skw[4])/rapnum);
    }

    /* print surface quality */
    if ( nerr > MAXERR ) 
      fprintf(stdout,"  -- TOTAL: %d ERRORS FOUND.\n",nerr);
    pt = &sm->tria[iel];
    if ( rapnum == ne )
      fprintf(stdout,"\n  -- RESULTING SURFACE MESH QUALITY   %d\n",rapnum);
    else
      fprintf(stdout,"\n  -- RESULTING SURFACE MESH QUALITY   %d / %d\n",rapnum,ne);
    if ( rapavg / rapnum < 100.0f )
      fprintf(stdout,"     AVERAGE QUALITY        %12.4f\n",rapavg / rapnum);
    fprintf(stdout,"     BEST  ELEMENT QUALITY  %12.4f\n",rapmin);
    if ( rapmax < 1.e7 )
      fprintf(stdout,"     WORST ELEMENT QUALITY  %12.4f\n",rapmax);
    else
      fprintf(stdout,"     WORST ELEMENT QUALITY  %12.4E\n",rapmax);
    fprintf(stdout,"     WORST ELEMENT   %d     %d %d %d\n",
	    ielne,(&sm->point[pt->v[0]])->tmp,(&sm->point[pt->v[1]])->tmp,
	    (&sm->point[pt->v[2]])->tmp);

    if ( abs(imprim) < 5 ) {
      E_pop();
      return(1);
    }
    if ((int)rapmin == (int)rapmax) {
      E_pop();
      return(1);
    }
    prihis(histo10,histo1,rapmin,rapmax);

    if ( nbwors > 0 )
      fprintf(stdout,"\n  -- NUMBER OF ELEMENTS WORSE THAN TARGET VALUE  %8d\n",nbwors);
  }
  
  E_pop();
  return(1);
}

