/* 
 *  mesh file format writer/manipulator.
 *
 *  Written by Pascal J. Frey, 1998, 1999.
 *  email: Pascal.Frey@inria.fr
*/

#ifdef __cplusplus
extern "C" {
#endif

#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"

#define LCOEFF   4.0


/* flag = 0 : save metric at vertices
 * flag = 1 : save gaussian curvatures
 * flag = 2 : face coplanarity 
 * flag = 3 : vertex degree
 * flag = 4 : length at vertices 
 * flag = 5 : Hausdorff distance
 * flag = 6 : gap at vertices
 * flag = 7 : average edge length
*/
int outbbf(pSurfMesh sm,char *fileout,short flag) {
  FILE      *out;
  pPoint     ppt,p0,p1,p2;
  pTriangle  pt,pta;
  pMetric    pm,pm0,pm1;
  Ball       bb;
  double     dhd,tail,ux,uy,uz,vx,vy,vz,dd,ddx,ddy,ddz;
  int        i,k,adj,np,ne,nfield,typage,line;

  /* default */
  E_put("outbbf");
  np = ne = 0;
  if ( flag != 2 ) {
    for (k=1; k<=sm->np; k++) {
      ppt = & sm->point[ k ];
      if ( ppt->tag & M_UNUSED )  continue;
      ppt->tmp = ++np;
    }
  }
  else {
    for (k=1; k<=sm->ne; k++) {
      pt = &sm->tria[k];
      if ( pt->v[0] )  ne++;
    }
  }

  out = fopen(fileout,"w");
  if ( !out ) {
    yerr.coderr = 8001;
    return(0);
  }
  else if (imprim)
    fprintf(stdout,"  %%%% %s OPENED\n",fileout);

  /* write file format */
  switch (flag) {
  case 1: /* gaussian curvature */
    nfield = 1;
    typage = 2;
    fprintf(out,"%d ",sm->dim);
    fprintf(out,"%d %d %d\n",nfield,np,typage);
    line = 0;
    for (k=1; k<=sm->np; k++) {
      ppt = &sm->point[k];
      if ( ppt->tag & M_UNUSED )  continue;
      pm = &sm->metric[k];
      fprintf(out,"%f ",pm->k1*pm->k2);
      if ( ++line == 8 ) {
	fprintf(out,"\n");
	line = 0;
      }
    }
    fprintf(out,"\n");
    break;

  case 2: /* face coplanarity */
    nfield = 1;
    typage = 1;
    fprintf(out,"%d ",sm->dim);
    fprintf(out,"%d %d %d\n",nfield,ne,typage);
    line = 0;
    for (k=1; k<=sm->ne; k++) {
      pt = &sm->tria[k];
      if ( !pt->v[0] )	continue;
      dhd = 1.;
      for (i=0; i<3; i++) {
	adj = pt->adj[i];
	if ( !adj || pt->tag[i] & M_RIDGE_GEO )
	  continue;
	pta = &sm->tria[adj];
	dhd = min(dhd,fabs(pt->n[0]*pta->n[0]+pt->n[1]*pta->n[1]+pt->n[2]*pta->n[2]));
      }
      fprintf(out,"%f ",acos(dhd)*180./M_PI);
      if ( ++line == 8 ) {
	fprintf(out,"\n");
	line = 0;
      }
    }
    fprintf(out,"\n");
    break;

  case 3:  /* vertex degree */
    nfield = 1;
    typage = 2;
    fprintf(out,"%d ",sm->dim);
    fprintf(out,"%d %d %d\n",nfield,np,typage);
    line = 0;
    for (k=1; k<=sm->nemax; k++) {
      pt = &sm->tria[k];
      if ( !pt->v[0] )  continue;
      for (i=0; i<3; i++) {
        ppt = &sm->point[pt->v[i]];
        if ( ppt->tag & M_UNUSED )  continue;
        bb.ilist = 0;
        if ( ppt->tag == M_NOTAG ) {
          bb.ilist = boulep(sm,k,i,&bb);
          ppt->tag = M_UNUSED;
        }
        fprintf(out,"%d ",bb.ilist);  
        if ( ++line == 8 ) {
	  fprintf(out,"\n");
	  line = 0;
        }
      }
    }
    fprintf(out,"\n");
    break;

  case 4:  /* edge lengths at vertices */
    nfield = 1;
    typage = 2;
    fprintf(out,"%d ",sm->dim);
    fprintf(out,"%d %d %d\n",nfield,np,typage);
    line = 0;
    for (k=1; k<=sm->np; k++) {
      p0 = &sm->point[k];
      p0->size = 0.0f;
    }
    for (k=1; k<=sm->ne; k++) {
      pt = &sm->tria[k];
      if ( !pt->v[0] )  continue;
      for (i=0; i<3; i++) {
        p0 = &sm->point[pt->v[i]];
        p1 = &sm->point[pt->v[idir[i+1]]];
        if ( opts.ctrl & ISO )
          tail = lenseg(p0,p1);
        else {
          pm0 = &sm->metric[pt->v[i]];
          pm1 = &sm->metric[pt->v[idir[i+1]]];
          tail = lenseg_a(p0,p1,pm0,pm1);
        }
        if ( tail > p0->size ) p0->size = tail;
      }
    }
    for (k=1; k<=sm->np; k++) {
      p0 = &sm->point[k];
      if ( p0->tag & M_UNUSED )  continue;
      fprintf(out,"%f ",p0->size);
      if ( ++line == 8 ) {
        fprintf(out,"\n");
        line = 0;
      }
    }
    fprintf(out,"\n");
    break;

  case 5:  /* Hausdorff distance */
    nfield = 1;
    typage = 1;
    fprintf(out,"%d ",sm->dim);
    fprintf(out,"%d %d %d\n",nfield,sm->ne,typage);
    line = 0;
    for (k=1; k<=sm->ne; k++) {
      pt = &sm->tria[k];
      if ( pt->v[0] ) {
        fprintf(out,"%f ",pt->dish);
        if ( ++line == 8 ) {
          fprintf(out,"\n");
          line = 0;
        }
      }
    }
    fprintf(out,"\n");
    break;

  case 6:  /* local gap at vertices */
    nfield = 1;
    typage = 2;
    fprintf(out,"%d ",sm->dim);
    fprintf(out,"%d %d %d\n",nfield,sm->np,typage);
    line = 0;
    for (k=1; k<=sm->ne; k++) {
      pt = &sm->tria[k];
      if ( !pt->v[0] )  continue;
      for (i=0; i<3; i++) {
        p1 = &sm->point[pt->v[i]];
        p1->size = 1.0 - sm->geom[pt->vn[i]].gap;
      }
    }
    for (k=1; k<=sm->np; k++) {
      p1 = &sm->point[k];
      if ( p1->tag & M_UNUSED )  continue;
      fprintf(out,"%f ",p1->size);
      if ( ++line == 8 ) {
        fprintf(out,"\n");
        line = 0;
      }
    }
    fprintf(out,"\n");
    break;

  case 7:  /* average edge length */
    nfield = 1;
    typage = 2;
    fprintf(out,"%d ",sm->dim);
    fprintf(out,"%d %d %d\n",nfield,np,typage);
    line = 0;
    for (k=1; k<=sm->np; k++) {
      p0 = &sm->point[k];
      p0->size  = 0.0f;
      p0->color = 0;
    }
    for (k=1; k<=sm->ne; k++) {
      pt = &sm->tria[k];
      if ( !pt->v[0] )  continue;
      p0 = &sm->point[pt->v[0]];
      p1 = &sm->point[pt->v[1]];
      p2 = &sm->point[pt->v[2]];
      
      ux = p1->c[0] - p0->c[0];
      uy = p1->c[1] - p0->c[1];
      uz = p1->c[2] - p0->c[2];
      
      vx = p2->c[0] - p0->c[0];
      vy = p2->c[1] - p0->c[1];
      vz = p2->c[2] - p0->c[2];
      
      ddx = uy*vz - uz*vy;
      ddy = uz*vx - ux*vz;
      ddz = ux*vy - uy*vx;
      dd  = sqrt(ddx*ddx + ddy*ddy + ddz*ddz);
      dd  = LCOEFF*sqrt(dd);
      
      p0->size += dd;
      p0->color++;
      p1->size += dd;
      p1->color++;
      p2->size += dd;
      p2->color++;
    }
    for (k=1; k<=sm->np; k++) {
      p0 = &sm->point[k];
      if ( p0->tag & M_UNUSED )  continue;
      fprintf(out,"%f ",p0->size /(double)p0->color);
      if ( ++line == 8 ) {
        fprintf(out,"\n");
        line = 0;
      }
    }
    fprintf(out,"\n");
    break;

  default: /* metric at vertices */
    typage = 2;
    if ( opts.ctrl & ISO ) {
      nfield = 1;
      fprintf(out,"%d ",sm->dim);
      fprintf(out,"%d %d %d\n",nfield,np,typage);
      line = 0;
      for (k=1; k<=sm->np; k++) {
        ppt = &sm->point[k];
        if ( ppt->tag & M_UNUSED )  continue;
        fprintf(out,"%f ",ppt->size);
        if ( ++line == 8 ) {
	  fprintf(out,"\n");
	  line = 0;
        }
      }
      fprintf(out,"\n");
    }
    else {
      nfield = sm->dim*(sm->dim+1)/2;
      fprintf(out,"%d ",sm->dim);
      fprintf(out,"%d %d %d\n",nfield,np,typage);
      if ( sm->dim == 2 ) {
        for (k=1; k<=sm->np; k++) {
          ppt = &sm->point[k];
          if ( ppt->tag & M_UNUSED )  continue;
          pm  = &sm->metric[k];
          fprintf(out,"%f %f %f\n",pm->m[0],pm->m[1],pm->m[3]);
        }
      }
      else {
        for (k=1; k<=sm->np; k++) {
          ppt = &sm->point[k];
          if ( ppt->tag & M_UNUSED )  continue;
          pm  = &sm->metric[k];
          fprintf(out,"%f %f %f %f %f %f\n",
                  pm->m[0],pm->m[1],pm->m[3],pm->m[2],pm->m[4],pm->m[5]);
        }
      }
      fprintf(out,"\n");
    }
  }

  fclose(out);
  E_pop();
  return(1);
}


#ifdef __cplusplus
}
#endif
