/*
 *  construct metric
 *
 *  Written by Pascal J. Frey, Inria-Rocquencourt
 *  Copyright (c) Inria, 2000-2003.  All rights reserved. 
*/

#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"

#define EPS3    1.e-18
#define EPS6    5.e-06


/* invert 3x3 symmetric matrix */
int invmat(double m[6],double mi[6]) {
  double  aa,bb,cc,det,vmin,vmax,maxx;
  int     k;

  /* check diagonal matrices */
  vmax = fabs(m[1]);
  maxx = fabs(m[2]);
  if( maxx > vmax ) vmax = maxx;
  maxx = fabs(m[4]);
  if( maxx > vmax ) vmax = maxx;
  if ( vmax < EPS ) {
    mi[0]  = 1./m[0];
    mi[3]  = 1./m[3];
    mi[5]  = 1./m[5];
    mi[1] = mi[2] = mi[4] = 0.0;
    return(1);
  }

  /* check ill-conditionned matrix */
  vmin = vmax = fabs(m[0]);
  for (k=1; k<6; k++) {
    maxx = fabs(m[k]);
    if ( maxx < vmin )  vmin = maxx;
    else if ( maxx > vmax )  vmax = maxx;
  }
  if ( vmax == 0.0 )  return(0);

  /* compute sub-dets */
  aa  = m[3]*m[5] - m[4]*m[4];
  bb  = m[4]*m[2] - m[1]*m[5];
  cc  = m[1]*m[4] - m[2]*m[3];
  det = m[0]*aa + m[1]*bb + m[2]*cc;
  if ( fabs(det) < EPS3 )  return(0);
  det = 1.0f / det;

  mi[0] = aa*det;
  mi[1] = bb*det;
  mi[2] = cc*det;
  mi[3] = (m[0]*m[5] - m[2]*m[2])*det;
  mi[4] = (m[1]*m[2] - m[0]*m[4])*det;
  mi[5] = (m[0]*m[3] - m[1]*m[1])*det;

  return(1);
}


/* invert 3x3 non-symmetric matrix */
int invmatg(double m[9],double mi[9]) {
  double  aa,bb,cc,det,vmin,vmax,maxx;
  int     k;

  /* check ill-conditionned matrix */
  vmin = vmax = fabs(m[0]);
  for (k=1; k<9; k++) {
    maxx = fabs(m[k]);
    if ( maxx < vmin )  vmin = maxx;
    else if ( maxx > vmax )  vmax = maxx;
  }
  if ( vmax == 0.0 )  return(0);

  /* compute sub-dets */
  aa = m[4]*m[8] - m[5]*m[7];
  bb = m[5]*m[6] - m[3]*m[8];
  cc = m[3]*m[7] - m[4]*m[6];
  det = m[0]*aa + m[1]*bb + m[2]*cc;
  if ( fabs(det) < EPS3 )  return(0);
  det = 1.0f / det;

  mi[0] = aa*det;
  mi[3] = bb*det;
  mi[6] = cc*det;
  mi[1] = (m[2]*m[7] - m[1]*m[8])*det;
  mi[4] = (m[0]*m[8] - m[2]*m[6])*det;
  mi[7] = (m[1]*m[6] - m[0]*m[7])*det;
  mi[2] = (m[1]*m[5] - m[2]*m[4])*det;
  mi[5] = (m[2]*m[3] - m[0]*m[5])*det;
  mi[8] = (m[0]*m[4] - m[1]*m[3])*det;

  return(1);
}


/* simultaneous reduction */
int simred_a(double m1[6],double m2[6],double m[6]) {
  double  dd,lambda[3],v[3][3],hh[3];
  double  maxd1,maxd2,ex,ey,ez,m1i[6],n[9],p[9],pi[9];
  int     i,j,k,order;

  maxd1 = fabs(m1[1]);
  maxd1 = max(maxd1,fabs(m1[2]));
  maxd1 = max(maxd1,fabs(m1[4]));
  
  maxd2 = fabs(m2[1]);
  maxd2 = max(maxd2,fabs(m2[2]));
  maxd2 = max(maxd2,fabs(m2[4]));

  /* check diag matrices */
  if ( maxd1 < EPS && maxd2 < EPS ) {
    m[0] = max(m1[0],m2[0]);
    m[3] = max(m1[3],m2[3]);
    m[5] = max(m1[5],m2[5]);
    m[1] = m[2] = m[4] = 0.0;
    return(1);
  }
  if ( !invmat(m1,m1i) )  return(0);

  /* n = (m1)^-1*m2 */
  n[0] = m1i[0]*m2[0] + m1i[1]*m2[1] + m1i[2]*m2[2];
  n[1] = m1i[0]*m2[1] + m1i[1]*m2[3] + m1i[2]*m2[4];
  n[2] = m1i[0]*m2[2] + m1i[1]*m2[4] + m1i[2]*m2[5];
  n[3] = m1i[1]*m2[0] + m1i[3]*m2[1] + m1i[4]*m2[2];
  n[4] = m1i[1]*m2[1] + m1i[3]*m2[3] + m1i[4]*m2[4];
  n[5] = m1i[1]*m2[2] + m1i[3]*m2[4] + m1i[4]*m2[5];
  n[6] = m1i[2]*m2[0] + m1i[4]*m2[1] + m1i[5]*m2[2];
  n[7] = m1i[2]*m2[1] + m1i[4]*m2[3] + m1i[5]*m2[4];
  n[8] = m1i[2]*m2[2] + m1i[4]*m2[4] + m1i[5]*m2[5];

  /* eigenvectors */
  order = eigenv(0,n,lambda,v);

  if ( order == 0 ) 
    return(0);
  else if ( order == 3 ) {
    m[0] = m[3] = m[5] = lambda[0];
    m[1] = m[2] = m[4] = 0.0;
    return(1);
  }
  else {
    /* matrix of passage */
    for (i=0,k=0; i<3; i++)
      for (j=0; j<3; j++)
        p[k++] = v[j][i];
    if ( !invmatg(p,pi) )  return(0);

    for (i=0; i<3; i++) {
      ex = v[i][0];
      ey = v[i][1];
      ez = v[i][2];
      maxd1 = ex*(m1[0]*ex+m1[1]*ey+m1[2]*ez) 
            + ey*(m1[1]*ex+m1[3]*ey+m1[4]*ez)
            + ez*(m1[2]*ex+m1[4]*ey+m1[5]*ez);
      maxd2 = ex*(m2[0]*ex+m2[1]*ey+m2[2]*ez) 
            + ey*(m2[1]*ex+m2[3]*ey+m2[4]*ez)
            + ez*(m2[2]*ex+m2[4]*ey+m2[5]*ez);
      hh[i] = max(maxd1,maxd2);
    }

    /* compose matrix tP^-1*lambda*P^-1 */
    m[0] = pi[0]*hh[0]*pi[0] + pi[3]*hh[1]*pi[3] + pi[6]*hh[2]*pi[6];
    m[1] = pi[0]*hh[0]*pi[1] + pi[3]*hh[1]*pi[4] + pi[6]*hh[2]*pi[7];
    m[2] = pi[0]*hh[0]*pi[2] + pi[3]*hh[1]*pi[5] + pi[6]*hh[2]*pi[8];
    m[3] = pi[1]*hh[0]*pi[1] + pi[4]*hh[1]*pi[4] + pi[7]*hh[2]*pi[7];
    m[4] = pi[1]*hh[0]*pi[2] + pi[4]*hh[1]*pi[5] + pi[7]*hh[2]*pi[8];
    m[5] = pi[2]*hh[0]*pi[2] + pi[5]*hh[1]*pi[5] + pi[8]*hh[2]*pi[8];

    /* for safety... */
    maxd1 = fabs(m[0]);
    for (i=1; i<6; i++)
      maxd1 = max(maxd1,fabs(m[i]));
    if ( maxd1 < EPSD ) {
      fprintf(stderr,"  ## ERR 9200, simred_a, null matrix.\n");
      exit(1);
    }
    dd = 1.0 / maxd1;
    for (i=0; i<6; i++) {
      m[i] *= dd;
      /*if ( fabs(m[i]) < EPS )  m[i] = 0.0;*/
    }

    maxd2 = fabs(m[1]);
    maxd2 = max(maxd2,fabs(m[2]));
    maxd2 = max(maxd2,fabs(m[4]));

    if ( maxd2 < EPS6 ) {
      m[1] = m[2] = m[4] = 0.0;
      m[0] = fabs(m[0]) * maxd1;
      m[3] = fabs(m[3]) * maxd1;
      m[5] = fabs(m[5]) * maxd1;
    }
    else {
      order = eigenv(1,m,lambda,v);
      for (i=0; i<6; i++)   m[i] *= maxd1;

      if ( lambda[0] < -EPS || lambda[1] < -EPS || lambda[2] < -EPS ) {
        fprintf(stderr,"  ## ERR 9201, simred_a, not a metric !\n");
        fprintf(stderr,"  %.6f %.6f %.6f\n%.6f %.6f\n%.6f\n",
                       m[0],m[1],m[2],m[3],m[4],m[5]);
        fprintf(stderr,"  Lambda %f %f %f\n",lambda[0],lambda[1],lambda[2]);
	    return(0);
      }
    }
    
    return(1);
  }
}

/* build matrix associated w/ metric */
int setmet_a(pPoint p1,pMetric pm,float kk1,float kk2,float *e1,float *e2) {
  double   h1,h2,h3,keps,reps,rerr,lambda[3];
  double   m[6],m1[6],m2[6];
  float    k1,k2,tail,e3[3];
  int      i,j,k;

  /* compute sizes */
  k1 = fabs(kk1);
  k2 = fabs(kk2);
  h1 = opts.hmax;
  h2 = opts.hmax;
  h3 = opts.hmax;
  if ( k1 < EPS && k2 < EPS )
    h1 = h2 = opts.hmax;
  else if ( k1 > EPS && k2 < EPS )
    h2 = opts.hmax;
  keps = 2.0 * (1.-opts.eps) * opts.alpha;
  reps = 2.0 * sqrt(2.0*opts.eps) / opts.hmax;
  rerr = 1.0 - ISQRT2;

  if ( k1 > EPS ) {
    if ( opts.bande*k1 < rerr )
      h1 = min(h1,2.0*(1.0-k1*opts.bande)
  	     * sqrt(opts.bande*(2.0-k1*opts.bande)/k1));
  }
  if ( k2 > EPS ) {
    if ( opts.bande*k2 < rerr )
      h2 = min(h2,2.0*(1.0-k2*opts.bande)
  	     * sqrt(opts.bande*(2.0-k2*opts.bande)/k2));
  }

  if ( opts.eps > 0.0 ) {
    if ( k1 > reps )  h1 = max(h1,keps / k1);
    if ( k2 > reps )  h2 = max(h2,keps / k2);
  }

  /* bound sizes */
  h1 = max(h1,opts.hmin);
  h2 = max(h2,opts.hmin);
	
  lambda[0] = 1.0 / (h1*h1);
  lambda[1] = 1.0 / (h2*h2);
  lambda[2] = 1.0 / (h3*h3);

  /* build matrix */
  e3[0] = e1[1]*e2[2] - e1[2]*e2[1];
  e3[1] = e1[2]*e2[0] - e1[0]*e2[2];
  e3[2] = e1[0]*e2[1] - e1[1]*e2[0];

  if ( pm->k1 < 0.0 ) {
    /* set coeffs directly */
    for (k=0,i=0; i<3; i++)
      for (j=i; j<3; j++)
        pm->m[k++] = lambda[0]*e1[i]*e1[j] + lambda[1]*e2[i]*e2[j] \
                   + lambda[2]*e3[i]*e3[j];
  }
  else {
    /* metric intersection */
    for (i=0; i<6; i++)  m1[i] = pm->m[i];
    for (k=0,i=0; i<3; i++)
      for (j=i; j<3; j++)
        m2[k++] = lambda[0]*e1[i]*e1[j] + lambda[1]*e2[i]*e2[j] \
                + lambda[2]*e3[i]*e3[j];

    if ( simred_a(m1,m2,m) )
      for (i=0; i<6; i++)  pm->m[i] = m[i];
    else {
      for (i=0; i<6; i++)
        pm->m[i] = SQRT3DIV2 * (pm->m[i] + m2[i]);
    }
  }
  pm->k1 = kk1;
  pm->k2 = kk2;
  tail   = min(h1,h2);
  if ( tail < p1->size )  p1->size = max(opts.hmin,tail);

  return(1);
}


/* build matrix associated with edge */
int edgmet_a(pPoint p1,pMetric pm,float h) {
  double   lambda[3],m1[6],m2[6],m[6];
  float    e1[3],e2[3],e3[3];
  int      i,j,k;

  e1[0] = 1.0; e1[1] = 0.0;  e1[2] = 0.0;
  e2[0] = 0.0; e2[1] = 1.0;  e2[2] = 0.0;
  e3[0] = 0.0; e3[1] = 0.0;  e3[2] = 1.0;

  lambda[0] = lambda[1] = 1.0f / (h*h);
  lambda[2] = 1.0f / (opts.hmax*opts.hmax);

  /* build matrix */
  if ( pm->k1 < 0.0 )
    for (k=0,i=0; i<3; i++)
      for (j=i; j<3; j++)
        pm->m[k++] = lambda[0]*e1[i]*e1[j] + lambda[1]*e2[i]*e2[j] \
                   + lambda[2]*e3[i]*e3[j];
  else {
    /* metric intersection */
    for (i=0; i<6; i++)  m1[i] = pm->m[i];
    for (k=0,i=0; i<3; i++)
      for (j=i; j<3; j++)
        m2[k++] = lambda[0]*e1[i]*e1[j] + lambda[1]*e2[i]*e2[j] \
                + lambda[2]*e3[i]*e3[j];

    if ( simred_a(m1,m2,m) ) 
      for (i=0; i<6; i++)  pm->m[i] = m[i];
    else {
      for (i=0; i<6; i++)
        pm->m[i] = SQRT3DIV2 * (pm->m[i]+m2[i]);
    }
  }
  pm->k1 = h;
  pm->k2 = h;

  return(1);
}

/* invert 3x3 symmetric matrix, return m1[6] */
int invmat2(double m[6],double m11[6]) {
  double t,tmp[3][3],m1[9];
  int    i,j,k,sw;

  for (i=0; i<3; i++)
    for (j=0; j<3; j++)
      m1[i*3+j] = i==j ? 1.0 : 0.0;
  for (k=0,i=0; i<3; i++)
    for (j=i; j<3; j++)
      tmp[i][j] = tmp[j][i] = m[k++];

  /* largest elt in col */
  for (i=0; i<3; i++) {
    sw = i;
    for (j=i+1; j<3; j++)
      if ( fabs(tmp[j][i]) > fabs(tmp[i][i]) )  sw = j;
    /* swap rows */
    if ( sw != i ) {
      for (k=0; k<3; k++) {
        t = tmp[i][k];
        tmp[i][k]  = tmp[sw][k];
        tmp[sw][k] = t;
        t = m1[i*3+k];
        m1[i*3+k]  = m1[sw*3+k];
        m1[sw*3+k] = t;
      }
    }
    /* singular mat */
    if ( egal(tmp[i][i],0.0) )  return(0);
    t = tmp[i][i];
    for (k=0; k<3; k++) {
      tmp[i][k] /= t;
      m1[i*3+k] /= t;
    }
    for (j=0; j<3; j++)
      if ( j != i ) {
        t = tmp[j][i];
        for (k=0; k<3; k++) {
          tmp[j][k] -= tmp[i][k]*t;
          m1[j*3+k] -= m1[i*3+k]*t;
        }
      }
  }

  for (k=0,i=0; i<3; i++)
    for (j=i; j<3; j++)
      m11[k++] = m1[i*3+j];

  return(1);
}


/* interpolate linearly metric */
int intme2_b(pMetric pm0,pMetric pm1,pMetric pm,double t) {
  double  l0[3],l1[3],lm[3],v0[3][3],v1[3][3],v[3][3];
  double  t1,m0[6],m1[6],m[6];
  int     i,j,k;

  for (i=0; i<6; i++) {
    m0[i] = pm0->m[i];
    m1[i] = pm1->m[i];
  }
  if ( !eigenv(1,m0,l0,v0) )  return(0);
  if ( !eigenv(1,m1,l1,v1) )  return(0);

  t1 = 1.0 - t;
  for (i=0; i<3; i++) {
    l0[i] = t1 / sqrt(l0[i]);
    l1[i] = t  / sqrt(l1[i]);
  }
  for (k=0,i=0; i<3; i++)
    for (j=i; j<3; j++) {
      m[k]  = l0[0]*v0[0][i]*v0[0][j] + l0[1]*v0[1][i]*v0[1][j] \
            + l0[2]*v0[2][i]*v0[2][j];
      m[k] += l1[0]*v1[0][i]*v1[0][j] + l1[1]*v1[1][i]*v1[1][j] \
            + l1[2]*v1[2][i]*v1[2][j];
      k++;
    }

  if ( !eigenv(1,m,lm,v) )  return(0);

  for (i=0; i<3; i++)
    lm[i] = 1.0 / (lm[i]*lm[i]);

  for (k=0,i=0; i<3; i++)
    for (j=i; j<3; j++)
      pm->m[k++]  = lm[0]*v[0][i]*v[0][j] + lm[1]*v[1][i]*v[1][j] \
                  + lm[2]*v[2][i]*v[2][j];

  return(1);
}


int intme2_a(pMetric pm0,pMetric pm1,pMetric pm,double t) {
  double     t1,m[6],mint[6],m0[6],m0i[6],m1[6],m1i[6];
  int        i;

  for (i=0; i<6; i++) {
    m0[i] = pm0->m[i];
    m1[i] = pm1->m[i];
  }
  if ( !invmat(m0,m0i) || !invmat(m1,m1i) )  return(0);

  t1 = 1.0 - t;
  for (i=0; i<6; i++)
    mint[i] = t1*m0i[i] + t*m1i[i];
  if ( !invmat(mint,m) )  return(0);

  for (i=0; i<6; i++)  pm->m[i] = m[i];

  return(1);
}


/* linear interpolation in triangle */
int intme3_a(pMetric pm0,pMetric pm1,pMetric pm2,pMetric pm,double *cb) {
  double     mi[6],m0[6],m0i[6],m1[6],m1i[6],m2[6],m2i[6];
  int        i;

  for (i=0; i<6; i++) {
    m0[i] = pm0->m[i];
    m1[i] = pm1->m[i];
    m2[i] = pm2->m[i];
  }
  if ( !invmat(m0,m0i) || !invmat(m1,m1i) || !invmat(m2,m2i) )  return(0);

  for (i=0; i<6; i++)
    mi[i] = cb[0]*m0i[i] + cb[1]*m1i[i] + cb[2]*m2i[i];
  if ( !invmat(mi,m1) )  return(0);
  for (i=0; i<6; i++)  pm->m[i] = m1[i];
 
  return(1);
}

