#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "libmesh3.h"
#include "yams.h"
#include "extern.h"
#include "sproto.h"


#define NMAX   10000
extern ubyte   ecp;


int loadMesh(pSurfMesh sm,char *filein,int memory,int choix) {
  LM_mesh_struct  ms;
  pPoint     ppt;
  pTriangle  pt1,pt2;
  pTetra     ptt;
  pEdge      pte;
  pPoint     p0,p1,p2,p3;
  pGeomSupp  g0;
  double     dd,dd1,dd2;
  float     *tabf,n1[3],n2[3],q1,q2;
  int       *tabi,i,k,iadr,nbl,disc,is,nn,nis,v[4];
  int        ncorner,prequis,nridge,nrequis,trequis,nvn,nbt,ref;
  char      *ptr,data[128];

  E_put("loadMesh");
  strcpy(data,filein);
  ptr = strstr(data,".mesh");
  if ( !ptr ) {
    strcat(data,".meshb");
    if( !LM_open_mesh(data, LM_READ, &ms) ) {
      ptr = strstr(data,".mesh");
      *ptr = '\0';
      strcat(data,".mesh");
      if( !LM_open_mesh(data, LM_READ, &ms) ) {
        fprintf(stderr,"  ** %s  NOT FOUND.\n",data);
        return(0);
      }
    }
  }
  else if( !LM_open_mesh(data, LM_READ, &ms) ) {
    fprintf(stderr,"  ** %s  NOT FOUND.\n",data);
    return(0);
  }

  if ( imprim )
    fprintf(stdout,"  %%%% %s OPENED\n",data);

  /* parse keywords */
  sm->dim     = ms.dimension;
  sm->npfixe  = ms.kw_counters[ LM_Vertices ];
  sm->nefixe  = ms.kw_counters[ LM_Triangles ];
  nbt = sm->nefixe;
  sm->nefixe += 2 * ms.kw_counters[ LM_Quadrilaterals ];
  sm->nafixe  = ms.kw_counters[ LM_Edges ];
  sm->nvfixe  = ms.kw_counters[ LM_Normals ];
  sm->ntfixe  = ms.kw_counters[ LM_Tangents ];
  sm->ntet    = ms.kw_counters[ LM_Tetrahedra ];

  /* check space dimension */
  if ( sm->dim < 2 || sm->dim > 3 ) {
    yerr.coderr = 0001;
    LM_close_mesh(&ms);
    return(0);
  }

  /* check if vertices and elements found */
  if ( !sm->npfixe || !sm->nefixe ) {
    yerr.coderr = 0001;
    LM_close_mesh(&ms);
    return(0);
  }

  /* mesh normals */
  if ( !sm->nvfixe ) {
    if ( sm->type & M_DETECT )
      sm->nvfixe = 1.50*sm->npfixe;
    else
      sm->nvfixe = 1.05*sm->npfixe;
  }

  /* memory allocation for mesh structure */
  if ( !zaldy1(sm->nefixe,sm->npfixe,sm->nvfixe,memory,sm,choix) ) {
    LM_close_mesh(&ms);
    return(0);
  }
  disc = 0;

  /* read mesh vertices */
  tabi = (int*)malloc(NMAX * 5 * sizeof(int));
  assert(tabi);
  tabf = (float*)tabi;

  sm->npfixe = 0;
  while ( nbl = LM_read_field(&ms, LM_Vertices, NMAX, tabi) ) {
    for (k=0; k<nbl; k++) {
      ++sm->npfixe;
      ppt = &sm->point[sm->npfixe];
      iadr      = k * (sm->dim+1);
      ppt->c[0] = tabf[ iadr + 0 ];
      ppt->c[1] = tabf[ iadr + 1 ];
      if ( sm->dim == 2 ) {
        ppt->c[2] = 0.0;
        ppt->ref  = tabi[ iadr + 2 ] & 0x7fff;
      }
      else {
        ppt->c[2] = tabf[ iadr + 2 ];
        ppt->ref  = tabi[ iadr + 3 ] & 0x7fff;
      }
      ppt->tag  = M_UNUSED;
      ppt->color= 0;
      ppt->size = -1.;
      ppt->tge  = 0;
      ppt->geom = M_CURVE;

      /* find extrema coordinates */
      if ( ppt->c[0] < info.xmin ) info.xmin = ppt->c[0];
      if ( ppt->c[0] > info.xmax ) info.xmax = ppt->c[0];
      if ( ppt->c[1] < info.ymin ) info.ymin = ppt->c[1];
      if ( ppt->c[1] > info.ymax ) info.ymax = ppt->c[1];
      if ( ppt->c[2] < info.zmin ) info.zmin = ppt->c[2];
      if ( ppt->c[2] > info.zmax ) info.zmax = ppt->c[2];
    }
  }

  /* read mesh triangles */
  sm->nefixe = 0;
  while ( nbl = LM_read_field(&ms, LM_Triangles, NMAX, tabi) ) {
    for (k=0; k<nbl; k++) {
      ++sm->nefixe;
      pt1 = &sm->tria[sm->nefixe];
      iadr      = k * 4;
      pt1->v[0] = tabi[ iadr + 0 ];
      pt1->v[1] = tabi[ iadr + 1 ];
      pt1->v[2] = tabi[ iadr + 2 ];
      pt1->ref  = tabi[ iadr + 3 ] & 0x7fff;
      if ( opts.noreff ) pt1->ref = 0;

      if ( pt1->v[0] < 1 || pt1->v[0] > sm->npfixe ||
           pt1->v[1] < 1 || pt1->v[1] > sm->npfixe ||
           pt1->v[2] < 1 || pt1->v[2] > sm->npfixe ) {
        yerr.inderr[0] = k;
        prierr(WAR,0020);
        disc++;
        pt1->v[0] = 0;
      }
    }
  }

  /* read mesh quadrilaterals */
  while ( nbl = LM_read_field(&ms, LM_Quadrilaterals, NMAX, tabi) ) {
    for (k=0; k<nbl; k++) {
      iadr = k * 5;
      v[0] = tabi[ iadr + 0 ];
      v[1] = tabi[ iadr + 1 ];
      v[2] = tabi[ iadr + 2 ];
      v[3] = tabi[ iadr + 3 ];
      ref  = tabi[ iadr + 4 ] & 0x7fff;
      if ( opts.noreff )  ref = 0;

      if ( v[0] < 1 || v[0] > sm->npfixe ||
           v[1] < 1 || v[1] > sm->npfixe ||
           v[2] < 1 || v[2] > sm->npfixe ||
           v[3] < 1 || v[3] > sm->npfixe ) {
        yerr.inderr[0] = k;
        prierr(WAR,0020);
        disc++;
      }

      p0 = &sm->point[ v[0] ];
      p1 = &sm->point[ v[1] ];
      p2 = &sm->point[ v[2] ];
      p3 = &sm->point[ v[3] ];
      qualfa(p0->c,p1->c,p2->c,&q1,n1);
      qualfa(p0->c,p2->c,p3->c,&q2,n2);
      dd1 = n1[0]*n2[0] + n1[1]*n2[1] + n1[2]*n2[2];

      qualfa(p0->c,p1->c,p3->c,&q1,n1);
      qualfa(p1->c,p2->c,p3->c,&q2,n2);
      dd2 = n1[0]*n2[0] + n1[1]*n2[1] + n1[2]*n2[2];

      if ( dd1 > dd2 ) {
        pt1 = &sm->tria[++sm->nefixe];
        pt1->v[0] = v[0];
        pt1->v[1] = v[1];
        pt1->v[2] = v[2];
        pt1->ref  = ref;
        
        pt1 = &sm->tria[++sm->nefixe];
        pt1->v[0] = v[0];
        pt1->v[1] = v[2];
        pt1->v[2] = v[3];
        pt1->ref  = ref;
      }
      else {
        pt1 = &sm->tria[++sm->nefixe];
        pt1->v[0] = v[0];
        pt1->v[1] = v[1];
        pt1->v[2] = v[3];
        pt1->ref  = ref;
        
        pt1 = &sm->tria[++sm->nefixe];
        pt1->v[0] = v[1];
        pt1->v[1] = v[2];
        pt1->v[2] = v[3];
        pt1->ref  = ref;
      }
    }
  }

  /* tetrahedra */
  if ( sm->ntet ) {
    sm->tetra = (Tetra*)M_malloc((sm->ntet+1)*sizeof(Tetra),"loadMesh");
    assert(sm->tetra);
    sm->ntet = 0;
    while ( nbl = LM_read_field(&ms, LM_Tetrahedra, NMAX, tabi) ) {
      for (k=0; k<nbl; k++) {
        ++sm->ntet;
        ptt = &sm->tetra[sm->ntet];
        iadr      = k * 5;
    for (i=0; i<4; i++)
          ptt->v[i] = tabi[iadr + i];
        ptt->ref  = tabi[iadr + 4] & 0x7fff;
        if ( opts.noreff )  ptt->ref = 0;
      }
    }
  }

  /* mark used vertices */
  for (k=1; k<=sm->nefixe; k++) {
    pt1 = &sm->tria[k];
    if ( pt1->v[0] )
      for (i=0; i<3; i++) {
        ppt = &sm->point[pt1->v[i]];
        ppt->tag &= ~M_UNUSED;
      }
  }

  /* count unused vertices */
  for (k=1; k<=sm->npfixe; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  info.nulp++;
  }

  /* read mesh corners */
  ncorner = 0;
  while ( nbl = LM_read_field(&ms, LM_Corners, NMAX, tabi) ) {
    for (k=0; k<nbl; k++) {
      is = tabi[ k ];
      if ( is < 0 || is > sm->npfixe )
        disc++;
      else {
        ++ncorner;
    ppt = &sm->point[ is ];
    ppt->tag |=  M_CORNER;
        ppt->tag &= ~M_UNUSED;
      }
    }
  }

  /* read required vertices */
  prequis = 0;
  while ( nbl = LM_read_field(&ms, LM_RequiredVertices, NMAX, tabi) ) {
    for (k=0; k<nbl; k++) {
      is = tabi[ k ];
      if ( is < 0 || is > sm->npfixe )
        disc++;
      else {
        ++prequis;
        ppt = &sm->point[is];
        ppt->tag |=  M_REQUIRED;
        ppt->tag &= ~M_UNUSED;
      }
    }
  }

  /* read mesh edges */
  if ( sm->nafixe ) {
    sm->edge = (pEdge)M_calloc(sm->nafixe+1,sizeof(Edge),"inmesh");
    if ( sm->edge == NULL ) {
      prierr(WAR,0021);
      LM_close_mesh(&ms);
      E_pop();
      return(1);
    }

    sm->nafixe = 0;
    while ( nbl = LM_read_field(&ms, LM_Edges, NMAX, tabi) ) {
      for (k=0; k<nbl; k++) {
        sm->nafixe++;
        pte   = &sm->edge[ sm->nafixe ];
        iadr  = k * 3;
        pte->p1  = tabi[ iadr + 0 ];
        pte->p2  = tabi[ iadr + 1 ];
        pte->ref = tabi[ iadr + 2 ] & 0x7fff;
        if ( opts.noreff )  pte->ref = 0;
        pte->tag = M_NOTAG;

        if ( pte->p1 < 1 || pte->p2 > sm->npfixe || pte->p2 < 1 || pte->p2 > sm->npfixe ) {
          disc++;
          sm->nafixe--;
          continue;
        }
        else {
          ppt = &sm->point[ pte->p1 ];
          ppt->tag &= ~M_UNUSED;
          ppt = &sm->point[ pte->p2 ];
          ppt->tag &= ~M_UNUSED;
        }

        /* vertex coloring */
        if ( pte->ref ) {
          ppt = &sm->point[ pte->p1 ];
          ppt->tag |= M_RIDGE_REF;
          if ( !ppt->color )
            ppt->color = pte->ref;
          else if ( ppt->color != pte->ref )
            ppt->tag |= M_RIDGE_REF;
          if ( !ppt->ref )  ppt->ref = pte->ref;
      
          ppt = &sm->point[ pte->p2 ];
          ppt->tag |= M_RIDGE_REF;
          if ( !ppt->color )
            ppt->color = pte->ref;     
          else if ( ppt->color != pte->ref )
            ppt->tag |= M_RIDGE_REF;
          if ( !ppt->ref )  ppt->ref = pte->ref;
        }
      }
    }

    /* ridges */
    nridge = 0;
    while ( nbl = LM_read_field(&ms, LM_Ridges, NMAX, tabi) ) {
      for (k=0; k<nbl; k++) {
        nridge++;
        iadr  = tabi[k];
        pte   = &sm->edge[iadr];
        pte->tag |= M_RIDGE_GEO;
        ppt = &sm->point[pte->p1];
        ppt->tag |= M_RIDGE_GEO;
        ppt = &sm->point[pte->p2];
        ppt->tag |= M_RIDGE_GEO;
      }
    }
    
    /* required edges */
    nrequis = 0;
    while ( nbl = LM_read_field(&ms, LM_RequiredEdges, NMAX, tabi) ) {
      for (k=0; k<nbl; k++) {
        nrequis++;
        iadr  = tabi[k];
        pte   = &sm->edge[iadr];
        pte->tag |= M_REQUIRED;
        ppt = &sm->point[pte->p1];
        ppt->tag |= M_REQUIRED;
        ppt = &sm->point[pte->p2];
        ppt->tag |= M_REQUIRED;
      }
    }
  }

  /* read required triangles */
  trequis = 0;
  while ( nbl = LM_read_field(&ms, LM_RequiredTriangles, NMAX, tabi) ) {
    for (k=0; k<nbl; k++) {
      is = tabi[ k ];
      if ( is < 0 || is > sm->nefixe )
        disc++;
      else {
        ++trequis;
        pt1 = &sm->tria[is];
        for (i=0; i<3; i++) {
          pt1->tag[i] |=  M_REQUIRED;
        }
      }
    }
  }


  /* normals */
  sm->nvfixe = 0;
  while ( nbl = LM_read_field(&ms, LM_Normals, NMAX, tabi) ) {
    for (k=0; k<nbl; k++) {
      ++sm->nvfixe;
      iadr = k*3;
      g0 = &sm->geom[ sm->nvfixe ];
      g0->vn[0] = tabf[ iadr+0 ];
      g0->vn[1] = tabf[ iadr+1 ];
      g0->vn[2] = tabf[ iadr+2 ];
      g0->gap = 1.0;
      dd = g0->vn[0]*g0->vn[0] + g0->vn[1]*g0->vn[1] + g0->vn[2]*g0->vn[2];
      if ( dd > 0.0 ) {
        dd = 1.0 / sqrt(dd);
        g0->vn[0] *= dd;
        g0->vn[1] *= dd;
        g0->vn[2] *= dd;
      }
      else 
        info.nuln++;
    }
  }

  if ( sm->nvfixe ) {
    /* normal at vertices */
    nvn = 0;
    while ( nbl = LM_read_field(&ms, LM_NormalAtVertices, NMAX, tabi) ) {
      for (k=0; k<nbl; k++) {
        nvn++;
        iadr = k*2;
        nn  = tabi[iadr+0];
        is  = tabi[iadr+1];
        if ( nn < 1 || nn > sm->npfixe ) {
          yerr.inderr[0] = nn;
          yerr.inderr[1] = -1;
          prierr(WAR,0010);
          continue;
        }
        else if ( is <= 0 || is > sm->nvfixe ) {
          yerr.inderr[0] = nn;
          yerr.inderr[1] = is;
          prierr(WAR,0010);
          continue;
        }
        ppt = &sm->point[nn];
        ppt->tmp = is;
      }
    }
    /* update normals */
    for (k=1; k<=sm->nefixe; k++) {
      pt1 = &sm->tria[k];
      if ( !pt1->v[0] )  continue;
      for (i=0; i<3; i++) {
        ppt = &sm->point[pt1->v[i]];
        if ( !pt1->vn[i] && ppt->tmp )
          pt1->vn[i] = ppt->tmp;
      }
    }
    
    /* normals at triangle vertices */
    nvn = 0;
    while ( nbl = LM_read_field(&ms, LM_NormalAtTriangleVertices, NMAX, tabi) ) {
      for (k=0; k<nbl; k++) {
        nvn++;
        iadr = k*3;
        nn  = tabi[iadr+0];
        is  = tabi[iadr+1];
        nis = tabi[iadr+2];
        if ( nn < 1 || nn > sm->nefixe ) {
          yerr.inderr[0] = nn;
          yerr.inderr[1] = -1;
          prierr(WAR,0010);
          continue;
        }
        else if ( is < 1 || is > 3 ) {
          yerr.inderr[0] = nn;
          yerr.inderr[1] = is;
          prierr(WAR,0011);
          continue;
        }
        else if ( nis < 1 || nis > sm->nvfixe ) {
          pt1 = &sm->tria[nn];
          pt1->vn[is-1]    = 0;
          yerr.inderr[0] = nn;
          yerr.inderr[1] = nis;
          prierr(WAR,0011);
          continue;
        }
        pt1 = &sm->tria[nn];
        pt1->vn[is-1] = nis;
      }
    }
    
    /* normals at quad vertices */
    nvn = 0;
    while ( nbl = LM_read_field(&ms, LM_NormalAtQuadrilateralVertices, NMAX, tabi) ) {
      for (k=0; k<nbl; k++) {
        nvn++;
        iadr = k*3;
        nn  = tabi[iadr+0];
        is  = tabi[iadr+1];
        nis = tabi[iadr+2];
        if ( nn <= 0 || nn > sm->nefixe ) {
          yerr.inderr[0] = nn;
          yerr.inderr[1] = -1;
          prierr(WAR,0010);
          continue;
        }
        else if ( is < 1 || is > 4 ) {
          yerr.inderr[0] = nn;
          yerr.inderr[1] = is;
          prierr(WAR,0011);
          continue;
        }
        else if ( nis < 1 || nis > sm->nv ) {
          pt1 = &sm->tria[nn];
          pt1->vn[is-1]  = 0;
          yerr.inderr[0] = nn;
          yerr.inderr[1] = nis;
          prierr(WAR,0011);
          continue;
        }
        nn  = nbt + 2 * (nn-1);
        pt1 = &sm->tria[nn];
        pt2 = &sm->tria[nn+1];
    is = is - 1;
        if ( is == 0 )
          pt1->vn[is] = pt1->vn[is] = nis;
        else if ( is == 1 )
          pt1->vn[is] = nis;
        else if ( is == 2 ) 
          pt1->vn[is] = pt2->vn[1] = nis;
        else
          pt2->vn[2] = nis;   
      }
    }
  }

  LM_close_mesh(&ms);
  free(tabi);

  if ( disc > 0 && imprim ) {
    yerr.inderr[0] = disc;
    prierr(WAR,0022);
  }

  E_pop();
  return(1);
}


/* save mesh to disk */
int saveMesh(pSurfMesh sm,char *fileout) {
  LM_mesh_struct   ms;
  pGeomSupp    gs;
  pGeomtge     gt;
  pPoint       ppt;
  pTriangle    pt1;
  pTetra       ptt;
  pEdge        pte;
  float       *tabf;
  int         *tabi,i,k,iadr,np,ne,nn,nt,nav,natv,tatv,nbl;
  int          nedge,nridge,ndang,nrequis;
  int          is1,is2,ncorner,prequis;
  char        *ptr,data[128];

  E_put("saveMesh");
  strcpy(data,fileout);
  ptr = strstr(data,".mesh");
  if ( !ptr ) {
    strcat(data,ecp ? ".mesh" : ".mesh");
    if( !LM_open_mesh(data, LM_WRITE, &ms, sm->dim) ) {
      ptr  = strstr(data,".mesh");
      *ptr = '\0';
      strcat(data,".mesh");
      if( !LM_open_mesh(data, LM_WRITE, &ms, sm->dim) ) {
        fprintf(stderr,"  ** UNABLE TO OPEN %s.\n",data);
        return(0);
      }
    }
  }
  else if( !LM_open_mesh(data, LM_WRITE, &ms, sm->dim) ) {
    fprintf(stderr,"  ** UNABLE TO OPEN %s.\n",data);
    return(0);
  }
  if ( imprim )
    fprintf(stdout,"  %%%% %s OPENED\n",data);

  /* mark connected component */
  ne = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = & sm->point[k];
    ppt->tag |= M_UNUSED;
    ppt->flag = ppt->color = 0;
  }

  if ( sm->connex > 0 ) {
    for (k=1; k<=sm->ne; k++) {
      pt1 = &sm->tria[k];
      if ( pt1->v[0] > 0 && pt1->cc == sm->connex ) {
        ne++;
        for (i=0; i<3; i++) {
          ppt = &sm->point[pt1->v[i]];
          ppt->tag &= ~M_UNUSED;
        }
      }
    }
  }
  else {
    /* mark used faces */
    for (k=1; k<=sm->ne; k++) {
      pt1 = &sm->tria[k];
      if ( !pt1->v[0] )  continue;
      ++ne;
      for (i=0; i<3; i++) {
        ppt = &sm->point[pt1->v[i]];
        ppt->tag &= ~M_UNUSED;
      }
    }
  }

  if ( sm->ntet ) {
    for (k=1; k<=sm->ntet; k++) {
      ptt = &sm->tetra[k];
      if ( !ptt->v[0] )  continue;
      for (i=0; i<4; i++) {
        ppt = &sm->point[ptt->v[i]];
        ppt->tag &= ~M_UNUSED;
      }
    }
  }

  /* mark used vertices */
  np = nav = 0;
  ncorner = prequis = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  continue;
    ppt->tmp = ++np;
    if ( ppt->tag == M_NOTAG )  nav++;
  }

  tabi = (int*)malloc(NMAX * 5 * sizeof(int));
  assert(tabi);
  tabf = (float*)tabi;

  nbl = 0;
  for(k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  continue;
    iadr = nbl * (sm->dim+1);
    tabf[ iadr + 0 ] = ppt->c[0];
    tabf[ iadr + 1 ] = ppt->c[1];
    if ( sm->dim == 3 ) {
      tabf[ iadr + 2 ] = ppt->c[2];
      tabi[ iadr + 3 ] = ppt->ref;
    }
    else
      tabi[ iadr + 2 ] = ppt->ref;
    ++nbl;
    if (ppt->tag & M_CORNER)    ncorner++;
    if (ppt->tag & M_REQUIRED ) prequis++;
    
    if ( nbl == NMAX ) {
      LM_write_field(&ms, LM_Vertices, nbl, tabf);
      nbl = 0;
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Vertices, nbl, tabf);

  /* write triangles */
  nedge  = sm->dim == 3 ? info.ndang : 0;
  nridge = nrequis = nn = nt = natv = tatv = 0;
  nbl    = 0;
  for (k=1; k<=sm->ne; k++) {
    pt1  = &sm->tria[k];
    if ( !pt1->v[0] )  continue;
    else if ( sm->connex > 0 && pt1->cc != sm->connex ) continue;
    iadr = nbl * 4;
    tabi[ iadr + 0 ] = sm->point[pt1->v[0]].tmp;
    tabi[ iadr + 1 ] = sm->point[pt1->v[1]].tmp;
    tabi[ iadr + 2 ] = sm->point[pt1->v[2]].tmp;
    tabi[ iadr + 3 ] = (int)(sm->connex < 0 ? pt1->cc : pt1->ref);
    ++nbl;
    for (i=0; i<3; i++) {
      ppt = &sm->point[pt1->v[i]];
      gs  = &sm->geom[pt1->vn[i]];
      gt  = &sm->tgte[ppt->tge];
      if ( ppt->tag > M_NOTAG ) {
        natv++;
        if ( ppt->tag & M_CORNER )  tatv++;
      }
      if ( !gs->new )  gs->new = ++nn;
      if ( !gt->new )  gt->new = ++nt;
      if ( !pt1->edg[i] && pt1->tag[i] == M_NOTAG )  continue;
      else if ( pt1->adj[i] && (k > pt1->adj[i]) )   continue;
      nedge++;
      if ( pt1->tag[i] & M_RIDGE_GEO )  nridge++;
      if ( pt1->tag[i] & M_REQUIRED )   nrequis++;
    }
    
    if ( nbl == NMAX ) {
      LM_write_field(&ms, LM_Triangles, nbl, tabi);
      nt += nbl;
      nbl = 0;
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Triangles, nbl, tabi);
  nt += nbl;

  /* tetrahedra */
  if ( opts.choix == 6 && sm->ntet ) {
    nbl    = 0;
    for (k=1; k<=sm->ntet; k++) {
      ptt  = &sm->tetra[k];
      if ( !ptt->v[0] )  continue;
      iadr = nbl * 5;
      for (i=0; i<4; i++)
        tabi[iadr + i] = sm->point[ptt->v[i]].tmp;
      tabi[iadr + 4] = ptt->ref;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Tetrahedra, nbl, tabi);
        nbl = 0;
      }
    }
    if ( nbl )  LM_write_field(&ms, LM_Tetrahedra, nbl, tabi);
    M_free(sm->tetra);
  }

  /* corners */
  ncorner = nbl = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  continue;
    else if ( ppt->tag & M_CORNER ) {
      tabi[nbl] = ppt->tmp;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Corners, nbl, tabi);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Corners, nbl, tabi);

  /* required vertices */
  prequis = nbl = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  continue;
    else if ( ppt->tag & M_REQUIRED ) {
      tabi[nbl] = ppt->tmp;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_RequiredVertices, nbl, tabi);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_RequiredVertices, nbl, tabi);

  /* edges */
  nedge = nbl = 0;
  for (k=1; k<=sm->ne; k++) {
    pt1 = &sm->tria[k];
    if ( !pt1->v[0] )  continue;
    else if ( sm->connex > 0 && pt1->cc != sm->connex ) continue;
    for (i=0; i<3; i++) {
      if ( !pt1->edg[i] && pt1->tag[i] == M_NOTAG ) continue;
      else if ( pt1->adj[i] && (k > pt1->adj[i]) )  continue;
      is1  = idir[i+1];
      is2  = idir[i+2];
      iadr = nbl * 3;
      tabi[ iadr+0 ] = sm->point[pt1->v[is1]].tmp;
      tabi[ iadr+1 ] = sm->point[pt1->v[is2]].tmp;
      tabi[ iadr+2 ] = pt1->edg[i];
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Edges, nbl, tabi);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Edges, nbl, tabi);

  /* dangling edges */
  if ( info.ndang > 0 ) {
    ndang = nbl = 0;
    for (k=1; k<=info.ndang; k++) {
      pte = &sm->edge[k];
      iadr = nbl * 3;
      if ( pte->p1 * pte->p2 ) {
        tabi[ iadr+0 ] = pte->p1;   
        tabi[ iadr+0 ] = pte->p2;   
        tabi[ iadr+0 ] = pte->ref;
        ++nbl;
        if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_Edges, nbl, tabi);
          nbl = 0;
        }
      }
    }
    if ( nbl )  LM_write_field(&ms, LM_Edges, nbl, tabi);
  }

  /* ridges */
  nedge = nbl = 0;
  for (k=1; k<=sm->ne; k++) {
    pt1 = &sm->tria[k];
    if ( !pt1->v[0] ) continue;
    else if ( sm->connex > 0 && pt1->cc != sm->connex ) continue;
    for (i=0; i<3; i++) {
      if ( !pt1->edg[i] && pt1->tag[i] == M_NOTAG ) continue;
      else if ( pt1->adj[i] && (k > pt1->adj[i]) )  continue;
      ++nedge;
      if ( (pt1->tag[i] & M_RIDGE_GEO) ) {
        tabi[nbl] = nedge;
        nbl++;
        if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_Ridges, nbl, tabi);
          nbl = 0;
        }
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Ridges, nbl, tabi);

  /* required edges */
  nedge = nbl = 0;
  for (k=1; k<=sm->ne; k++) {
    pt1 = &sm->tria[k];
    if ( !pt1->v[0] ) continue;
    else if ( sm->connex > 0 && pt1->cc != sm->connex ) continue;
    for (i=0; i<3; i++) {
      if ( !pt1->edg[i] && pt1->tag[i] == M_NOTAG ) continue;
      else if ( pt1->adj[i] && (k > pt1->adj[i]) )  continue;
      ++nedge;
      if ( pt1->tag[i] & M_REQUIRED ) {
        tabi[nbl] = nedge;
        nbl++;
        if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_RequiredEdges, nbl, tabi);
          nbl = 0;
        }
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_RequiredEdges, nbl, tabi);

  if ( imprim ) {
    fprintf(stdout,"     NUMBER OF GIVEN VERTICES    %8d\n",sm->npfixe);
    fprintf(stdout,"     NUMBER OF GIVEN TRIANGLES   %8d\n",sm->nefixe);
    if ( sm->ntet)
      fprintf(stdout,"     NUMBER OF GIVEN TETRAHEDRA  %8d\n",sm->nefixe);
    fprintf(stdout,"     TOTAL NUMBER OF VERTICES    %8d\n",np);
    if ( ne < sm->nefixe )
      fprintf(stdout,"     TOTAL NUMBER OF TRIANGLES   %8d   (%.2f %%)\n",
          ne,100*(float)ne/(float)sm->nefixe);
    else
      fprintf(stdout,"     TOTAL NUMBER OF TRIANGLES   %8d\n",ne);
    if ( info.cc > 1 )
      fprintf(stdout,"     NUMBER OF SUB-DOMAINS       %8d\n",info.cc);
    if ( sm->connex > 0 )
      fprintf(stdout,"     SUB-DOMAIN SAVED            %8d\n",sm->connex);
  }

  if ( !(sm->type & M_EXTEND) || sm->dim == 2 ) {
    LM_close_mesh(&ms);
    E_pop();
    return(1);
  }


  /* normals */
  nn = nbl = 0;
  for (k=1; k<=sm->nvmax; k++) {
    gs = &sm->geom[k];
    if ( gs->new > 0 ) {
      iadr = nbl * 3;
      tabf[iadr+0] = gs->vn[0];
      tabf[iadr+1] = gs->vn[1];
      tabf[iadr+2] = gs->vn[2];
      gs->new = ++nn;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Normals, nbl, tabf);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Normals, nbl, tabf);

  /* normals at vertices */
  np = nbl = 0;
  for (k=1; k<=sm->ne; k++) {
    pt1 = &sm->tria[k];
    if ( !pt1->v[0] ) continue;
    for (i=0; i<3; i++) {
      ppt = &sm->point[pt1->v[i]];
      if ( ppt->tag == M_NOTAG && !ppt->flag ) {
        iadr = nbl * 2;
        gs = &sm->geom[pt1->vn[i]];
        tabi[iadr+0] = ppt->tmp;
        tabi[iadr+1] = gs->new;
        ppt->flag = 1;
        ++nbl;
        if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_NormalAtVertices, nbl, tabi);
          nbl = 0;
        }
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_NormalAtVertices, nbl, tabi);

  /* normals at triangle vertices */
  nn = nbl = 0;
  for (k=1; k<=sm->ne; k++) {
    pt1 = &sm->tria[k];
    if ( !pt1->v[0] )  continue;
    else if ( sm->connex > 0 && pt1->cc != sm->connex )  continue;
    ++nn;
    for (i=0; i<3; i++) {
      ppt = &sm->point[pt1->v[i]];
      if ( ppt->tag > M_NOTAG ) {
        gs = &sm->geom[pt1->vn[i]];
        iadr = nbl * 3;
        tabi[iadr+0] = nn;
        tabi[iadr+1] = i+1;
        tabi[iadr+2] = gs->new;
        ++nbl;
        if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_NormalAtTriangleVertices, nbl, tabi);
          nbl = 0;
        }
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_NormalAtTriangleVertices, nbl, tabi);

  /* tangents */
  nt = nbl = 0;
  for (k=1; k<=sm->ntmax; k++) {
    gt = &sm->tgte[k];
    if ( gt->new > 0 ) {
      iadr = nbl * 3;
      tabf[iadr+0] = gt->t[0];
      tabf[iadr+1] = gt->t[1];
      tabf[iadr+2] = gt->t[2];
      gt->new  = ++nt;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Tangents, nbl, tabf);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Tangents, nbl, tabf);

  /* tangents at vertices */
  nbl = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    ppt->flag = 0;
    if ( ppt->tag & M_UNUSED ) continue;
    if ( ppt->tag > M_NOTAG && !(ppt->tag & M_CORNER) ) {
      iadr = nbl * 2;
      gt = &sm->tgte[ppt->tge];
      tabi[iadr+0]  = ppt->tmp;
      tabi[iadr+1]  = gt->new;
      ppt->flag = 1;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_TangentAtVertices, nbl, tabi);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_TangentAtVertices, nbl, tabi);

  LM_close_mesh(&ms);
  free(tabi);

  E_pop();
  return(1);
}


/* save mesh to disk */
int saveQuadMesh(pSurfMesh sm,char *fileout) {
  LM_mesh_struct   ms;
  pGeomSupp    gs;
  pGeomtge     gt;
  pPoint       ppt;
  pQuad        pq1;
  pEdge        pte;
  float       *tabf;
  int         *tabi,i,k,iadr,np,ne,nn,nq,nav,natv,tatv,nbl;
  int          nedge,nridge,ndang,nrequis;
  int          is1,ncorner,prequis;
  char        *ptr,data[128];

  E_put("saveQuadMesh");
  strcpy(data,fileout);
  ptr = strstr(data,".mesh");
  if ( !ptr ) {
    strcat(data,".meshb");
    if( !LM_open_mesh(data, LM_WRITE, &ms, sm->dim) ) {
      ptr  = strstr(data,".mesh");
      *ptr = '\0';
      strcat(data,".mesh");
      if( !LM_open_mesh(data, LM_WRITE, &ms, sm->dim) ) {
        fprintf(stderr,"  ** UNABLE TO OPEN %s.\n",data);
        return(0);
      }
    }
  }
  else if( !LM_open_mesh(data, LM_WRITE, &ms, sm->dim) ) {
    fprintf(stderr,"  ** UNABLE TO OPEN %s.\n",data);
    return(0);
  }
  if ( imprim )
    fprintf(stdout,"  %%%% %s OPENED\n",data);

  /* mark connected component */
  ne = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = & sm->point[k];
    ppt->tag |= M_UNUSED;
    ppt->flag = ppt->color = 0;
  }

  if ( sm->connex > 0 ) {
    for (k=1; k<=sm->nq; k++) {
      pq1 = &sm->quad[k];
      if ( pq1->v[0] > 0 && pq1->cc == sm->connex ) {
    ne++;
    for (i=0; i<4; i++) {
      ppt = &sm->point[pq1->v[i]];
      ppt->tag &= ~M_UNUSED;
    }
      }
    }
  }
  else {
    /* mark used faces */
    for (k=1; k<=sm->nq; k++) {
      pq1 = &sm->quad[k];
      if ( !pq1->v[0] )  continue;
      ++ne;
      for (i=0; i<4; i++) {
        ppt = &sm->point[pq1->v[i]];
        ppt->tag &= ~M_UNUSED;
      }
    }
  }

  /* mark used vertices */
  np = nav = 0;
  ncorner = prequis = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  continue;
    ppt->tmp = ++np;
    if ( ppt->tag == M_NOTAG )  nav++;
  }

  tabi = (int*)malloc(NMAX * 5 * sizeof(int));
  assert(tabi);
  tabf = (float*)tabi;

  nbl = 0;
  for(k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  continue;
    iadr = nbl * (sm->dim+1);
    tabf[ iadr + 0 ] = ppt->c[0];
    tabf[ iadr + 1 ] = ppt->c[1];
    if ( sm->dim == 3 ) {
      tabf[ iadr + 2 ] = ppt->c[2];
      tabi[ iadr + 3 ] = ppt->ref;
    }
    else
      tabi[ iadr + 2 ] = ppt->ref;
    ++nbl;
    if (ppt->tag & M_CORNER)    ncorner++;
    if (ppt->tag & M_REQUIRED ) prequis++;
    
    if ( nbl == NMAX ) {
      LM_write_field(&ms, LM_Vertices, nbl, tabf);
      nbl = 0;
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Vertices, nbl, tabf);

  /* write quads */
  nedge  = sm->dim == 3 ? info.ndang : 0;
  nridge = nrequis = nn = nq = natv = tatv = 0;
  nbl    = 0;
  for (k=1; k<=sm->nq; k++) {
    pq1  = &sm->quad[k];
    if ( !pq1->v[0] )  continue;
    else if ( sm->connex > 0 && pq1->cc != sm->connex ) continue;
    iadr = nbl * 5;
    tabi[ iadr + 0 ] = sm->point[pq1->v[0]].tmp;
    tabi[ iadr + 1 ] = sm->point[pq1->v[1]].tmp;
    tabi[ iadr + 2 ] = sm->point[pq1->v[2]].tmp;
    tabi[ iadr + 3 ] = sm->point[pq1->v[3]].tmp;
    tabi[ iadr + 4 ] = (int)(sm->connex < 0 ? pq1->cc : pq1->ref);
    ++nbl;
    for (i=0; i<4; i++) {
      ppt = &sm->point[pq1->v[i]];
      gs  = &sm->geom[pq1->vn[i]];
      gt  = &sm->tgte[ppt->tge];
      if ( ppt->tag > M_NOTAG ) {
        natv++;
        if ( ppt->tag & M_CORNER )  tatv++;
      }
      if ( !gs->new )  gs->new = ++nn;
      if ( !gt->new )  gt->new = ++nq;
      if ( !pq1->edg[i] && pq1->tag[i] == M_NOTAG )  continue;
      else if ( pq1->adj[i] && (k > pq1->adj[i]) )   continue;
      nedge++;
      if ( pq1->tag[i] & M_RIDGE_GEO )  nridge++;
      if ( pq1->tag[i] & M_REQUIRED )   nrequis++;
    }
    
    if ( nbl == NMAX ) {
      LM_write_field(&ms, LM_Quadrilaterals, nbl, tabi);
      nq += nbl;
      nbl = 0;
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Quadrilaterals, nbl, tabi);
  nq += nbl;

  /* corners */
  ncorner = nbl = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  continue;
    else if ( ppt->tag & M_CORNER ) {
      tabi[nbl] = ppt->tmp;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Corners, nbl, tabi);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Corners, nbl, tabi);

  /* required vertices */
  prequis = nbl = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    if ( ppt->tag & M_UNUSED )  continue;
    else if ( ppt->tag & M_REQUIRED ) {
      tabi[nbl] = ppt->tmp;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_RequiredVertices, nbl, tabi);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_RequiredVertices, nbl, tabi);

  /* edges */
  nedge = nbl = 0;
  for (k=1; k<=sm->nq; k++) {
    pq1 = &sm->quad[k];
    if ( !pq1->v[0] )  continue;
    else if ( sm->connex > 0 && pq1->cc != sm->connex ) continue;
    for (i=0; i<4; i++) {
      if ( !pq1->edg[i] && pq1->tag[i] == M_NOTAG ) continue;
      else if ( pq1->adj[i] && (k > pq1->adj[i]) )  continue;
      is1  = idirq[i+1];
      iadr = nbl * 3;
      tabi[ iadr+0 ] = sm->point[pq1->v[i]].tmp;
      tabi[ iadr+1 ] = sm->point[pq1->v[is1]].tmp;
      tabi[ iadr+2 ] = pq1->edg[i];
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Edges, nbl, tabi);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Edges, nbl, tabi);

  /* dangling edges */
  if ( info.ndang > 0 ) {
    ndang = nbl = 0;
    for (k=1; k<=info.ndang; k++) {
      pte = &sm->edge[k];
      iadr = nbl * 3;
      if ( pte->p1 * pte->p2 ) {
        tabi[ iadr+0 ] = pte->p1;   
        tabi[ iadr+0 ] = pte->p2;   
        tabi[ iadr+0 ] = pte->ref;
        ++nbl;
        if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_Edges, nbl, tabi);
          nbl = 0;
        }
      }
    }
    if ( nbl )  LM_write_field(&ms, LM_Edges, nbl, tabi);
  }

  /* ridges */
  nedge = nbl = 0;
  for (k=1; k<=sm->nq; k++) {
    pq1 = &sm->quad[k];
    if ( !pq1->v[0] ) continue;
    else if ( sm->connex > 0 && pq1->cc != sm->connex ) continue;
    for (i=0; i<4; i++) {
      if ( !pq1->edg[i] && pq1->tag[i] == M_NOTAG ) continue;
      else if ( pq1->adj[i] && (k > pq1->adj[i]) )  continue;
      ++nedge;
      if ( pq1->tag[i] & M_RIDGE_GEO ) {
    tabi[nbl] = nedge;
    nbl++;
    if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_Ridges, nbl, tabi);
          nbl = 0;
        }
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Ridges, nbl, tabi);

  /* required edges */
  nedge = nbl = 0;
  for (k=1; k<=sm->nq; k++) {
    pq1 = &sm->quad[k];
    if ( !pq1->v[0] ) continue;
    else if ( sm->connex > 0 && pq1->cc != sm->connex ) continue;
    for (i=0; i<4; i++) {
      if ( !pq1->edg[i] && pq1->tag[i] == M_NOTAG ) continue;
      else if ( pq1->adj[i] && (k > pq1->adj[i]) )  continue;
      ++nedge;
      if ( pq1->tag[i] & M_REQUIRED ) {
    tabi[nbl] = nedge;
    nbl++;
    if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_RequiredEdges, nbl, tabi);
          nbl = 0;
        }
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_RequiredEdges, nbl, tabi);

  if ( imprim ) {
    fprintf(stdout,"     NUMBER OF GIVEN VERTICES    %8d\n",sm->npfixe);
    fprintf(stdout,"     NUMBER OF GIVEN TRIANGLES   %8d\n",sm->nefixe);
    fprintf(stdout,"     TOTAL NUMBER OF VERTICES    %8d\n",np);
    fprintf(stdout,"     TOTAL NUMBER OF QUADS       %8d\n",nq);
    if ( info.cc > 1 )
      fprintf(stdout,"     NUMBER OF SUB-DOMAINS       %8d\n",info.cc);
    if ( sm->connex > 0 )
      fprintf(stdout,"     SUB-DOMAIN SAVED            %8d\n",sm->connex);
  }

  if ( !(sm->type & M_EXTEND) || sm->dim == 2 ) {
    LM_close_mesh(&ms);
    E_pop();
    return(1);
  }

  /* normals */
  nn = nbl = 0;
  for (k=1; k<=sm->nvmax; k++) {
    gs = &sm->geom[k];
    if ( gs->new > 0 ) {
      iadr = nbl * 3;
      tabf[iadr+0] = gs->vn[0];
      tabf[iadr+1] = gs->vn[1];
      tabf[iadr+2] = gs->vn[2];
      gs->new = ++nn;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Normals, nbl, tabf);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Normals, nbl, tabf);

  /* normals at vertices */
  np = nbl = 0;
  for (k=1; k<=sm->nq; k++) {
    pq1 = &sm->quad[k];
    if ( !pq1->v[0] ) continue;
    for (i=0; i<4; i++) {
      ppt = &sm->point[pq1->v[i]];
      if ( ppt->tag == M_NOTAG && !ppt->flag ) {
        iadr = nbl * 2;
        gs = &sm->geom[pq1->vn[i]];
        tabi[iadr+0] = ppt->tmp;
        tabi[iadr+1] = gs->new;
        ppt->flag = 1;
        ++nbl;
        if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_NormalAtVertices, nbl, tabi);
          nbl = 0;
        }
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_NormalAtVertices, nbl, tabi);

  /* normals at quad vertices */
  nn = nbl = 0;
  for (k=1; k<=sm->nq; k++) {
    pq1 = &sm->quad[k];
    if ( !pq1->v[0] )  continue;
    else if ( sm->connex > 0 && pq1->cc != sm->connex )  continue;
    ++nn;
    for (i=0; i<4; i++) {
      ppt = &sm->point[pq1->v[i]];
      if ( ppt->tag > M_NOTAG ) {
    gs = &sm->geom[pq1->vn[i]];
    iadr = nbl * 3;
    tabi[iadr+0] = nn;
    tabi[iadr+1] = i+1;
    tabi[iadr+2] = gs->new;
    ++nbl;
        if ( nbl == NMAX ) {
          LM_write_field(&ms, LM_NormalAtQuadrilateralVertices, nbl, tabi);
          nbl = 0;
        }
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_NormalAtQuadrilateralVertices, nbl, tabi);

  /* tangents */
  nq = nbl = 0;
  for (k=1; k<=sm->ntmax; k++) {
    gt = &sm->tgte[k];
    if ( gt->new > 0 ) {
      iadr = nbl * 3;
      tabf[iadr+0] = gt->t[0];
      tabf[iadr+1] = gt->t[1];
      tabf[iadr+2] = gt->t[2];
      gt->new  = ++nq;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_Tangents, nbl, tabf);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_Tangents, nbl, tabf);

  /* tangents at vertices */
  nbl = 0;
  for (k=1; k<=sm->npmax; k++) {
    ppt = &sm->point[k];
    ppt->flag = 0;
    if ( ppt->tag & M_UNUSED ) continue;
    if ( ppt->tag > M_NOTAG && !(ppt->tag & M_CORNER) ) {
      iadr = nbl * 2;
      gt = &sm->tgte[ppt->tge];
      tabi[iadr+0]  = ppt->tmp;
      tabi[iadr+1]  = gt->new;
      ppt->flag = 1;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ms, LM_TangentAtVertices, nbl, tabi);
        nbl = 0;
      }
    }
  }
  if ( nbl )  LM_write_field(&ms, LM_TangentAtVertices, nbl, tabi);


  LM_close_mesh(&ms);
  free(tabi);

  E_pop();
  return(1);
}


int loadSol(pSurfMesh sm,char *filein) {
  LM_mesh_struct   ss;
  pPoint   ppt;
  pMetric  pm;
  double   sizeh,m[6],lambda[3],vp[2][2],vp3[3][3];
  float   *tabf,hmin,hmax;
  int      i,k,iadr,dim,nbl,np,size,type,offset;
  char    *ptr,data[128];

  E_put("loadSol");
  strcpy(data,filein);
  ptr = strstr(data,".mesh");
  if ( ptr )  *ptr = '\0';
  strcat(data,ecp ? ".sol" : ".solb");
  if( !LM_open_mesh(data, LM_READ, &ss) ) {
    ptr = strstr(data,".sol");
    if ( ptr )  *ptr = '\0';
    strcat(data,".sol");
    if( !LM_open_mesh(data, LM_READ, &ss) ) {
      E_pop();
      return(0);
    }
  }
  fprintf(stdout,"  %%%% %s OPENED\n",data);

  np = ss.kw_counters[ LM_SolAtVertices ];
  if ( !np ) {
    fprintf(stdout,"  ** MISSING DATA\n");
    E_pop();
    return(0);
  }
  else if ( np != sm->np ) {
    fprintf(stdout,"  ** WRONG SOLUTION NUMBER (%d,%d)\n",np,sm->np);
    E_pop();
    return(0);
  }

  dim  = ss.dimension;
  size = ss.sol_headers[ LM_SolAtVertices ][1];
  type = ss.sol_headers[ LM_SolAtVertices ][2];
  switch(type) {
  case 1:
    offset = 1;
    break;
  case 3:
    offset = dim==2 ? 3 : 6;
    break;
  default:
    fprintf(stdout,"  ** DATA IGNORED\n");
    E_pop();
    return(0);
  }

  if ( abs(imprim) > 5 )
    fprintf(stdout,"  -- READING DATA FILE %s\n",data);

  /* read mesh solutions */
  tabf = (float*)malloc(NMAX * size * sizeof(float));
  assert(tabf);

  opts.ctrl &= ~REL;
  hmin =  FLT_MAX;
  hmax = -FLT_MAX;

  if ( type == 1 ) {
    np = 0;
    while ( (nbl = LM_read_field(&ss,LM_SolAtVertices,NMAX, tabf)) ) {
      for (k=0; k<nbl; k++) {
        ++np;
        ppt = &sm->point[np];
        ppt->size = tabf[k];
        hmin = min(ppt->size,hmin);
        hmax = max(ppt->size,hmax);
      }
    }
  }

  else if ( sm->dim == 2 && type == 3 ) {
    if ( !sm->metric && !zaldy3(sm,3) )  return(0);
    np = 0;
    while ( (nbl = LM_read_field(&ss, LM_SolAtVertices, NMAX, tabf)) ) {
      for (k=0; k<nbl; k++) {
    iadr = k * size;
        ++np;
        ppt = &sm->point[np];
        pm  = &sm->metric[np];
        memset(pm->m,6*sizeof(float),0.);
        for (i=0; i<3; i++)
          m[i] = tabf[iadr+i];

        eigen2(m,lambda,vp);
        sizeh     = max(lambda[0],lambda[1]);
        ppt->size = max(1.0 / sqrt(sizeh),EPS);
        hmin = min(ppt->size,hmin);
        hmax = max(ppt->size,hmax);
        pm->m[0] = m[0];
        pm->m[1] = m[1];
        pm->m[3] = m[2];
        pm->m[5] = 0.1*min(fabs(m[0]),fabs(m[2]));
        pm->m[2] = pm->m[4] = 0.0;
        pm->k1 = pm->k2 = (float)FLT_MAX;
      }
    }

    if ( opts.ctrl & ISO ) opts.ctrl ^= ISO; 
  }

  else if ( sm->dim == 3 && type == 3 ) {
    if ( !sm->metric && !zaldy3(sm,3) )  return(0);
    np = 0;
    while ( (nbl = LM_read_field(&ss, LM_SolAtVertices, NMAX, tabf)) ) {
      for (k=0; k<nbl; k++) {
    iadr = k * size;
        ++np;
    ppt = &sm->point[np];
    pm  = &sm->metric[np];
    memset(pm->m,6*sizeof(float),0.);
    for (i=0; i<6; i++)
      m[i] = tabf[iadr+i];

        pm->m[0] = m[0];
        pm->m[1] = m[1];
        pm->m[2] = m[3];
        pm->m[3] = m[2];
        pm->m[4] = m[4];
        pm->m[5] = m[5];
        pm->k1   = pm->k2 = (float)FLT_MAX;
        for (i=0; i<6; i++)  m[i] = pm->m[i];
        if ( !eigenv(1,m,lambda,vp3) ) {
          fprintf(stderr,"  ## ERR 9201, inbbf, Not a metric tensor. Discarded\n");
          M_free(sm->metric);
          sm->metric = 0;
          break;
        }
        sizeh     = max(max(lambda[0],lambda[1]),lambda[2]);
        ppt->size = max(1.0 / sqrt(sizeh),EPS);
        hmin = min(ppt->size,hmin);
        hmax = max(ppt->size,hmax);
      }
    }

    if ( opts.ctrl & ISO ) opts.ctrl &= ~ISO; 
  }
  LM_close_mesh(&ss);
  free(tabf);

  /* adjust sizes */
  if ( opts.hmin < 0.0 )
    opts.hmin = max(opts.hmin,hmin);
    if ( opts.hmax < 0.0 )
      opts.hmax = max(opts.hmax,hmax);

  E_pop();
  return(1);
}


int saveSol(pSurfMesh sm,char *fileout) {
  LM_mesh_struct   ss;
  pPoint       ppt;
  pMetric      pm;
  float       *tabf;
  int          k,iadr,offset,nbl,type;

  E_put("saveSol");
  if( !LM_open_mesh(fileout, LM_WRITE, &ss, sm->dim) ) {
    fprintf(stderr,"  ** UNABLE TO OPEN %s.\n",fileout);
    return(0);
  }
  else if ( imprim ) 
    fprintf(stdout,"  %%%% %s OPENED\n",fileout);

  if ( opts.ctrl & ISO ) {
    tabf = (float*)malloc(NMAX * sizeof(float));
    assert(tabf);

    type = 1;
    nbl  = 0;
    for (k=1; k<=sm->np; k++) {
      ppt = &sm->point[k];
      if ( ppt->tag & M_UNUSED )  continue;
      tabf[nbl] = ppt->size;
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ss, LM_SolAtVertices, nbl, tabf, 1, type);
        nbl = 0;
      }
    }
    if ( nbl )  LM_write_field(&ss, LM_SolAtVertices, nbl, tabf, 1, type);
  }

  else {
    offset = sm->dim == 2 ? 3: 6;
    tabf = (float*)malloc(NMAX * offset * sizeof(float));
    assert(tabf);

    type = 3;
    nbl  = 0;
    for (k=1; k<=sm->np; k++) {
      ppt = &sm->point[k];
      if ( ppt->tag & M_UNUSED )  continue;
      pm   = &sm->metric[k];
      iadr = nbl * offset;
      /* swap data */
      if ( offset == 6 ) {
        tabf[ iadr + 0 ] = pm->m[0]; 
        tabf[ iadr + 1 ] = pm->m[1]; 
        tabf[ iadr + 2 ] = pm->m[3]; 
        tabf[ iadr + 3 ] = pm->m[2]; 
        tabf[ iadr + 4 ] = pm->m[4]; 
        tabf[ iadr + 5 ] = pm->m[5]; 
      }
      else {
        tabf[ iadr + 0 ] = pm->m[0]; 
        tabf[ iadr + 1 ] = pm->m[1]; 
        tabf[ iadr + 2 ] = pm->m[3]; 
      }
      ++nbl;
      if ( nbl == NMAX ) {
        LM_write_field(&ss, LM_SolAtVertices, nbl, tabf, 1, type);
        nbl = 0;
      }
    }
    if ( nbl )  LM_write_field(&ss, LM_SolAtVertices, nbl, tabf, 1, type);
  }

  free(tabf);
  LM_close_mesh(&ss);

  E_pop();
  return(1);
}
