/*
 *  vertex smoothing (Hilbert's)
 *
 *  Written by Pascal J. Frey, Inria-Rocquencourt
 *  Copyright (c) Inria, 2002-2003. All rights reserved. 
*/
#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"

#define HLBCOEF   1.e-02

typedef struct scoor {
  double  kg;
  int     nv;
} Coor;


static int curvpt(pSurfMesh sm,Coor *pts) {
  pTriangle   pt,pt1,pta;
  pPoint      p1,p2;
  Ball        bb;
  Coor       *npt;
  double      dd,ux,uy,uz;
  double      x,y,z,x2,xy,y2,z2,maxd,sum,prd,k1,k2;
  double      ma[6],mb[3],mx[3];
  float      *np,c[3],nn[3],b1[3],b2[3];
  int         i,k,kk,nk,l,b,i1,i2,nb;

  /* default */
  E_put("curvpt");
  if ( imprim < -4 )  primsg(1015);

  /* analyze triangle vertices */
  ++sm->mark;
  for (k=1; k<=sm->ne; k++) {
    pt = &sm->tria[k];
    if ( !pt->v[0] )  continue;

    for (i=0; i<3; i++) {
      i1 = idir[i+1];
      i2 = idir[i+2];
      p1 = &sm->point[pt->v[i]];

      /*if ( p1->tag > M_NOTAG )  continue;*/
      if ( p1->color == sm->mark ) continue;
      npt     = &pts[pt->v[i]];
      npt->kg = 0.0;
      npt->nv = pt->vn[i];
      np      = sm->geom[pt->vn[i]].vn;

      /* referentiel (b1,b2,normale) */
      if ( fabs(np[0]) > EPS ) {
	    b1[0] = -(np[1]+np[2]) / np[0];
	    b1[1] = b1[2] = 1.0;
      }
      else if ( fabs(np[1]) > EPS ) {
	    b1[1] = -(np[0]+np[2]) / np[1];
	    b1[0] = b1[2] = 1.0;
      }
      else if ( fabs(np[2]) > EPS ) {
	    b1[2] = -(np[0]+np[1]) / np[2];
	    b1[0] = b1[1] = 1.0;
      }
      else {
        b  = pt->v[i1];
        p2 = &sm->point[b];
	    b1[0] = p2->c[0] - p1->c[0];
	    b1[1] = p2->c[1] - p1->c[1];
	    b1[2] = p2->c[2] - p1->c[2];
	    dd = b1[0]*np[0] + b1[1]*np[1] + b1[2]*np[2];
	    b1[0] = b1[0] - dd*np[0];
   	    b1[1] = b1[1] - dd*np[1];
	    b1[2] = b1[2] - dd*np[2];
      }
      dd = b1[0]*b1[0] + b1[1]*b1[1] + b1[2]*b1[2];
      if ( dd == 0.0f ) continue;
      dd = 1.0 / sqrt(dd);
      b1[0] *= dd;
      b1[1] *= dd;
      b1[2] *= dd;

      b2[0] = np[1]*b1[2] - np[2]*b1[1];
      b2[1] = np[2]*b1[0] - np[0]*b1[2];
      b2[2] = np[0]*b1[1] - np[1]*b1[0];

      /* get ball of vertex */
      bb.ilist = boulep(sm,k,i,&bb);
      if ( bb.ilist < 1 || !bb.closed )  continue; 

      for (l=0; l<6; l++)  ma[l] = 0.0f;
      for (l=0; l<3; l++)  mb[l] = mx[l] = 0.0f;
      maxd = 0.0f;
      nb   = 0;
      for (l=1; l<=bb.ilist; l++) {
	    kk = bb.list[l];
	    nk = bb.nump[l];
	    i1 = idir[nk+1];
	    pt1 = &sm->tria[kk];
	    p2  = &sm->point[pt1->v[i1]];
	    ux = p2->c[0] - p1->c[0];
	    uy = p2->c[1] - p1->c[1];
	    uz = p2->c[2] - p1->c[2];

    	/* coordinates of u in local frame */
	    x = ux*b1[0] + uy*b1[1] + uz*b1[2];
	    y = ux*b2[0] + uy*b2[1] + uz*b2[2];
	    z = ux*np[0] + uy*np[1] + uz*np[2];
        x2 = x*x;
        xy = 2*x*y;
        y2 = y*y;
        z2 = z*z;
	    ma[0] += x2*x2;
	    ma[1] += x2*xy;
	    ma[2] += x2*y2;
	    ma[3] += 4*x2*y2;
	    ma[4] += xy*y2;
	    ma[5] += y2*y2;
	    mb[0] += x2*z;
	    mb[1] += xy*z;
	    mb[2] += y2*z;
        if ( maxd < z2 )  maxd = z2;
        nb++;

        /* use mid-point */
        if ( coorpo(sm,kk,nk,c,0.5,nn) ) {
          ux = c[0] - p1->c[0];
	      uy = c[1] - p1->c[1];
	      uz = c[2] - p1->c[2];
	      x  = ux*b1[0] + uy*b1[1] + uz*b1[2];
	      y  = ux*b2[0] + uy*b2[1] + uz*b2[2];
	      z  = ux*np[0] + uy*np[1] + uz*np[2];
          x2 = x*x;
          xy = 2*x*y;
          y2 = y*y;
          z2 = z*z;
	      ma[0] += x2*x2;
	      ma[1] += x2*xy;
	      ma[2] += x2*y2;
	      ma[3] += 4*x2*y2;
	      ma[4] += xy*y2;
	      ma[5] += y2*y2;
	      mb[0] += x2*z;
	      mb[1] += xy*z;
	      mb[2] += y2*z;
          if ( maxd < z2 )  maxd = z2;
          nb++;
        }
         /* use point voyeur */
        if ( pt1->adj[nk] ) {
          pta = &sm->tria[pt1->adj[nk]];
          dd  = pt1->n[0]*pta->n[0] + pt1->n[1]*pta->n[1] \
              + pt1->n[2]*pta->n[2];
          if ( fabs(dd) > COS1DEG ) {
            i2 = pt1->voy[nk];
	        p2 = &sm->point[pta->v[i2]];
	        ux = p2->c[0] - p1->c[0];
	        uy = p2->c[1] - p1->c[1];
	        uz = p2->c[2] - p1->c[2];
	        x  = ux*b1[0]  + uy*b1[1]  + uz*b1[2];
	        y  = ux*b2[0]  + uy*b2[1]  + uz*b2[2];
	        z  = ux*np[0] + uy*np[1] + uz*np[2];
	        x2 = x*x;
	        y2 = y*y;
	        xy = 2.0*x*y;
	        ma[0] += x2*x2;
	        ma[1] += x2*xy;
	        ma[2] += x2*y2;
	        ma[3] += 4*x2*y2;
            ma[4] += xy*y2;
	        ma[5] += y2*y2;
	        mb[0] += x2*z;
	        mb[1] += xy*z;
	        mb[2] += y2*z;
	        nb++;
          }
        }
      }
      p1->color = sm->mark;
      if ( maxd < EPSD || nb < 3 || !sol3x3(ma,mb,mx) )
        continue;

      sum = 2.0*(mx[0]+mx[2]);
      prd = 4.0*(mx[0]*mx[2]-mx[1]*mx[1]);
      dd  = sum*sum - 4.0*prd;
      if ( dd > 0.0 )  dd = sqrt(dd);
      k1 = 0.5 * (sum + dd);
      k2 = 0.5 * (sum - dd);

      npt->kg = 4.0*(mx[0]*mx[2]-mx[1]*mx[1]);
      npt->kg = 2.0*(mx[0]+mx[2]);
    }
  }

  E_pop();
  return(1);
}


int hilbert(pSurfMesh sm) {
  pPoint      ppt;
  Coor       *newpts,*npt;
  double      dx,dy,dz,dd,res,res0,cc,delta;
  double      xmin,xmax,ymin,ymax,zmin,zmax;
  float      *nv;
  int         k,it;

  E_put("hilbert");
  if ( imprim < -4 )    primsg(1055);
  if ( opts.iter < 0 )  opts.iter = 50;

  newpts = (Coor*)M_malloc((sm->np+1)*sizeof(Coor),"hilbert");
  if ( !newpts )  return(0);
  delta = info.delta;
  res0  = 0.0;

  for (it=1; it<=opts.iter; it++) {
    /* 1st stage: compute properties */
    if ( !norpoi(sm,0,0) )     return(0);
    if ( !curvpt(sm,newpts) )  return(0);

    /* 2nd stage: update coord */
    res = 0.0;
    cc  = HLBCOEF *delta;
    xmin = ymin = zmin =  1.0e30;
    xmax = ymax = zmax = -1.0e30;
    for (k=1; k<=sm->np; k++) {
      ppt = &sm->point[k];
      /*if ( ppt->tag > M_NOTAG )  continue;*/
      npt = &newpts[k];
      nv  = sm->geom[npt->nv].vn;
      dd  = cc * npt->kg;
      
      dx = dd*nv[0];
      dy = dd*nv[1];
      dz = dd*nv[2];
      res += sqrt(dx*dx + dy*dy + dz*dz);
      
      ppt->c[0] += dx;
      ppt->c[1] += dy;
      ppt->c[2] += dz;
      npt->kg    = 0.0f;

      if ( ppt->c[0] < xmin )  xmin = ppt->c[0];
      if ( ppt->c[0] > xmax )  xmax = ppt->c[0];
      if ( ppt->c[1] < ymin )  ymin = ppt->c[1];
      if ( ppt->c[1] > ymax )  ymax = ppt->c[1];
      if ( ppt->c[2] < zmin )  zmin = ppt->c[2];
      if ( ppt->c[2] > zmax )  zmax = ppt->c[2];

      ppt->color = 0;
    }
    delta  = (xmax-xmin) * (xmax-xmin);
    delta += (ymax-ymin) * (ymax-ymin);
    delta += (zmax-zmin) * (zmax-zmin);
    delta  = sqrt(delta);

    if ( imprim > 4 ) {
      fprintf(stdout,"    iter %8d:  %8d moved  res %E\n",it,sm->np,res); 
      fflush(stdout);
    }
    if ( it == 1 )  res0 = res;
    else if ( res < 0.01 * res0 )
      break;
  }

  M_free(newpts);
  E_pop();
  return(1);
}
