/*
 *  Delete badly shaped elements
 *
 *  Written by Pascal J. Frey, Inria-Rocquencourt
 *  Copyright (c) Inria, 2001.  All rights reserved. 
*/

#include "yams.h"
#include "defines.h"
#include "sproto.h"
#include "extern.h"

#define RINMIN   1.0E-03
#define SIZMIN   1.0E-06


/* return 1: flat tria, 2: needle, 3: obtuse, 0: OK */
int typelt(pSurfMesh sm,int k) {
  pTriangle  pt;
  pPoint     p0,p1,p2;
  double     d1,d2,d3,cc,ux,uy,uz,dmin,dmax;
  double     peri,surfa,rins;
  ubyte      imin,imax,rinsok;

  cc = 1.0 / info.delta;
  pt = &sm->tria[k];
  p0 = &sm->point[pt->v[0]];
  p1 = &sm->point[pt->v[1]];
  p2 = &sm->point[pt->v[2]];

  /* in-radius */
  ux = p1->c[0] - p0->c[0];
  uy = p1->c[1] - p0->c[1];
  uz = p1->c[2] - p0->c[2];
  d1 = sqrt(ux*ux + uy*uy + uz*uz);
  dmin = dmax = d1;
  imin = imax = 2;

  ux = p2->c[0] - p0->c[0];
  uy = p2->c[1] - p0->c[1];
  uz = p2->c[2] - p0->c[2];
  d2 = sqrt(ux*ux + uy*uy + uz*uz);
  if ( d2 < dmin ) {
    dmin = d2;
    imin = 1;
  }
  else if ( d2 > dmax ) {
    dmax = d2;
    imax = 1;
  }
  ux = p2->c[0] - p1->c[0];
  uy = p2->c[1] - p1->c[1];
  uz = p2->c[2] - p1->c[2];
  d3 = sqrt(ux*ux + uy*uy + uz*uz);
  if ( d3 < dmin ) {
    dmin = d3;
    imin = 0;
  }
  else if ( d3 > dmax ) {
    dmax = d3;
    imax = 0;
  }

  rins  = 0.0;
  peri  = 0.5 * (d1 + d2 + d3);
  surfa = peri * (peri-d1) * (peri-d2) * (peri-d3);
  if ( surfa > 0.0 ) {
    surfa = sqrt(surfa);
    if ( peri != 0.0 )  rins  = surfa / peri;
  }
  rinsok = cc*rins > RINMIN;

  /* null triangle */
  if ( dmin < EPSD )
    return(4);
  else if ( (dmax-dmin) < SIZMIN*info.delta ) 
    return(1);

  /* needle */
  else if ( dmin < 0.05*dmax )
    return(2);

  /* obtuse angle */
  else if ( !rinsok )
    return(3);
  
  return(0);
}


/* delete badly shaped elements */
int delnul(pSurfMesh sm,float declic) {
  pTriangle  pt;
  pPoint     ppt;
  Global     gl;
  Ball       bb;
  float      n1[3],n2[3],q1,q2;
  int        k,kk,l,nk,nsw,nde,npp,typ;
  ubyte      i,i1,i2;

  /* default */
  E_put("delnul");
  if ( imprim < 0 )  primsg(2005);

  /* update aniso qual */
  if ( !(opts.ctrl & ISO) ) {
    if ( sm->metric )  
      updqua_a(sm);
    else 
      declic = EPS / ALPHAC;
  }
  nsw = npp = nde = 0;

  /* remove isolated triangles */
  //if ( updrf && !updref(sm) )  return(0);

  sm->mark++;
  if ( !hipini(sm,2) )  return(0);

  do {
    k = hippop(sm->tria);
    if ( !k )  break;

    pt = &sm->tria[k];
    if ( !pt->v[0] || pt->flag1 > sm->mark )  continue;
    else if ( pt->qual > declic )  continue;

    npp++;
    pt->flag1 = sm->mark + 1;

    /* attempt to flip */
    for (i=0; i<3; i++) {
      if ( pt->tag[i] > M_NOTAG )  continue;
      if ( flipa0(sm,k,i,n1,n2,&q1,&q2) ) {
        nsw += swapar(sm,k,i,n1,n2,q1,q2,0.,0.);
        break;
      }
    }
    if ( i < 3 )  continue;

    /* Attempt to collapse */
    for (i=0; i<3; i++) {
      i1 = idir[i+1];
      i2 = idir[i+2];
      if ( pt->tag[i] & M_RIDGE_GEO ) {
        if ( deled0(sm,k,i1,i2,&gl) ) {
          nde += coledg(sm,k,i1,i2,&gl);
          break;
        }
        else if ( deled0(sm,k,i2,i1,&gl) ) {
          nde += coledg(sm,k,i2,i1,&gl);
          break;
        }
      }
      else {
        if ( delpo0(sm,k,i) ) {
          nde++;
          break;
        }
      }
    }
    if ( i < 3 )  continue;

    /* check neighbors */
    for (i=0; i<3; i++) {
      ppt = &sm->point[pt->v[i]];
      if ( ppt->tag > M_NOTAG )  continue;

      i2 = idir[i+2];
      bb.ilist = boulep(sm,k,i2,&bb);
      if ( bb.ilist < 3 || !bb.closed )  continue;

      for (l=1; l<=bb.ilist; l++) {
        kk = bb.list[l];
        nk = bb.nump[l];
        if ( kk == k )  continue;
        if ( delpo0(sm,kk,idir[nk+1]) || delpo0(sm,kk,idir[nk+2]) ) {
          nde++;
          break;
        }
      }
      if ( l <= bb.ilist )  break;
    }
    if ( i < 3 )  continue;

    /* unable to remove */
    typ = 0;
    //typ = typelt(sm,k);
  }
  while ( k );

  if ( abs(imprim) > 2 && nde+nsw ) {
    yerr.inderr[0] = npp;
    yerr.inderr[1] = nde;
    yerr.inderr[2] = nsw;
    primsg(4009);
  }

  info.nulp += nde;
  hipfree();
  E_pop();
  return(1);
}

