/*!\file: sealevelchange_core.cpp
 * \brief: core of the sea-level change solution 
 */ 

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../classes/Inputs/TriaInput.h"
#include "../classes/Inputs/TransientInput.h"
#include "../classes/Inputs/DatasetInput.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"


/*main cores:*/
void sealevelchange_core(FemModel* femmodel){ /*{{{*/

	/*Start profiler*/
	femmodel->profiler->Start(SLRCORE);

	/*Parameters, variables:*/
	bool save_results;
	bool isslr=0;
	bool isgia=0;
	int  grd=0;
	int  isexternal=0;
	int  iscoupler=0;
	int solution_type;

	/*Retrieve parameters:*/
	femmodel->parameters->FindParam(&isslr,TransientIsslrEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&save_results,SaveResultsEnum);
	femmodel->parameters->FindParam(&grd,SolidearthSettingsGRDEnum); 
	femmodel->parameters->FindParam(&isexternal,SolidearthIsExternalEnum); 
	femmodel->parameters->FindParam(&iscoupler,TransientIscouplerEnum);

	/*in case we are running SealevelchangeSolutionEnum, then bypass transient settings:*/
	if(solution_type==SealevelchangeSolutionEnum){
		isslr=1;
		isgia=1;
	}

	/*Should we be here?:*/
	if(!isslr)return;

	/*Verbose: */
	if(VerboseSolution()) _printf0_("   computing sea level change\n");

	/*Run gia core: */
	if(isgia){
		#ifdef _HAVE_GIA_
		gia_core(femmodel);
		#else
		_error_("ISSM was not compiled with gia capabilities. Exiting");
		#endif
	}

	/*set SLR configuration: */
	femmodel->SetCurrentConfiguration(SealevelchangeAnalysisEnum);

	/*run geometry core: */
	sealevelchange_geometry(femmodel);

	/*any external forcings?:*/
	if(isexternal)solidearthexternal_core(femmodel);

	/*Run geodetic:*/
	//if(modelid==earthid)  //not sure how we proceed yet on coupling.
	if(grd)grd_core(femmodel);

	/*Run steric core for sure:*/
	dynstr_core(femmodel);

	if(iscoupler){
		/*transfer sea level back to ice caps:*/
		TransferSealevel(femmodel,SealevelEnum);
		TransferSealevel(femmodel,BedEnum);

		//reset cumdeltathickness  to 0: 
		InputUpdateFromConstantx(femmodel->inputs,femmodel->elements,0.,SealevelchangeCumDeltathicknessEnum);
	}

	/*Save results: */
	if(save_results){
		int     numoutputs;
		char **requested_outputs = NULL;
		if(VerboseSolution()) _printf0_("   saving results\n");
		femmodel->parameters->FindParam(&requested_outputs,&numoutputs,SealevelchangeRequestedOutputsEnum);
		femmodel->RequestedOutputsx(&femmodel->results,requested_outputs,numoutputs);
		if(numoutputs){for(int i=0;i<numoutputs;i++){xDelete<char>(requested_outputs[i]);} xDelete<char*>(requested_outputs);}
	}

	/*requested dependents: */
	if(solution_type==SealevelchangeSolutionEnum)femmodel->RequestedDependentsx();
	
	/*End profiler*/
	femmodel->profiler->Stop(SLRCORE);
}
/*}}}*/
void solidearthexternal_core(FemModel* femmodel){ /*{{{*/

	/*variables:*/
	Vector<IssmDouble> *bedrock  = NULL; 
	Vector<IssmDouble> *bedrock_rate = NULL;
	Vector<IssmDouble> *bedrockeast  = NULL; 
	Vector<IssmDouble> *bedrockeast_rate = NULL;
	Vector<IssmDouble> *bedrocknorth  = NULL; 
	Vector<IssmDouble> *bedrocknorth_rate = NULL;
	Vector<IssmDouble> *geoid= NULL; 
	Vector<IssmDouble> *geoid_rate= NULL; 
	int horiz=0;
	int modelid=-1;
   
	/*parameters: */
	bool isslr=0;
	int  solution_type;
	IssmDouble          dt;

	/*Retrieve parameters:*/
	femmodel->parameters->FindParam(&isslr,TransientIsslrEnum);
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);

	/*in case we are running SealevelchangeSolutionEnum, then bypass transient settings:*/
	if(solution_type==SealevelchangeSolutionEnum)isslr=1;

	/*Should we be here?:*/
	if(!isslr)return;

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing external solid earth contributions\n");

	/*Retrieve geoid viscous and elastic rates, bedrock uplift viscous and elastic rates + steric rate, as vectors:*/
	GetVectorFromInputsx(&bedrock,femmodel,BedEnum,VertexSIdEnum);
	GetVectorFromInputsx(&geoid,femmodel,SealevelEnum,VertexSIdEnum); //In ISSM, Sealevel is absolute.
	if(horiz){
		GetVectorFromInputsx(&bedrockeast,femmodel,BedEastEnum,VertexSIdEnum);
		GetVectorFromInputsx(&bedrocknorth,femmodel,BedNorthEnum,VertexSIdEnum);
	}
	/*Deal with Mme: */
	if (femmodel->inputs->GetInputObjectEnum(SolidearthExternalDisplacementEastRateEnum)==DatasetInputEnum){
		/*retrieve model id: */
		femmodel->parameters->FindParam(&modelid,SolidearthExternalModelidEnum);
		/*replace dataset of forcings with only one, the modelid'th:*/
		femmodel->MmeToInputFromId(modelid,SolidearthExternalDisplacementNorthRateEnum, P1Enum);
		femmodel->MmeToInputFromId(modelid,SolidearthExternalDisplacementEastRateEnum, P1Enum);
		femmodel->MmeToInputFromId(modelid,SolidearthExternalDisplacementUpRateEnum, P1Enum);
		femmodel->MmeToInputFromId(modelid,SolidearthExternalGeoidRateEnum, P1Enum);
		femmodel->MmeToInputFromId(modelid,SolidearthExternalBarystaticSeaLevelRateEnum, P1Enum);
	}
	
	GetVectorFromInputsx(&geoid_rate,femmodel,SolidearthExternalGeoidRateEnum,VertexSIdEnum);
	GetVectorFromInputsx(&bedrock_rate,femmodel,SolidearthExternalDisplacementUpRateEnum,VertexSIdEnum);
	if(horiz){
		GetVectorFromInputsx(&bedrockeast_rate,femmodel,SolidearthExternalDisplacementEastRateEnum,VertexSIdEnum);
		GetVectorFromInputsx(&bedrocknorth_rate,femmodel,SolidearthExternalDisplacementNorthRateEnum,VertexSIdEnum);
	}

	/*compute: sea level change = initial sea level + (N_gia_rate+N_esa_rate)  * dt + steric_rate + dynamic_rate dt*/
	geoid->AXPY(geoid_rate,dt);
	bedrock->AXPY(bedrock_rate,dt);
	if(horiz){
		bedrockeast->AXPY(bedrockeast_rate,dt);
		bedrocknorth->AXPY(bedrocknorth_rate,dt);
	}

	/*update element inputs:*/
	InputUpdateFromVectorx(femmodel,bedrock,BedEnum,VertexSIdEnum);	
	InputUpdateFromVectorx(femmodel,geoid,SealevelEnum,VertexSIdEnum);	
	if(horiz){
		InputUpdateFromVectorx(femmodel,bedrockeast,BedEastEnum,VertexSIdEnum);	
		InputUpdateFromVectorx(femmodel,bedrocknorth,BedNorthEnum,VertexSIdEnum);	
	}

	/*Free ressources:*/	
	delete bedrock; delete bedrock_rate;
	delete geoid; delete geoid_rate;
	if(horiz){
		delete bedrockeast; delete bedrockeast_rate;
		delete bedrocknorth; delete bedrocknorth_rate;
	}
}
/*}}}*/
void grd_core(FemModel* femmodel){ /*{{{*/

	/*Gravity rotation deformation core GRD: */

	/*variables:*/
	Vector<IssmDouble> *RSLg    = NULL;
	Vector<IssmDouble> *RSLg_barystatic  = NULL; 
	Vector<IssmDouble> *U_grd  = NULL; 
	Vector<IssmDouble> *N_grd  = NULL; 
	Vector<IssmDouble> *U_north_grd   = NULL; 
	Vector<IssmDouble> *U_east_grd    = NULL; 
	Vector<IssmDouble> *bedrock  = NULL; 
	Vector<IssmDouble> *bedrockeast  = NULL; 
	Vector<IssmDouble> *bedrocknorth  = NULL; 
	Vector<IssmDouble> *geoid= NULL; 
	SealevelMasks* masks=NULL;

	/*parameters:*/
	bool iscoupler;
	int  solution_type;
	int  modelid,earthid;
	bool istransientmasstransport;
	int  frequency,count;
	int  horiz;
	IssmDouble dt;
	IssmDouble oceanarea;
	int bp_compute_fingerprints=0;

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing GRD sea level patterns\n");

	/*retrieve more parameters:*/
	femmodel->parameters->FindParam(&iscoupler,TransientIscouplerEnum);
	femmodel->parameters->FindParam(&frequency,SolidearthSettingsRunFrequencyEnum);
	femmodel->parameters->FindParam(&count,SealevelchangeRunCountEnum);
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);

	if(iscoupler){
		femmodel->parameters->FindParam(&modelid,ModelIdEnum);
		femmodel->parameters->FindParam(&earthid,EarthIdEnum);
		femmodel->parameters->FindParam(&istransientmasstransport,TransientIsmasstransportEnum);
	}
	else{
		/* we are here, we are not running in a coupler, so we will indeed compute SLR,
		 * so make sure we are identified as being the Earth.:*/
		modelid=1; earthid=1; 
		/* in addition, if we are running solution_type SealevelchangeSolutionEnum, make sure we 
		 * run, irresepective of the time settings:*/
		count=frequency;
	}

	/*If we are running in coupled mode, the Earth model needs to run its own mass transport (if 
	 * not already done by the mass trasnport module. For ice caps, they rely on the transient mass 
	 * transport module exclusively:*/
	if(iscoupler) if(modelid==earthid) if(!istransientmasstransport) EarthMassTransport(femmodel);

	/*increment counter, or call solution core if count==frequency:*/
	if (count<frequency){
		count++; femmodel->parameters->SetParam(count,SealevelchangeRunCountEnum); 
		return;
	}

	/*call sea-level change sub cores:*/
	if(iscoupler){
		/*transfer cumulated deltathickness forcing from ice caps to earth model: */
		TransferForcing(femmodel,SealevelchangeCumDeltathicknessEnum);

		/*we have accumulated thicknesses, dump them in deltathcikness: */
		if(modelid==earthid)InputDuplicatex(femmodel,SealevelchangeCumDeltathicknessEnum,SurfaceloadIceThicknessChangeEnum);
	}

	/*run cores:*/
	if(modelid==earthid){

		/*call masks core: */
		masks=sealevel_masks(femmodel);

		/*set bottom pressures:*/
		SetBottomPressure(femmodel);

		/*call barystatic core  (generalized eustatic - Farrel and Clark, Eq 4, 1st, 3rd and 4rd terms on the RHS) */
		RSLg_barystatic=sealevelchange_core_barystatic(femmodel,masks,&oceanarea); 

		/*call self attraction and loading module (ocean loading tems  - 2nd and 5th terms on the RHS of Farrel and Clark) */
		RSLg=sealevelchange_core_sal(femmodel,masks,RSLg_barystatic,oceanarea); 

		/*compute bedrock motion and derive geoid: */
		sealevelchange_core_deformation(&N_grd,&U_grd,&U_north_grd,&U_east_grd,femmodel,RSLg,masks);

		/*Update bedrock motion and geoid:*/
		GetVectorFromInputsx(&geoid,femmodel,SealevelEnum,VertexSIdEnum);
		GetVectorFromInputsx(&bedrock,femmodel,BedEnum,VertexSIdEnum);
		if(horiz){
			GetVectorFromInputsx(&bedrockeast,femmodel,BedEastEnum,VertexSIdEnum);
			GetVectorFromInputsx(&bedrocknorth,femmodel,BedNorthEnum,VertexSIdEnum);
		}
		
		geoid->AXPY(N_grd,1);
		bedrock->AXPY(U_grd,1);
		if(horiz){
			bedrockeast->AXPY(U_east_grd,1);
			bedrocknorth->AXPY(U_north_grd,1);
		}

		/*get some of the updates into elements:*/
		InputUpdateFromVectorx(femmodel,U_grd,SealevelUEsaEnum,VertexSIdEnum); 
		InputUpdateFromVectorx(femmodel,N_grd,SealevelNEsaEnum,VertexSIdEnum); 
		InputUpdateFromVectorx(femmodel,RSLg,SealevelRSLEnum,VertexSIdEnum); 
		InputUpdateFromVectorx(femmodel,RSLg_barystatic,SealevelRSLBarystaticEnum,VertexSIdEnum); 
		if (horiz){
			InputUpdateFromVectorx(femmodel,U_north_grd,SealevelUNorthEsaEnum,VertexSIdEnum);	
			InputUpdateFromVectorx(femmodel,U_east_grd,SealevelUEastEsaEnum,VertexSIdEnum);	
		} 
	}

	/*reset counter to 1:*/
	femmodel->parameters->SetParam(1,SealevelchangeRunCountEnum); //reset counter.

	/*free ressources:{{{*/
	delete RSLg;
	delete RSLg_barystatic;
	delete U_grd;
	delete N_grd;
	delete bedrock; 
	delete geoid; 
	if(horiz){
		delete U_north_grd;
		delete U_east_grd;
		delete bedrockeast; 
		delete bedrocknorth; 
	}
	delete masks;
	/*}}}*/

} 
/*}}}*/
void dynstr_core(FemModel* femmodel){ /*{{{*/

	/*variables:*/
	Vector<IssmDouble> *sealevel  = NULL; 
	Vector<IssmDouble> *steric_rate_g  = NULL; 
	Vector<IssmDouble> *dynamic_rate_g = NULL;

	/*parameters: */
	bool isslr=0;
	int  solution_type;
	IssmDouble          dt;
	int  step;
	IssmDouble time;

	IssmDouble cumgmtslr=0;
	IssmDouble cumbslr=0;
	IssmDouble cumgmslr=0;
	IssmDouble gmtslr=0;

	/*Retrieve parameters:*/
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&isslr,TransientIsslrEnum);
	
	/*in case we are running SealevelchangeSolutionEnum, then bypass transient settings:*/
	if(solution_type==SealevelchangeSolutionEnum)isslr=1;

	/*Should we be here?:*/
	if(!isslr)return;

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing steric sea level change\n");

	/*Retrieve sealevel and add steric + dynamic rates:*/
	GetVectorFromInputsx(&sealevel,femmodel,SealevelEnum,VertexSIdEnum);
	GetStericRate(&steric_rate_g,femmodel);
	GetDynamicRate(&dynamic_rate_g,femmodel);

	/*compute: sea level change = initial sea level + (N_gia_rate+N_esa_rate)  * dt + steric_rate + dynamic_rate dt*/
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	sealevel->AXPY(steric_rate_g,dt);
	sealevel->AXPY(dynamic_rate_g,dt);

	/*cumulate thermal steric rate:*/
	femmodel->parameters->FindParam(&cumgmtslr,CumGmtslrEnum); 
	femmodel->parameters->FindParam(&cumbslr,CumBslrEnum); 

	gmtslr=steric_rate_g->Max()*dt;
	cumgmtslr+=gmtslr;
	cumgmslr=cumbslr+cumgmtslr;

	femmodel->parameters->SetParam(cumgmtslr,CumGmtslrEnum);
	femmodel->parameters->SetParam(cumgmslr,CumGmslrEnum);
	
	/*Outputs some metrics:*/
	femmodel->parameters->FindParam(&step,StepEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);

	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,GmtslrEnum,gmtslr,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumGmtslrEnum,cumgmtslr,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumGmslrEnum,cumgmslr,step,time));

	/*update element inputs:*/
	InputUpdateFromVectorx(femmodel,sealevel,SealevelEnum,VertexSIdEnum);	

	/*Free ressources:*/	
	delete sealevel;
	delete steric_rate_g;
	delete dynamic_rate_g;
}
/*}}}*/

//Geometry:
SealevelMasks* sealevel_masks(FemModel* femmodel) {  /*{{{*/

	if(VerboseSolution()) _printf0_("	  computing sea level masks\n");
	
	/*initialize SealevelMasks structure: */
	SealevelMasks* masks=new SealevelMasks(femmodel->elements->Size());

	/*go through elements and fill the masks: */
	for(Object* & object : femmodel->elements->objects){
		Element*   element=xDynamicCast<Element*>(object);
		element->SetSealevelMasks(masks);
	}

	return masks;
}/*}}}*/
void sealevelchange_geometry(FemModel* femmodel) {  /*{{{*/

	/*Geometry core where we compute indices into tables pre computed in the SealevelRiseAnalysis: */

	/*parameters: */
	bool spherical=true;
	IssmDouble *latitude  = NULL;
	IssmDouble *longitude = NULL;
	IssmDouble *radius    = NULL;
	IssmDouble *xx    = NULL;
	IssmDouble *yy    = NULL;
	IssmDouble *zz    = NULL;
	int  horiz;
	bool geometrydone = false;

		
	/*retrieve parameters:*/
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);
	femmodel->parameters->FindParam(&geometrydone,SealevelchangeGeometryDoneEnum);

	if(geometrydone){
		if(VerboseSolution()) _printf0_("	  geometrical offsets have already been computed, skipping \n");
		return; //don't need to run this again.
	}

	/*Verbose: */
	if(VerboseSolution()) _printf0_("	  computing geometrical offsets into precomputed Green tables \n");

	/*first, recover lat,long and radius vectors from vertices: */
	VertexCoordinatesx(&latitude,&longitude,&radius,femmodel->vertices,spherical); 
	if(horiz) VertexCoordinatesx(&xx,&yy,&zz,femmodel->vertices); 


	/*Run sealevel geometry routine in elements:*/
	for(Object* & object : femmodel->elements->objects){
		Element*   element=xDynamicCast<Element*>(object);
		element->SealevelchangeGeometry(latitude,longitude,radius,xx,yy,zz);
	}

	/*Free ressources:*/
	if(horiz){
		xDelete<IssmDouble>(xx);
		xDelete<IssmDouble>(yy);
		xDelete<IssmDouble>(zz);
	}
	xDelete<IssmDouble>(latitude);
	xDelete<IssmDouble>(longitude);
	xDelete<IssmDouble>(radius);

	/*Record the fact that we ran this module already: */
	femmodel->parameters->SetParam(true,SealevelchangeGeometryDoneEnum); 


}/*}}}*/

//GRD: 
Vector<IssmDouble>* sealevelchange_core_barystatic(FemModel* femmodel,SealevelMasks* masks, IssmDouble* poceanarea){ /*{{{*/

	/*Barystatic core of the SLR solution (terms that are constant with respect to sea-level)*/

	Vector<IssmDouble> *RSLgi    = NULL;
	IssmDouble          RSLgi_oceanaverage   = 0;

	/*parameters: */
	int  gsize;
	IssmDouble oceanarea;
	int        step;
	IssmDouble time;

	/*barystatic contribution:*/
	IssmDouble bslr;
	IssmDouble bslrice;
	IssmDouble* bslrice_partition=NULL;
	IssmDouble bslrhydro;
	IssmDouble* bslrhydro_partition=NULL;
	IssmDouble cumbslr;
	IssmDouble cumbslrice;
	IssmDouble cumbslrhydro;
	IssmDouble* cumbslrice_partition=NULL;
	int npartice;
	IssmDouble* cumbslrhydro_partition=NULL;
	int nparthydro;
	
	if(VerboseSolution()) _printf0_("	  computing bslr components on ice\n");

	/*Figure out size of g-set deflection vector and allocate solution vector: */
	gsize = femmodel->nodes->NumberOfDofs(GsetEnum);

	/*some parameters:*/
	femmodel->parameters->FindParam(&step,StepEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);
	femmodel->parameters->FindParam(&cumbslr,CumBslrEnum);
	femmodel->parameters->FindParam(&cumbslrice,CumBslrIceEnum);
	femmodel->parameters->FindParam(&cumbslrhydro,CumBslrHydroEnum);
	femmodel->parameters->FindParam(&npartice,SolidearthNpartIceEnum);
	femmodel->parameters->FindParam(&nparthydro,SolidearthNpartHydroEnum);
	if(npartice) femmodel->parameters->FindParam(&cumbslrice_partition,&npartice,NULL,CumBslrIcePartitionEnum);
	if(nparthydro) femmodel->parameters->FindParam(&cumbslrhydro_partition,&nparthydro,NULL,CumBslrHydroPartitionEnum);

	/*Initialize:*/
	RSLgi = new Vector<IssmDouble>(gsize);

	/*call the bslr main module: */
	femmodel->SealevelchangeBarystatic(RSLgi,&oceanarea,&bslr, &bslrice, &bslrhydro, &bslrice_partition, &bslrhydro_partition,masks); //this computes 

	/*we need to average RSLgi over the ocean: RHS term  4 in Eq.4 of Farrel and clarke. Only the elements can do that: */
	RSLgi_oceanaverage=femmodel->SealevelchangeOceanAverage(RSLgi,masks, oceanarea);

	/*RSLg is the sum of the pure bslr component (term 3) and the contribution from the perturbation to the graviation potential due to the 
	 * presence of ice (terms 1 and 4 in Eq.4 of Farrel and Clarke):*/
	RSLgi->Shift(-bslr-RSLgi_oceanaverage);

	/*save bslr and cumulated bslr value for results:{{{ */
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,BslrEnum,-bslr,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,BslrIceEnum,-bslrice,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,BslrHydroEnum,-bslrhydro,step,time));

	//cumulative barystatic contribution: 
	cumbslr=cumbslr-bslr;
	cumbslrice=cumbslrice-bslrice;
	cumbslrhydro=cumbslrhydro-bslrhydro;

	femmodel->parameters->SetParam(cumbslr,CumBslrEnum);
	femmodel->parameters->SetParam(cumbslrice,CumBslrIceEnum);
	femmodel->parameters->SetParam(cumbslrhydro,CumBslrHydroEnum);

	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumBslrEnum,cumbslr,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumBslrIceEnum,cumbslrice,step,time));
	femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,CumBslrHydroEnum,cumbslrhydro,step,time));

	//cumulative barystatic contributions by partition:
	if(npartice){
		for(int i=0;i<npartice;i++) cumbslrice_partition[i] -= bslrice_partition[i];
		femmodel->parameters->SetParam(cumbslrice_partition,npartice,1,CumBslrIcePartitionEnum);
		femmodel->results->AddResult(new GenericExternalResult<IssmDouble*>(femmodel->results->Size()+1,CumBslrIcePartitionEnum,cumbslrice_partition,npartice,1,step,time));
	}

	if(nparthydro){
		for(int i=0;i<nparthydro;i++) cumbslrhydro_partition[i] -= bslrhydro_partition[i];
		femmodel->parameters->SetParam(cumbslrhydro_partition,nparthydro,1,CumBslrHydroPartitionEnum);
		femmodel->results->AddResult(new GenericExternalResult<IssmDouble*>(femmodel->results->Size()+1,CumBslrHydroPartitionEnum,cumbslrhydro_partition,nparthydro,1,step,time));
	}
	/*}}}*/
	
	/*Assign output pointers and return: */
	*poceanarea=oceanarea;
	return RSLgi;
}/*}}}*/
Vector<IssmDouble>* sealevelchange_core_sal(FemModel* femmodel, SealevelMasks* masks, Vector<IssmDouble>* RSLg_barystatic,IssmDouble oceanarea){ /*{{{*/

	/*this core computes the contributions from Eq.4 of Farrel and Clarke, rhs terms 2 and 5.
	  sal core of the SLR solution */

	Vector<IssmDouble> *RSLg    = NULL;
	Vector<IssmDouble> *RSLg_old    = NULL;

	Vector<IssmDouble> *RSLgo    = NULL; //ocean convolution of the perturbation to gravity potential.
	Vector<IssmDouble> *RSLgo_rot= NULL; // rotational feedback 
	IssmDouble          RSLgo_oceanaverage = 0;  //average of RSLgo over the ocean.

	/*parameters: */
	int count;
	bool save_results;
	int  gsize;
	bool converged=true;
	bool rotation=true;
	bool verboseconvolution=true;
	int max_nonlinear_iterations;
	IssmDouble           eps_rel;
	IssmDouble           eps_abs;
	IssmDouble			Ixz, Iyz, Izz; 
	
	if(VerboseSolution()) _printf0_("	  converging on ocean components\n");

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&max_nonlinear_iterations,SolidearthSettingsMaxiterEnum);
	femmodel->parameters->FindParam(&eps_rel,SolidearthSettingsReltolEnum);
	femmodel->parameters->FindParam(&eps_abs,SolidearthSettingsAbstolEnum);

	/*computational flag: */
	femmodel->parameters->FindParam(&rotation,SolidearthSettingsRotationEnum);

	/*Figure out size of g-set deflection vector and allocate solution vector: */
	gsize = femmodel->nodes->NumberOfDofs(GsetEnum);

	/*Initialize:*/
	RSLg = new Vector<IssmDouble>(gsize);
	RSLg->Assemble();
	RSLg_barystatic->Copy(RSLg);  //first initialize RSLg with the barystatic component computed in sealevelchange_core_barystatic.

	RSLg_old = new Vector<IssmDouble>(gsize);
	RSLg_old->Assemble();

	count=1;
	converged=false;

	/*Start loop: */
	for(;;){

		//save pointer to old sea level
		delete RSLg_old; RSLg_old=RSLg; 

		/*Initialize solution vector: */
		RSLg  = new Vector<IssmDouble>(gsize); RSLg->Assemble();
		RSLgo = new Vector<IssmDouble>(gsize); RSLgo->Assemble();

		/*call the sal module: */
		femmodel->SealevelchangeSal(RSLgo, RSLg_old,  masks, verboseconvolution);

		/*assemble solution vector: */
		RSLgo->Assemble(); 

		if(rotation){

			/*call rotational feedback  module: */
			RSLgo_rot = new Vector<IssmDouble>(gsize); RSLgo_rot->Assemble();
			femmodel->SealevelchangeRotationalFeedback(RSLgo_rot,RSLg_old,&Ixz,&Iyz,&Izz, masks); 
			RSLgo_rot->Assemble(); 

			/*save changes in inertia tensor as results: */
			femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,SealevelInertiaTensorXZEnum,Ixz));
			femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,SealevelInertiaTensorYZEnum,Iyz));
			femmodel->results->AddResult(new GenericExternalResult<IssmDouble>(femmodel->results->Size()+1,SealevelInertiaTensorZZEnum,Izz));

			RSLgo->AXPY(RSLgo_rot,1); 
		}

		/*we need to average RSLgo over the ocean: RHS term  5 in Eq.4 of Farrel and clarke. Only the elements can do that: */
		RSLgo_oceanaverage=femmodel->SealevelchangeOceanAverage(RSLgo,masks, oceanarea);

		/*RSLg is the sum of the barystatic term, and the ocean terms: */
		RSLg_barystatic->Copy(RSLg); RSLg->AXPY(RSLgo,1); 
		RSLg->Shift(-RSLgo_oceanaverage);

		/*convergence criterion:*/
		slrconvergence(&converged,RSLg,RSLg_old,eps_rel,eps_abs);
		
	
		/*free ressources: */
		delete RSLgo;
		delete RSLgo_rot;

		/*Increase count: */
		count++;
		if(converged==true){
			break;
		}
		if(count>=max_nonlinear_iterations){
			_printf0_("   maximum number of nonlinear iterations (" << max_nonlinear_iterations << ") exceeded\n"); 
			converged=true;
			break;
		}	

		/*some minor verbosing adjustment:*/
		if(count>1)verboseconvolution=false;

	}
	if(VerboseConvergence()) _printf0_("\n              total number of iterations: " << count-1 << "\n");
	
	
	delete RSLg_old;

	return RSLg;
} /*}}}*/
void sealevelchange_core_deformation(Vector<IssmDouble>** pN_grd, Vector<IssmDouble>** pU_grd, Vector<IssmDouble>** pU_north_grd,Vector<IssmDouble>** pU_east_grd,FemModel* femmodel,Vector<IssmDouble>* RSLg, SealevelMasks* masks){ /*{{{*/

	Vector<IssmDouble> *U_grd  = NULL; 
	Vector<IssmDouble> *N_grd  = NULL; 
	Vector<IssmDouble> *U_north_grd   = NULL; 
	Vector<IssmDouble> *U_east_grd    = NULL; 

	/*parameters: */
	int  gsize;
	bool spherical=true;

	IssmDouble          *latitude   = NULL;
	IssmDouble          *longitude  = NULL;
	IssmDouble          *radius     = NULL;
	IssmDouble          *xx     = NULL;
	IssmDouble          *yy     = NULL;
	IssmDouble          *zz     = NULL;
	int  horiz;
	
	if(VerboseSolution()) _printf0_("	  computing vertical and horizontal geodetic signatures\n");

	/*retrieve some parameters:*/
	femmodel->parameters->FindParam(&horiz,SolidearthSettingsHorizEnum);

	/*find size of vectors:*/
	gsize      = femmodel->nodes->NumberOfDofs(GsetEnum);

	/*intialize vectors:*/
	U_grd = new Vector<IssmDouble>(gsize);
	N_grd = new Vector<IssmDouble>(gsize);
	if (horiz){
		U_north_grd = new Vector<IssmDouble>(gsize);
		U_east_grd = new Vector<IssmDouble>(gsize);
	}
	
	/*retrieve geometric information: */
	VertexCoordinatesx(&latitude,&longitude,&radius,femmodel->vertices,spherical); 
	VertexCoordinatesx(&xx,&yy,&zz,femmodel->vertices); 

	/*call the elastic main modlule:*/ 
	femmodel->SealevelchangeDeformation(N_grd, U_grd,U_north_grd,U_east_grd,RSLg, masks);

	/*Assign output pointers:*/
	*pU_grd=U_grd;
	*pN_grd=N_grd;
	if(horiz){
		*pU_east_grd=U_east_grd;
		*pU_north_grd=U_north_grd;
	}

	/*Free ressources: */
	xDelete<IssmDouble>(longitude);
	xDelete<IssmDouble>(latitude);
	xDelete<IssmDouble>(xx);
	xDelete<IssmDouble>(yy);
	xDelete<IssmDouble>(zz);
	xDelete<IssmDouble>(radius);
}
/*}}}*/

//Ocean:
void GetDynamicRate(Vector<IssmDouble> ** pdynamic_rate_g, FemModel* femmodel){ /*{{{*/

	int dslmodel=-1;
	IssmDouble time;

	/*variables:*/
	Vector<IssmDouble> *dynamic_rate_g  = NULL; 

	/*Update steric rates before retrieving them on Vertex SID set:*/
	femmodel->parameters->FindParam(&dslmodel,DslModelEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);
	if(dslmodel==1){
		TransientInput* transient_input  = femmodel->inputs->GetTransientInput(DslSeaSurfaceHeightChangeAboveGeoidEnum);
		TriaInput* tria_input=transient_input->GetTriaInput(time);
		Input* tria_input_copy=tria_input->copy();
		tria_input_copy->ChangeEnum(DslDynamicRateEnum);
		femmodel->inputs->AddInput(tria_input_copy);
	}
	else if(dslmodel==2){
	
		IssmDouble modelid;
		
		/*Recover modelid:*/
		femmodel->parameters->FindParam(&modelid,DslModelidEnum);
		modelid--; //from matlab. 
		
		/*find the DslSeaSurfaceHeightChangeAboveGeoidEnum dataset of transient inputs:*/
		DatasetInput* dataset_input=femmodel->inputs->GetDatasetInput(DslSeaSurfaceHeightChangeAboveGeoidEnum);
		
		/*Go find the modelid'th transient input:*/
		TriaInput* tria_input=dataset_input->GetTriaInputByOffset(reCast<int, IssmDouble>(modelid));
		
		/*Plug into DslDynamicRate input: */
		Input* tria_input_copy=tria_input->copy();
		tria_input_copy->ChangeEnum(DslDynamicRateEnum);
		femmodel->inputs->AddInput(tria_input_copy);
	}
	else _error_("not implemented yet");

	GetVectorFromInputsx(&dynamic_rate_g,femmodel,DslDynamicRateEnum,VertexSIdEnum);
	*pdynamic_rate_g=dynamic_rate_g;
}
/*}}}*/
void GetStericRate(Vector<IssmDouble> ** psteric_rate_g, FemModel* femmodel){ /*{{{*/

	int dslmodel=-1;
	IssmDouble time;

	/*variables:*/
	Vector<IssmDouble> *steric_rate_g  = NULL; 

	/*Update steric rates before retrieving them on Vertex SID set:*/
	femmodel->parameters->FindParam(&dslmodel,DslModelEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);
	if(dslmodel==1){
		TransientInput* transient_input  = femmodel->inputs->GetTransientInput(DslGlobalAverageThermostericSeaLevelChangeEnum);
		TriaInput* tria_input=transient_input->GetTriaInput(time);
		Input* tria_input_copy=tria_input->copy();
		tria_input_copy->ChangeEnum(DslStericRateEnum);
		femmodel->inputs->AddInput(tria_input_copy);
	}
	else if (dslmodel==2){
		IssmDouble modelid;
		
		/*Recover modelid:*/
		femmodel->parameters->FindParam(&modelid,DslModelidEnum);
		
		modelid--; //from matlab. 
		
		/*find the DslGlobalAverageThermostericSeaLevelChangeEnum dataset of transient inputs:*/
		DatasetInput* dataset_input=femmodel->inputs->GetDatasetInput(DslGlobalAverageThermostericSeaLevelChangeEnum);
		
		/*Go find the modelid'th transient input:*/
		TriaInput* tria_input=dataset_input->GetTriaInputByOffset(reCast<int, IssmDouble>(modelid));
		
		/*Plug into DslStericRate input: */
		Input* tria_input_copy=tria_input->copy();
		tria_input_copy->ChangeEnum(DslStericRateEnum);
		femmodel->inputs->AddInput(tria_input_copy);
	}
	else _error_("not implemented yet");

	GetVectorFromInputsx(&steric_rate_g,femmodel,DslStericRateEnum,VertexSIdEnum);
	*psteric_rate_g=steric_rate_g;
}
/*}}}*/
void SetBottomPressure(FemModel* femmodel){ /*{{{*/

	int dslmodel=-1;
	int type;

	/*If we are running a dsl mme model, we need to update the bottom pressure first: */
	femmodel->parameters->FindParam(&dslmodel,DslModelEnum);
	if(dslmodel==2){
	
		IssmDouble modelid;
		/*figure out the type of DslSeaWaterPressureChangeAtSeaFloor input:*/
		type=femmodel->inputs->GetInputObjectEnum(DslSeaWaterPressureChangeAtSeaFloorEnum);

		if(type==DatasetInputEnum){
		
			/*find the DslSeaWaterPressureChangeAtSeaFloor dataset of transient inputs:*/
			DatasetInput* dataset_input=femmodel->inputs->GetDatasetInput(DslSeaWaterPressureChangeAtSeaFloorEnum);
		
			/*Recover modelid:*/
			femmodel->parameters->FindParam(&modelid,DslModelidEnum);
			modelid--; //from matlab. 
		
			/*Go find the modelid'th transient input:*/
			TransientInput* transient_input=dataset_input->GetTransientInputByOffset(reCast<int, IssmDouble>(modelid));

			/*Plug into DslSeaWaterPressureChangeAtSeaFloor input: */
			Input* transient_input_copy=transient_input->copy();
			transient_input_copy->ChangeEnum(DslSeaWaterPressureChangeAtSeaFloorEnum);
			femmodel->inputs->AddInput(transient_input_copy);
		}
	}
}
/*}}}*/

/*Support routines:*/
void TransferForcing(FemModel* femmodel,int forcingenum){ /*{{{*/

	/*forcing being transferred from models to earth: */
	IssmDouble** forcings=NULL;
	IssmDouble*  forcing=NULL; 
	Vector<IssmDouble>* forcingglobal=NULL; 
	int*         nvs=NULL;

	/*transition vectors:*/
	IssmDouble** transitions=NULL;
	int          ntransitions; 
	int*         transitions_m=NULL;
	int*         transitions_n=NULL;
	int          nv;

	/*communicators:*/
	ISSM_MPI_Comm tocomm;
	ISSM_MPI_Comm* fromcomms=NULL;
	ISSM_MPI_Status status;
	int         my_rank;
	int         modelid,earthid;
	int         nummodels;

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&modelid,ModelIdEnum);
	femmodel->parameters->FindParam(&earthid,EarthIdEnum);
	femmodel->parameters->FindParam(&nummodels,NumModelsEnum);
	my_rank=IssmComm::GetRank();

	/*retrieve the inter communicators that will be used to send data from each ice cap to the earth: */
	if(modelid==earthid){
		GenericParam<ISSM_MPI_Comm*>* parcoms = dynamic_cast<GenericParam<ISSM_MPI_Comm*>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcoms)_error_("TransferForcing error message: could not find IcecapToEarthComm communicator");
		fromcomms=parcoms->GetParameterValue();
	}
	else {
		GenericParam<ISSM_MPI_Comm>* parcom = dynamic_cast<GenericParam<ISSM_MPI_Comm>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcom)_error_("TransferForcing error message: could not find IcecapToEarthComm communicator");
		tocomm=parcom->GetParameterValue();
	}

	/*For each icecap, retrieve the forcing vector that will be sent to the earth model: */
	if(modelid!=earthid){
		nv=femmodel->vertices->NumberOfVertices();
		GetVectorFromInputsx(&forcing,femmodel,forcingenum,VertexSIdEnum);
	}

	/*Send the forcing to the earth model:{{{*/
	if(my_rank==0){
		if(modelid==earthid){
			forcings=xNew<IssmDouble*>(nummodels-1);
			nvs=xNew<int>(nummodels-1);
			for(int i=0;i<earthid;i++){
				ISSM_MPI_Recv(nvs+i, 1, ISSM_MPI_INT, 0,i, fromcomms[i], &status);
				forcings[i]=xNew<IssmDouble>(nvs[i]);
				ISSM_MPI_Recv(forcings[i], nvs[i], ISSM_MPI_DOUBLE, 0,i, fromcomms[i], &status);
			}

		}
		else{
			ISSM_MPI_Send(&nv, 1, ISSM_MPI_INT, 0, modelid, tocomm);
			ISSM_MPI_Send(forcing, nv, ISSM_MPI_DOUBLE, 0, modelid, tocomm);
		}
	}
	/*}}}*/

	/*On the earth model, consolidate all the forcings into one, and update the elements dataset accordingly: {{{*/
	if(modelid==earthid){

		/*Out of all the delta thicknesses, build one delta thickness vector made of all the ice cap contributions. 
		 *First, build the global delta thickness vector in the earth model: */
		nv=femmodel->vertices->NumberOfVertices();
		GetVectorFromInputsx(&forcingglobal,femmodel,forcingenum,VertexSIdEnum);

		/*Retrieve transition vectors, used to plug from each ice cap into the global forcing:*/
		femmodel->parameters->FindParam(&transitions,&ntransitions,&transitions_m,&transitions_n,SealevelchangeTransitionsEnum);

		if(ntransitions!=earthid)_error_("TransferForcing error message: number of transition vectors is not equal to the number of icecaps!");

		/*Go through all the delta thicknesses coming from each ice cap: */
		if(my_rank==0){
			for(int i=0;i<earthid;i++){

				IssmDouble* forcingfromcap= forcings[i]; //careful, this only exists on rank 0 of the earth model!
				IssmDouble* transition=transitions[i];
				int         M=transitions_m[i];

				/*build index to plug values: */
				int*        index=xNew<int>(M); for(int i=0;i<M;i++)index[i]=reCast<int>(transition[i])-1; //matlab indexing!

				/*We are going to plug this vector into the earth model, at the right vertices corresponding to this particular 
				 * ice cap: */
				forcingglobal->SetValues(M,index,forcingfromcap,ADD_VAL);
				xDelete<int>(index);
			}
		}

		/*Assemble vector:*/
		forcingglobal->Assemble();

		/*Plug into elements:*/
		InputUpdateFromVectorx(femmodel,forcingglobal,forcingenum,VertexSIdEnum);
	} 
	/*}}}*/

	/*Free ressources:{{{*/
	if(forcings){
		for(int i=0;i<nummodels-1;i++){
			IssmDouble* temp=forcings[i]; xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(forcings);
	}
	if(forcing)xDelete<IssmDouble>(forcing);
	if(forcingglobal)delete forcingglobal;
	if(transitions){
		for(int i=0;i<earthid;i++){
			IssmDouble* temp=transitions[i];
			xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(transitions);
		xDelete<int>(transitions_m);
		xDelete<int>(transitions_n);
	}
	if(nvs)xDelete<int>(nvs);
	/*}}}*/

} /*}}}*/
void TransferSealevel(FemModel* femmodel,int forcingenum){ /*{{{*/

	/*forcing being transferred from earth to ice caps: */
	IssmDouble*  forcing=NULL; 
	IssmDouble*  forcingglobal=NULL; 

	/*transition vectors:*/
	IssmDouble** transitions=NULL;
	int          ntransitions; 
	int*         transitions_m=NULL;
	int*         transitions_n=NULL;
	int          nv;

	/*communicators:*/
	ISSM_MPI_Comm fromcomm;
	ISSM_MPI_Comm* tocomms=NULL;
	ISSM_MPI_Status status;
	int         my_rank;
	int         modelid,earthid;
	int         nummodels;
	int         numcoms;

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&modelid,ModelIdEnum);
	femmodel->parameters->FindParam(&earthid,EarthIdEnum);
	femmodel->parameters->FindParam(&nummodels,NumModelsEnum);
	my_rank=IssmComm::GetRank();

	/*retrieve the inter communicators that will be used to send data from earth to ice caps:*/
	if(modelid==earthid){
		GenericParam<ISSM_MPI_Comm*>* parcoms = dynamic_cast<GenericParam<ISSM_MPI_Comm*>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcoms)_error_("TransferSealevel error message: could not find IcecapToEarthComm communicator");
		tocomms=parcoms->GetParameterValue();
		//femmodel->parameters->FindParam((int**)(&tocomms),&numcoms,IcecapToEarthCommEnum);
	}
	else{
		GenericParam<ISSM_MPI_Comm>* parcom = dynamic_cast<GenericParam<ISSM_MPI_Comm>*>(femmodel->parameters->FindParamObject(IcecapToEarthCommEnum));
		if(!parcom)_error_("TransferSealevel error message: could not find IcecapToEarthComm communicator");
		fromcomm=parcom->GetParameterValue();
		//femmodel->parameters->FindParam((int*)(&fromcomm), IcecapToEarthCommEnum);
	}

	/*Retrieve sea-level on earth model: */
	if(modelid==earthid){
		nv=femmodel->vertices->NumberOfVertices();
		GetVectorFromInputsx(&forcingglobal,femmodel,forcingenum,VertexSIdEnum);
	}

	/*Send the forcing to the ice caps:{{{*/
	if(my_rank==0){

		if(modelid==earthid){

			/*Retrieve transition vectors, used to figure out global forcing contribution to each ice cap's own elements: */
			femmodel->parameters->FindParam(&transitions,&ntransitions,&transitions_m,&transitions_n,SealevelchangeTransitionsEnum);

			if(ntransitions!=earthid)_error_("TransferSeaLevel error message: number of transition vectors is not equal to the number of icecaps!");

			for(int i=0;i<earthid;i++){
				nv=transitions_m[i];
				forcing=xNew<IssmDouble>(nv);
				IssmDouble* transition=transitions[i];
				for(int j=0;j<nv;j++){
					forcing[j]=forcingglobal[reCast<int>(transition[j])-1];
				}
				ISSM_MPI_Send(&nv, 1, ISSM_MPI_INT, 0, i, tocomms[i]);
				ISSM_MPI_Send(forcing, nv, ISSM_MPI_DOUBLE, 0, i, tocomms[i]);
			}
		}
		else{
			ISSM_MPI_Recv(&nv, 1, ISSM_MPI_INT, 0, modelid, fromcomm, &status);
			forcing=xNew<IssmDouble>(nv);
			ISSM_MPI_Recv(forcing, nv, ISSM_MPI_DOUBLE, 0, modelid, fromcomm, &status);
		}
	}
	/*}}}*/

	/*On each ice cap, spread the forcing across cpus, and update the elements dataset accordingly: {{{*/
	if(modelid!=earthid){

		ISSM_MPI_Bcast(&nv,1,ISSM_MPI_INT,0,IssmComm::GetComm());
		if(my_rank!=0)forcing=xNew<IssmDouble>(nv);
		ISSM_MPI_Bcast(forcing,nv,ISSM_MPI_DOUBLE,0,IssmComm::GetComm());

		/*Plug into elements:*/
		InputUpdateFromVectorx(femmodel,forcing,forcingenum,VertexSIdEnum);
	} 
	/*}}}*/

	/*Free ressources:{{{*/
	if(forcingglobal)xDelete<IssmDouble>(forcingglobal);
	if(forcing)xDelete<IssmDouble>(forcing);
	if(transitions){
		for(int i=0;i<ntransitions;i++){
			IssmDouble* temp=transitions[i];
			xDelete<IssmDouble>(temp);
		}
		xDelete<IssmDouble*>(transitions);
		xDelete<int>(transitions_m);
		xDelete<int>(transitions_n);
	}
	/*}}}*/

} /*}}}*/
void EarthMassTransport(FemModel* femmodel){ /*{{{*/

	IssmDouble time,dt;
	Vector<IssmDouble> *oldthickness    = NULL;
	Vector<IssmDouble> *newthickness    = NULL;
	Vector<IssmDouble> *deltathickness    = NULL;
	Vector<IssmDouble> *cumdeltathickness    = NULL;
	int nv;

	if(VerboseSolution()) _printf0_("              computing earth mass transport\n");

	/*This module has to be recoded! We do not grab spc thicknesses from an earth model 
	 * anymore! Thicknesses come from a mass transport module applied to each basin! 
	 * Commeting out for now:*/
	_error_("EarthMassTransport error message: not supported anymore!");

	/*This mass transport module for the Earth is because we might have thickness variations as spcs 
	 * specified in the md.slr class, outside of what we will get from the icecaps. That's why we get t
	 * the thickness variations from SealevelchangeSpcthicknessEnum.*/

	/*No mass transport module was called, so we are just going to retrieve the geometry thickness 
	 * at this time step, at prior time step, and plug the difference as deltathickness: */
	/*femmodel->parameters->FindParam(&time,TimeEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	nv=femmodel->vertices->NumberOfVertices();

	GetVectorFromInputsx(&newthickness,femmodel,SealevelchangeSpcthicknessEnum,VertexSIdEnum);
	GetVectorFromInputsx(&oldthickness,femmodel,SealevelchangeSpcthicknessEnum,VertexSIdEnum,time-dt);*/

	/*compute deltathickness: */
	/*deltathickness = new Vector<IssmDouble>(nv); 
	newthickness->Copy(deltathickness); deltathickness->AXPY(oldthickness,-1); */

	/*plug into elements:*/
//	InputUpdateFromVectorx(femmodel,deltathickness,SurfaceloadIceThicknessChangeEnum,VertexSIdEnum);

	/*add to cumulated delta thickness: */
	/*GetVectorFromInputsx(&cumdeltathickness,femmodel,SealevelchangeCumDeltathicknessEnum,VertexSIdEnum);
	cumdeltathickness->AXPY(deltathickness,1); 
	InputUpdateFromVectorx(femmodel,cumdeltathickness,SealevelchangeCumDeltathicknessEnum,VertexSIdEnum);*/

	/*free ressources:*/
	/*delete oldthickness;
	delete newthickness;
	delete deltathickness;
	delete cumdeltathickness;*/

} /*}}}*/
void slrconvergence(bool* pconverged, Vector<IssmDouble>* RSLg,Vector<IssmDouble>* RSLg_old,IssmDouble eps_rel,IssmDouble eps_abs){ /*{{{*/

	bool converged=true;
	IssmDouble ndS,nS; 
	Vector<IssmDouble> *dRSLg    = NULL;

	//compute norm(du) and norm(u) if requested
	dRSLg=RSLg_old->Duplicate(); RSLg_old->Copy(dRSLg); dRSLg->AYPX(RSLg,-1.0);
	ndS=dRSLg->Norm(NORM_TWO); 

	if (xIsNan<IssmDouble>(ndS)) _error_("convergence criterion is NaN!");

	if(!xIsNan<IssmDouble>(eps_rel)){
		nS=RSLg_old->Norm(NORM_TWO);
		if (xIsNan<IssmDouble>(nS)) _error_("convergence criterion is NaN!");
	}

	//clean up
	delete dRSLg;

	//print
	if(!xIsNan<IssmDouble>(eps_rel)){
		if((ndS/nS)<eps_rel){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "              convergence criterion: norm(dS)/norm(S)" << ndS/nS*100 << " < " << eps_rel*100 << " %\n");
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "              convergence criterion: norm(dS)/norm(S)" << ndS/nS*100 << " > " << eps_rel*100 << " %\n");
			converged=false;
		}
	}
	if(!xIsNan<IssmDouble>(eps_abs)){
		if(ndS<eps_abs){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "              convergence criterion: norm(dS)" << ndS << " < " << eps_abs << " \n");
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "              convergence criterion: norm(dS)" << ndS << " > " << eps_abs << " \n");
			converged=false;
		}
	}

	/*assign output*/
	*pconverged=converged;

} /*}}}*/
