clear all;
addpath('../Data','../Functions');

steps=[7];

if any(steps==1) %{{{
	disp('   Step 1: Global mesh creation');

	numrefine=1;
	resolution=300*1e3;			% inital resolution [m]
	radius = 6.371012*10^6;		% mean radius of Earth, m
	mindistance_coast=150*1e3;	% coastal resolution [m]
	mindistance_land=300*1e3;	% resolution on the continents [m]
	maxdistance=600*1e3;		% max element size (on mid-oceans) [m]

	md=model;
	md.mesh=gmshplanet('radius',radius*1e-3,'resolution',resolution*1e-3); % attributes in [km]

	for i=1:numrefine,

		ocean_mask=gmtmask(md.mesh.lat,md.mesh.long);

		distance=zeros(md.mesh.numberofvertices,1);

		pos=find(~ocean_mask);	coaste.lat=md.mesh.lat(pos);	coaste.long=md.mesh.long(pos); 
		pos=find(ocean_mask);	coasto.lat=md.mesh.lat(pos);	coasto.long=md.mesh.long(pos); 

		for j=1:md.mesh.numberofvertices
			phi1=md.mesh.lat(j)/180*pi; lambda1=md.mesh.long(j)/180*pi;
			if ocean_mask(j),
				phi2=coaste.lat/180*pi; lambda2=coaste.long/180*pi;
				deltaphi=abs(phi2-phi1); deltalambda=abs(lambda2-lambda1);
				d=radius*2*asin(sqrt(sin(deltaphi/2).^2+cos(phi1).*cos(phi2).*sin(deltalambda/2).^2));
			else
				phi2=coasto.lat/180*pi; lambda2=coasto.long/180*pi;
				deltaphi=abs(phi2-phi1); deltalambda=abs(lambda2-lambda1);
				d=radius*2*asin(sqrt(sin(deltaphi/2).^2+cos(phi1).*cos(phi2).*sin(deltalambda/2).^2));
			end
			distance(j)=min(d);
		end
		pos=find(distance<mindistance_coast); distance(pos)=mindistance_coast;
		
		pos2=find(ocean_mask~=1 & distance>mindistance_land);
		distance(pos2)=mindistance_land;

		dist=min(maxdistance,distance);
		md.mesh=gmshplanet('radius',radius*1e-3,'resolution',resolution*1e-3,'refine',md.mesh,'refinemetric',dist);
	end

	ocean_mask=gmtmask(md.mesh.lat,md.mesh.long);
	pos = find(ocean_mask==0); 
	md.mask.ocean_levelset=-ones(md.mesh.numberofvertices,1); 
	md.mask.ocean_levelset(pos)=1; 

	save ./Models/SlrGRACE_Mesh md;

	plotmodel (md,'data',md.mask.ocean_levelset,'edgecolor','k');
end %}}}

if any(steps==2) %{{{
	disp('   Step 2: Define loads in meters of ice height equivalent');
	md = loadmodel('./Models/SlrGRACE_Mesh');

	year_month = 2007+15/365;
	time_range = [year_month year_month];

	water_load = grace(md.mesh.elements,md.mesh.lat,md.mesh.long,time_range(1),time_range(2));

	md.solidearth.surfaceload.icethicknesschange = water_load*md.materials.rho_freshwater/md.materials.rho_ice;

	save ./Models/SlrGRACE_Loads md;

	plotmodel (md,'data',md.solidearth.surfaceload.icethicknesschange,'view',[90 -90],'caxis',[-.1 .1],'title','Ice height equivalent [m]');
end %}}}

if any(steps==3) %{{{
	disp('   Step 3: Parameterization');
	md = loadmodel('./Models/SlrGRACE_Loads');

	md.solidearth.lovenumbers = lovenumbers('maxdeg',10000);

	md.mask.ice_levelset=-md.mask.ocean_levelset; 

	md.solidearth.initialsealevel=zeros(md.mesh.numberofvertices,1);
	md.dsl.global_average_thermosteric_sea_level_change=[0;0];
	md.dsl.sea_surface_height_change_above_geoid=zeros(md.mesh.numberofvertices+1,1);
	md.dsl.sea_water_pressure_change_at_sea_floor=zeros(md.mesh.numberofvertices+1,1);

	md.solidearth.settings.ocean_area_scaling=1;

	% arbitary to pass consistency check. 
	md.geometry.bed=-ones(md.mesh.numberofvertices,1);
	md.geometry.surface=ones(md.mesh.numberofvertices,1);
	md.geometry.base=md.geometry.bed; 
	md.geometry.thickness=md.geometry.surface-md.geometry.base; 

	md.initialization.temperature=273.25*ones(md.mesh.numberofvertices,1);
	md.materials.rheology_B=paterson(md.initialization.temperature);
	md.materials.rheology_n=3*ones(md.mesh.numberofelements,1);

	md.miscellaneous.name='SlrGRACE';

	save ./Models/SlrGRACE_Parameterization md;
end %}}}

if any(steps==4) %{{{
	disp('   Step 4: Solve Slr solver');
	md = loadmodel('./Models/SlrGRACE_Parameterization');

	md.solidearth.settings.rigid=1;
	md.solidearth.settings.elastic=1;
	md.solidearth.settings.rotation=1;

	md.solidearth.requested_outputs = {'Sealevel','SealevelRSL'};

	md.cluster=generic('name',oshostname(),'np',3);
	md.verbose=verbose('111111111');

	md=solve(md,'Slr');

	save ./Models/SlrGRACE_Solution md;
end %}}}

if any(steps==5) %{{{
	disp('   Step 5: Plot solutions');
	md = loadmodel('./Models/SlrGRACE_Solution');

	sol1 = md.solidearth.surfaceload.icethicknesschange*100;						% [cm]
	sol2 = md.results.SealevelriseSolution.SealevelRSL*1000;	% [mm]

	sol_name={'Change in water equivalent height [cm]', 'Relative sea-level [mm]'};
	fig_name={'Fig_dH.pdf','Fig_slr.pdf'};

	res = 1.0;

	[lat_grid, lon_grid] = meshgrid(linspace(-90,90,180/res), linspace(-180,180,360/res));
	sol_grid = zeros(size(lat_grid));

	for kk=1:2
		sol=eval(sprintf('sol%d',kk));

		if length(sol)==md.mesh.numberofelements
			for jj=1:md.mesh.numberofelements
				ii=(jj-1)*3;
				pp(ii+1:ii+3)=md.mesh.elements(jj,:);
			end
			for jj=1:md.mesh.numberofvertices
				pos=ceil(find(pp==jj)/3);
				temp(jj)=mean(sol(pos));
			end
			sol=temp';
		end

		F = scatteredInterpolant(md.mesh.lat,md.mesh.long,sol);
		F.Method = 'linear';
		F.ExtrapolationMethod = 'linear';
		
		sol_grid = F(lat_grid, lon_grid);
		sol_grid(isnan(sol_grid))=0;
		sol_grid(lat_grid>85 & sol_grid==0) = NaN;

		set(0,'DefaultAxesFontSize',18,'DefaultAxesLineWidth',1,'DefaultTextFontSize',18,'DefaultLineMarkerSize',8)
		figure1=figure('Position', [100, 100, 1000, 500]);
		gcf; load coast; cla;
		pcolor(lon_grid,lat_grid,sol_grid); shading flat; hold on;
		if (kk==1)
			geoshow(flipud(lat),flipud(long),'DisplayType','polygon','FaceColor','white');
		else
			geoshow(lat,long,'DisplayType','polygon','FaceColor','white');
		end
		plot(long,lat,'k'); hold off;
		c1=colorbar;
		colormap('haxby');
		caxis([-floor(min(abs(min(sol)),abs(max(sol)))) floor(min(abs(min(sol)),abs(max(sol))))]);
		xlim([-180 180]);
		ylim([-90 90]);
		grid on;
		title(sol_name(kk));
		set(gcf,'color','w');
		%export_fig(fig_name{kk});
	end
end %}}}

if any(steps==6) %{{{
	disp('   Step 6: Transient run');
	md = loadmodel('./Models/SlrGRACE_Parameterization');

	disp('Projecting  loads onto the mesh...');
	time_range = 2007 + [15 350]/365;
	water_load = grace(md.mesh.elements,md.mesh.lat,md.mesh.long,time_range(1),time_range(2));

	[ne,nt]=size(water_load);
	md.solidearth.surfaceload.icethicknesschange = zeros(ne+1,nt);
	md.solidearth.surfaceload.icethicknesschange(1:ne,:) = water_load*md.materials.rho_freshwater/md.materials.rho_ice;
	md.solidearth.surfaceload.icethicknesschange(ne+1,:)=[1:nt]; % times

	md.transient.issmb=0;
	md.transient.ismasstransport=0;
	md.transient.isstressbalance=0;
	md.transient.isthermal=0;
	md.transient.isgia=1; 
	md.transient.isslr=1;

	md.timestepping.start_time=-1;
	md.timestepping.final_time=nt;
	md.timestepping.time_step=1;
	md.timestepping.interp_forcings=0;
	md.settings.output_frequency=1;

	md.cluster=generic('name',oshostname(),'np',3);
	md.verbose=verbose('111111111');

	md.solidearth.requested_outputs = {'Sealevel','SealevelRSL'};
	
	md=solve(md,'tr');

	save ./Models/SlrGRACE_Transient md;
end %}}}

if any(steps==7) %{{{
	disp('   Step 7: Plot transient');
	md = loadmodel('./Models/SlrGRACE_Transient');

	time = md.solidearth.surfaceload.icethicknesschange(end,:);

	for tt=1:length(time)
		gmsl(tt) = md.results.TransientSolution(tt).Bslr*1000; % GMSL rate mm/yr
		sol1(:,tt) = md.solidearth.surfaceload.icethicknesschange(1:end-1,tt)*100;	% ice equivalent height [cm/yr]
		sol2(:,tt) = md.results.TransientSolution(tt+1).SealevelRSL*1000;	% mm/yr
	end
	sol_name = {'Change in water equivalent height [cm]', 'Relative sea-level [mm]'};
	movie_name = {'Movie_dH.avi','Movie_slr.avi'};

	res = 1.0;

	[lat_grid, lon_grid] = meshgrid(linspace(-90,90,180/res), linspace(-180,180,360/res));
	sol_grid = zeros(size(lat_grid));

	for kk=1:2
		sol=eval(sprintf('sol%d',kk));

		if length(sol)==md.mesh.numberofelements
			for jj=1:md.mesh.numberofelements
				ii=(jj-1)*3;
				pp(ii+1:ii+3)=md.mesh.elements(jj,:);
			end
			for jj=1:md.mesh.numberofvertices
				pos=ceil(find(pp==jj)/3);
				temp2(jj,:)=mean(sol(pos,:));
			end
			sol=temp2;
		end

		vidObj = VideoWriter(movie_name{kk});
		vidObj.FrameRate=2; % frames per second
		open(vidObj);

		for jj=1:length(time)
			F = scatteredInterpolant(md.mesh.lat,md.mesh.long,sol(:,jj));
			F.Method = 'linear';
			F.ExtrapolationMethod = 'linear';
			
			sol_grid = F(lat_grid, lon_grid);
			sol_grid(isnan(sol_grid))=0;
			sol_grid(lat_grid>85 & sol_grid==0) = NaN;

			set(0,'DefaultAxesFontSize',18,'DefaultAxesLineWidth',1,'DefaultTextFontSize',18,'DefaultLineMarkerSize',8)
			figure1=figure('Position', [100, 100, 1000, 500]);
			gcf; load coast; cla;
			pcolor(lon_grid,lat_grid,sol_grid); shading flat; hold on;
			if (kk==1)
				geoshow(flipud(lat),flipud(long),'DisplayType','polygon','FaceColor','white');
			else
				geoshow(lat,long,'DisplayType','polygon','FaceColor','white');
			end
			plot(long,lat,'k'); hold off;
			c1=colorbar;
			colormap('haxby');
			caxis([-floor(min(abs(min(min(sol))),abs(max(max(sol))))) floor(min(abs(min(min(sol))),abs(max(max(sol)))))]);
			xlim([-180 180]);
			ylim([-90 90]);
			grid on;
			title(sol_name(kk));
			set(gcf,'color','w');
			writeVideo(vidObj,getframe(gcf));
			close
		end
		close(vidObj);
	end
	!open *.avi;

	% plot GMSL time series
	plot(time,gmsl,'-*','linewidth',3); grid on;
	xlabel('# month');
	ylabel('GMSL [mm/yr]');
	set(gcf,'color','w');
	%export_fig('Fig7.pdf');
end %}}}
