/*!\file Tria.c
 * \brief: implementation of the Tria object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./Tria.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../DataSet/DataSet.h"
#include "../include/typedefs.h"


/*For debugging purposes: */
#define ELID 141 //element for which to print debug statements
#define RANK 2 //rank of cpu for which to print debug statements.
//#define _DEBUGELEMENTS_
//#define _DEBUGGAUSS_

Tria::Tria(){
	return;
}

Tria::~Tria(){
	return;
}

Tria::Tria(int tria_id,int tria_mid,int tria_mparid,int tria_numparid,int tria_node_ids[3],double tria_h[3],double tria_s[3],double tria_b[3],double tria_k[3],double tria_melting[3],
				double tria_accumulation[3],double tria_geothermalflux[3],int tria_friction_type,double tria_p,double tria_q,int tria_shelf, bool tria_onwater){
	
	int i;
	
	id=tria_id;
	mid=tria_mid;
	mparid=tria_mparid;
	numparid=tria_numparid;
	for(i=0;i<3;i++){
		node_ids[i]=tria_node_ids[i];
		node_offsets[i]=UNDEF;
		nodes[i]=NULL;
		h[i]=tria_h[i];
		s[i]=tria_s[i];
		b[i]=tria_b[i];
		k[i]=tria_k[i];
		melting[i]=tria_melting[i];
		accumulation[i]=tria_accumulation[i];
		geothermalflux[i]=tria_geothermalflux[i]; 
	}
	matice=NULL;
	matice_offset=UNDEF;
	matpar=NULL;
	matpar_offset=UNDEF;
	numpar=NULL;
	numpar_offset=UNDEF;
	friction_type=tria_friction_type;
	p=tria_p;
	q=tria_q;
	shelf=tria_shelf;
	onbed=1;
	onwater=tria_onwater;
	
	return;
}

#undef __FUNCT__
#define __FUNCT__ "Tria::Echo"

void Tria::Echo(void){

	printf("Tria:\n");
	printf("   id: %i\n",id);
	printf("   mid: %i\n",mid);
	printf("   mparid: %i\n",mparid);
	printf("   node_ids=[%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2]);
	printf("   node_offsets=[%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2]);
	printf("   matice_offset=%i\n",matice_offset);
	printf("   matpar_offset=%i\n",matpar_offset);
	printf("   h=[%g,%g,%g]\n",h[0],h[1],h[2]);
	printf("   s=[%g,%g,%g]\n",s[0],s[1],s[2]);
	printf("   b=[%g,%g,%g]\n",b[0],b[1],b[2]);
	printf("   k=[%g,%g,%g]\n",k[0],k[1],k[2]);
	printf("   melting=[%g,%g,%g]\n",melting[0],melting[1],melting[2]);
	printf("   accumulation=[%g,%g,%g]\n",accumulation[0],accumulation[1],accumulation[2]);
	printf("   geothermalflux=[%g,%g,%g]\n",geothermalflux[0],geothermalflux[1],geothermalflux[2]);
	printf("   friction_type: %i\n",friction_type);
	printf("   p: %g\n",p);
	printf("   q: %g\n",q);
	printf("   shelf: %i\n",shelf);
	printf("   onbed: %i\n",onbed);
	printf("   onwater: %i\n",onwater);
	printf("   nodes: \n");
	if(nodes[0])nodes[0]->Echo();
	if(nodes[1])nodes[1]->Echo();
	if(nodes[2])nodes[2]->Echo();
	if(matice)matice->Echo();
	if(matpar)matpar->Echo();

	return;
}
#undef __FUNCT__
#define __FUNCT__ "Tria::DeepEcho"

void Tria::DeepEcho(void){

	printf("Tria:\n");
	printf("   id: %i\n",id);
	printf("   mid: %i\n",mid);
	printf("   mparid: %i\n",mparid);
	printf("   node_ids=[%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2]);
	printf("   node_offsets=[%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2]);
	printf("   matice_offset=%i\n",matice_offset);
	printf("   matpar_offset=%i\n",matpar_offset);
	printf("   h=[%g,%g,%g]\n",h[0],h[1],h[2]);
	printf("   s=[%g,%g,%g]\n",s[0],s[1],s[2]);
	printf("   b=[%g,%g,%g]\n",b[0],b[1],b[2]);
	printf("   k=[%g,%g,%g]\n",k[0],k[1],k[2]);
	printf("   melting=[%g,%g,%g]\n",melting[0],melting[1],melting[2]);
	printf("   accumulation=[%g,%g,%g]\n",accumulation[0],accumulation[1],accumulation[2]);
	printf("   geothermalflux=[%g,%g,%g]\n",geothermalflux[0],geothermalflux[1],geothermalflux[2]);
	printf("   friction_type: %i\n",friction_type);
	printf("   p: %g\n",p);
	printf("   q: %g\n",q);
	printf("   shelf: %i\n",shelf);
	printf("   onbed: %i\n",onbed);
	printf("   onwater: %i\n",onwater);
	printf("   nodes: \n");
	if(nodes[0])nodes[0]->Echo();
	if(nodes[1])nodes[1]->Echo();
	if(nodes[2])nodes[2]->Echo();
	if(matice)matice->Echo();
	if(matpar)matpar->Echo();

	return;
}
void  Tria::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Tria: */
	enum_type=TriaEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Tria data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&mid,sizeof(mid));marshalled_dataset+=sizeof(mid);
	memcpy(marshalled_dataset,&mparid,sizeof(mparid));marshalled_dataset+=sizeof(mparid);
	memcpy(marshalled_dataset,&node_ids,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(marshalled_dataset,&nodes,sizeof(nodes));marshalled_dataset+=sizeof(nodes);
	memcpy(marshalled_dataset,&node_offsets,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);
	memcpy(marshalled_dataset,&matice,sizeof(matice));marshalled_dataset+=sizeof(matice);
	memcpy(marshalled_dataset,&matice_offset,sizeof(matice_offset));marshalled_dataset+=sizeof(matice_offset);
	memcpy(marshalled_dataset,&matpar,sizeof(matpar));marshalled_dataset+=sizeof(matpar);
	memcpy(marshalled_dataset,&matpar_offset,sizeof(matpar_offset));marshalled_dataset+=sizeof(matpar_offset);
	memcpy(marshalled_dataset,&numparid,sizeof(numparid));marshalled_dataset+=sizeof(numparid);
	memcpy(marshalled_dataset,&numpar,sizeof(numpar));marshalled_dataset+=sizeof(numpar);
	memcpy(marshalled_dataset,&numpar_offset,sizeof(numpar_offset));marshalled_dataset+=sizeof(numpar_offset);
	memcpy(marshalled_dataset,&h,sizeof(h));marshalled_dataset+=sizeof(h);
	memcpy(marshalled_dataset,&s,sizeof(s));marshalled_dataset+=sizeof(s);
	memcpy(marshalled_dataset,&b,sizeof(b));marshalled_dataset+=sizeof(b);
	memcpy(marshalled_dataset,&k,sizeof(k));marshalled_dataset+=sizeof(k);
	memcpy(marshalled_dataset,&melting,sizeof(melting));marshalled_dataset+=sizeof(melting);
	memcpy(marshalled_dataset,&accumulation,sizeof(accumulation));marshalled_dataset+=sizeof(accumulation);
	memcpy(marshalled_dataset,&geothermalflux,sizeof(geothermalflux));marshalled_dataset+=sizeof(geothermalflux);
	memcpy(marshalled_dataset,&friction_type,sizeof(friction_type));marshalled_dataset+=sizeof(friction_type);
	memcpy(marshalled_dataset,&onbed,sizeof(onbed));marshalled_dataset+=sizeof(onbed);
	memcpy(marshalled_dataset,&onwater,sizeof(onwater));marshalled_dataset+=sizeof(onwater);
	memcpy(marshalled_dataset,&p,sizeof(p));marshalled_dataset+=sizeof(p);
	memcpy(marshalled_dataset,&q,sizeof(q));marshalled_dataset+=sizeof(q);
	memcpy(marshalled_dataset,&shelf,sizeof(shelf));marshalled_dataset+=sizeof(shelf);
	
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
		
int   Tria::MarshallSize(){
	return sizeof(id)
		+sizeof(mid)
		+sizeof(mparid)
		+sizeof(node_ids)
		+sizeof(nodes)
		+sizeof(node_offsets)
		+sizeof(matice)
		+sizeof(matice_offset)
		+sizeof(matpar)
		+sizeof(matpar_offset)
		+sizeof(numparid)
		+sizeof(numpar)
		+sizeof(numpar_offset)
		+sizeof(h)
		+sizeof(s)
		+sizeof(b)
		+sizeof(k)
		+sizeof(melting)
		+sizeof(accumulation)
		+sizeof(geothermalflux)
		+sizeof(friction_type)
		+sizeof(onbed)
		+sizeof(onwater)
		+sizeof(p)
		+sizeof(q)
		+sizeof(shelf)
		+sizeof(int); //sizeof(int) for enum type
}

char* Tria::GetName(void){
	return "tria";
}

void  Tria::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&mid,marshalled_dataset,sizeof(mid));marshalled_dataset+=sizeof(mid);
	memcpy(&mparid,marshalled_dataset,sizeof(mparid));marshalled_dataset+=sizeof(mparid);
	memcpy(&node_ids,marshalled_dataset,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(&nodes,marshalled_dataset,sizeof(nodes));marshalled_dataset+=sizeof(nodes);
	memcpy(&node_offsets,marshalled_dataset,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);
	memcpy(&matice,marshalled_dataset,sizeof(matice));marshalled_dataset+=sizeof(matice);
	memcpy(&matice_offset,marshalled_dataset,sizeof(matice_offset));marshalled_dataset+=sizeof(matice_offset);
	memcpy(&matpar,marshalled_dataset,sizeof(matpar));marshalled_dataset+=sizeof(matpar);
	memcpy(&matpar_offset,marshalled_dataset,sizeof(matpar_offset));marshalled_dataset+=sizeof(matpar_offset);
	memcpy(&numparid,marshalled_dataset,sizeof(numparid));marshalled_dataset+=sizeof(numparid);
	memcpy(&numpar,marshalled_dataset,sizeof(numpar));marshalled_dataset+=sizeof(numpar);
	memcpy(&numpar_offset,marshalled_dataset,sizeof(numpar_offset));marshalled_dataset+=sizeof(numpar_offset);
	memcpy(&h,marshalled_dataset,sizeof(h));marshalled_dataset+=sizeof(h);
	memcpy(&s,marshalled_dataset,sizeof(s));marshalled_dataset+=sizeof(s);
	memcpy(&b,marshalled_dataset,sizeof(b));marshalled_dataset+=sizeof(b);
	memcpy(&k,marshalled_dataset,sizeof(k));marshalled_dataset+=sizeof(k);
	memcpy(&melting,marshalled_dataset,sizeof(melting));marshalled_dataset+=sizeof(melting);
	memcpy(&accumulation,marshalled_dataset,sizeof(accumulation));marshalled_dataset+=sizeof(accumulation);
	memcpy(&geothermalflux,marshalled_dataset,sizeof(geothermalflux));marshalled_dataset+=sizeof(geothermalflux);
	memcpy(&friction_type,marshalled_dataset,sizeof(friction_type));marshalled_dataset+=sizeof(friction_type);
	memcpy(&onbed,marshalled_dataset,sizeof(onbed));marshalled_dataset+=sizeof(onbed);
	memcpy(&onwater,marshalled_dataset,sizeof(onwater));marshalled_dataset+=sizeof(onwater);
	memcpy(&p,marshalled_dataset,sizeof(p));marshalled_dataset+=sizeof(p);
	memcpy(&q,marshalled_dataset,sizeof(q));marshalled_dataset+=sizeof(q);
	memcpy(&shelf,marshalled_dataset,sizeof(shelf));marshalled_dataset+=sizeof(shelf);

	/*nodes and materials are not pointing to correct objects anymore:*/
	for(i=0;i<3;i++)nodes[i]=NULL;
	matice=NULL;
	matpar=NULL;
	numpar=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
int Tria::Enum(void){

	return TriaEnum();

}
int    Tria::GetId(){ return id; }

int    Tria::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::Configure"
void  Tria::Configure(void* ploadsin,void* pnodesin,void* pmaterialsin,void* pparametersin){

	int i;
	
	DataSet* loadsin=NULL;
	DataSet* nodesin=NULL;
	DataSet* materialsin=NULL;
	DataSet* parametersin=NULL;

	/*Recover pointers :*/
	loadsin=(DataSet*)ploadsin;
	nodesin=(DataSet*)pnodesin;
	materialsin=(DataSet*)pmaterialsin;
	parametersin=(DataSet*)pparametersin;

	/*Link this element with its nodes, ie find pointers to the nodes in the nodes dataset.: */
	ResolvePointers((Object**)nodes,node_ids,node_offsets,3,nodesin);
	
	/*Same for materials: */
	ResolvePointers((Object**)&matice,&mid,&matice_offset,1,materialsin);
	ResolvePointers((Object**)&matpar,&mparid,&matpar_offset,1,materialsin);

	/*Same for numpar: */
	ResolvePointers((Object**)&numpar,&numparid,&numpar_offset,1,parametersin);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreateKMatrix"

void  Tria::CreateKMatrix(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum()){
		
		CreateKMatrixDiagnosticHoriz( Kgg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==DiagnosticAnalysisEnum()){
	
		if (sub_analysis_type==HorizAnalysisEnum()){

			CreateKMatrixDiagnosticHoriz( Kgg,inputs,analysis_type,sub_analysis_type);
		}
		else throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","sub_analysis: ",sub_analysis_type," not supported yet"));

	}
	else if (analysis_type==SlopeComputeAnalysisEnum()){

		CreateKMatrixSlopeCompute( Kgg,inputs,analysis_type,sub_analysis_type);

	}
	else if (analysis_type==PrognosticAnalysisEnum()){

		CreateKMatrixPrognostic( Kgg,inputs,analysis_type,sub_analysis_type);

	}
	else{
		throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","analysis: ",analysis_type," not supported yet"));
	}

}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreateKMatrixDiagnosticHoriz"

void  Tria::CreateKMatrixDiagnosticHoriz(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* material data: */
	double viscosity; //viscosity
	double newviscosity; //viscosity
	double oldviscosity; //viscosity
	
	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/
	double oldepsilon[3]; /* oldepsilon=[exx,eyy,exy];*/

	/* matrices: */
	double B[3][numdof];
	double Bprime[3][numdof];
	double D[3][3]={{ 0,0,0 },{0,0,0},{0,0,0}};              // material matrix, simple scalar matrix.
	double D_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.
	
	double Jdet;
	
	/*input parameters for structural analysis (diagnostic): */
	double  vxvy_list[numgrids][2]={{0,0},{0,0},{0,0}};
	double  oldvxvy_list[numgrids][2]={{0,0},{0,0},{0,0}};
	double  thickness;
	int     dofs[2]={0,1};

	ParameterInputs* inputs=NULL;

	/*First, if we are on water, return empty matrix: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover extra inputs from users, at current convergence iteration: */
	inputs->Recover("velocity",&vxvy_list[0][0],2,dofs,numgrids,(void**)nodes);
	inputs->Recover("old_velocity",&oldvxvy_list[0][0],2,dofs,numgrids,(void**)nodes);

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set Ke_gg to 0: */
	for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gg[i][j]=0.0;

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("El id %i Rank %i TriaElemnet input list before gaussian loop: \n",ELID,RANK); 
		printf("   rho_ice: %g \n",matpar->GetRhoIce());
		printf("   gravity: %g \n",matpar->GetG())
		printf("   rho_water: %g \n",matpar->GetRhoWater());
		printf("   Velocity: \n");
		for (i=0;i<numgrids;i++){
			printf("      node %i  [%g,%g]\n",i,vxvy_list[i][0],vxvy_list[i][1]);
		}
		printf("   flow_law_parameter [%g ]\n",matice->GetB());
		printf("   drag [%g %g %g ]\n",k[0],k[1],k[2]);
		printf("   thickness [%g %g %g]\n",h[0],h[1],h[2]);
		printf("   surface [%g %g %g ]\n",s[0],s[1],s[2]);
		printf("   bed [%g %g %g]\n",b[0],b[1],b[2]);
	}
	#endif


	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("   gaussian points: \n");
		for(i=0;i<num_gauss;i++){
			printf("    %g %g %g : %g\n",first_gauss_area_coord[i],second_gauss_area_coord[i],third_gauss_area_coord[i],gauss_weights[i]);
		}
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);


		/*Compute thickness at gaussian point: */
		GetParameterValue(&thickness, &h[0],gauss_l1l2l3);

		/*Get strain rate from velocity: */
		GetStrainRate(&epsilon[0],&vxvy_list[0][0],&xyz_list[0][0],gauss_l1l2l3);
		GetStrainRate(&oldepsilon[0],&oldvxvy_list[0][0],&xyz_list[0][0],gauss_l1l2l3);
		
		/*Get viscosity: */
		matice->GetViscosity2d(&viscosity, &epsilon[0]);
		matice->GetViscosity2d(&oldviscosity, &oldepsilon[0]);
		
		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
		   onto this scalar matrix, so that we win some computational time: */
		newviscosity=viscosity+numpar->viscosity_overshoot*(viscosity-oldviscosity);
		D_scalar=newviscosity*thickness*gauss_weight*Jdet;

		for (i=0;i<3;i++){
			D[i][i]=D_scalar;
		}
		
		#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && id==ELID){ 
			printf("   gaussian loop %i\n",ig);
			printf("      thickness %g\n",thickness);
			printf("      slope [%g,%g]\n",slope[0],slope[1]);
			printf("      alpha2_list [%g,%g,%g]\n",alpha2_list[0],alpha2_list[1],alpha2_list[2]);
			printf("      epsilon [%g,%g,%g]\n",epsilon[0],epsilon[1],epsilon[2]);
			printf("      Matice: \n");
			matice->Echo();
			printf("      Matpar: \n");
			matpar->Echo();
			printf("\n    viscosity: %g \n",viscosity);
			printf("      jacobian: %g \n",Jdet);
			printf("      gauss_weight: %g \n",gauss_weight);
		}
		#endif

		/*Get B and Bprime matrices: */
		GetB(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		/*  Do the triple product tB*D*Bprime: */
		TripleMultiply( &B[0][0],3,numdof,1,
					  &D[0][0],3,3,0,
					  &Bprime[0][0],3,numdof,0,
					  &Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
		
		#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && id==ELID){ 
			printf("      B:\n");
			for(i=0;i<3;i++){
				for(j=0;j<numdof;j++){
					printf("%g ",B[i][j]);
				}
				printf("\n");
			}
			printf("      Bprime:\n");
			for(i=0;i<3;i++){
				for(j=0;j<numdof;j++){
					printf("%g ",Bprime[i][j]);
				}
				printf("\n");
			}
		}
		#endif
	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);


	/*Do not forget to include friction: */
	if(!shelf){
		CreateKMatrixDiagnosticHorizFriction(Kgg,inputs,analysis_type,sub_analysis_type);
	}

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("      Ke_gg erms:\n");
		for( i=0; i<numdof; i++){
			for (j=0;j<numdof;j++){
				printf("%g ",Ke_gg[i][j]);
			}
			printf("\n");
		}
		printf("      Ke_gg row_indices:\n");
		for( i=0; i<numdof; i++){
			printf("%i ",doflist[i]);
		}

	}
	#endif

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreateKMatrixPrognostic"
void  Tria::CreateKMatrixPrognostic(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};//local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_thickness1[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_thickness2[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	
	double Jdettria;
	
	/*input parameters for structural analysis (diagnostic): */
	double  vxvy_list[numgrids][2]={0.0};
	double  vx_list[numgrids]={0.0};
	double  vy_list[numgrids]={0.0};
	double  dvx[2];
	double  dvy[2];
	double  vx,vy;
	double  dvxdx,dvydy;
	double  v_gauss[2]={0.0};
	double  K[2][2]={0.0};
	double  KDL[2][2]={0.0};
	double  dt;
	int     dofs[2]={0,1};
	int     found=0;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover extra inputs from users, at current convergence iteration: */
	found=inputs->Recover("velocity_average",&vxvy_list[0][0],2,dofs,numgrids,(void**)nodes);
	if(!found)throw ErrorException(__FUNCT__," could not find velocity_average  in inputs!");

	for(i=0;i<numgrids;i++){
		vx_list[i]=vxvy_list[i][0];
		vy_list[i]=vxvy_list[i][1];
	}
	
	found=inputs->Recover("dt",&dt);
	if(!found)throw ErrorException(__FUNCT__," could not find dt in inputs!");

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//Create Artificial diffusivity once for all if requested
	if(numpar->artdiff){
		//Get the Jacobian determinant
		gauss_l1l2l3[0]=1.0/3.0; gauss_l1l2l3[1]=1.0/3.0; gauss_l1l2l3[2]=1.0/3.0;
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		//Build K matrix (artificial diffusivity matrix)
		v_gauss[0]=1.0/3.0*(vxvy_list[0][0]+vxvy_list[1][0]+vxvy_list[2][0]);
		v_gauss[1]=1.0/3.0*(vxvy_list[0][1]+vxvy_list[1][1]+vxvy_list[2][1]);
		
		K[0][0]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]);
		K[1][1]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		DL_scalar=gauss_weight*Jdettria;

		/*  Do the triple product tL*D*L: */
		TripleMultiply( &L[0],1,numdof,1,
					  &DL_scalar,1,1,0,
					  &L[0],1,numdof,0,
					  &Ke_gg_gaussian[0][0],0);
		
		/*Get B  and B prime matrix: */
		GetB_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);
		
		//Get vx, vy and their derivatives at gauss point
		GetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
		GetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);
		
		GetParameterDerivativeValue(&dvx[0], &vx_list[0],&xyz_list[0][0], gauss_l1l2l3);
		GetParameterDerivativeValue(&dvy[0], &vy_list[0],&xyz_list[0][0], gauss_l1l2l3);

		dvxdx=dvx[0];
		dvydy=dvy[1];

		DL_scalar=dt*gauss_weight*Jdettria;

		//Create DL and DLprime matrix
		DL[0][0]=DL_scalar*dvxdx;
		DL[1][1]=DL_scalar*dvydy;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		//Ke_gg_thickness=B'*DL*B+B'*DLprime*Bprime;

		TripleMultiply( &B[0][0],2,numdof,1,
					  &DL[0][0],2,2,0,
					  &B[0][0],2,numdof,0,
					  &Ke_gg_thickness1[0][0],0);

		TripleMultiply( &B[0][0],2,numdof,1,
					  &DLprime[0][0],2,2,0,
					  &Bprime[0][0],2,numdof,0,
					  &Ke_gg_thickness2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness2[i][j];
		
		if(numpar->artdiff){
			
			/* Compute artificial diffusivity */
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						  &KDL[0][0],2,2,0,
						  &Bprime[0][0],2,numdof,0,
						  &Ke_gg_gaussian[0][0],0);

			/* Add artificial diffusivity matrix */
			for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
			
		}

		#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && id==ELID){ 
			printf("      B:\n");
			for(i=0;i<3;i++){
				for(j=0;j<numdof;j++){
					printf("%g ",B[i][j]);
				}
				printf("\n");
			}
			printf("      Bprime:\n");
			for(i=0;i<3;i++){
				for(j=0;j<numdof;j++){
					printf("%g ",Bprime[i][j]);
				}
				printf("\n");
			}
		}
		#endif
	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("      Ke_gg erms:\n");
		for( i=0; i<numdof; i++){
			for (j=0;j<numdof;j++){
				printf("%g ",Ke_gg[i][j]);
			}
			printf("\n");
		}
		printf("      Ke_gg row_indices:\n");
		for( i=0; i<numdof; i++){
			printf("%i ",doflist[i]);
		}

	}
	#endif

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreatePVectorPrognostic"
void  Tria::CreatePVectorPrognostic(Vec pg ,void* vinputs,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;
	
	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_list[numgrids]={0.0};
	double  accumulation_g;
	double  melting_list[numgrids]={0.0};
	double  melting_g;
	double  thickness_list[numgrids]={0.0};
	double  thickness_g;
	double  dt;
	int     dofs[1]={0};
	int     found=0;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover extra inputs from users, at current convergence iteration: */
	found=inputs->Recover("accumulation",&accumulation_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)throw ErrorException(__FUNCT__," could not find accumulation in inputs!");
	found=inputs->Recover("melting",&melting_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)throw ErrorException(__FUNCT__," could not find melting in inputs!");
	found=inputs->Recover("thickness",&thickness_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)throw ErrorException(__FUNCT__," could not find thickness in inputs!");
	found=inputs->Recover("dt",&dt);
	if(!found)throw ErrorException(__FUNCT__," could not find dt in inputs!");

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting and thickness at gauss point */
		GetParameterValue(&accumulation_g, &accumulation_list[0],gauss_l1l2l3);
		GetParameterValue(&melting_g, &melting_list[0],gauss_l1l2l3);
		GetParameterValue(&thickness_g, &thickness_list[0],gauss_l1l2l3);
		
		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(thickness_g+dt*(accumulation_g-melting_g))*L[i];
		
	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreateKMatrixDiagnosticHorizFriction"
void  Tria::CreateKMatrixDiagnosticHorizFriction(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[2][numdof];
	double DL[2][2]={{ 0,0 },{0,0}}; //for basal drag
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix contribution from drag
	
	double Jdet;
	
	/*slope: */
	double  slope[2]={0.0,0.0};
	double  slope_magnitude;

	/*input parameters for structural analysis (diagnostic): */
	double  vxvy_list[numgrids][2]={{0,0},{0,0},{0,0}};
	int     dofs[2]={0,1};

	/*friction: */
	double alpha2_list[numgrids]={0.0,0.0,0.0};
	double alpha2;

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;
	
	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set Ke_gg to 0: */
	for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gg[i][j]=0.0;

	if (shelf){
		/*no friction, do nothing*/
		return;
	}

	if (friction_type!=2)throw ErrorException(__FUNCT__," non-viscous friction not supported yet!");

	/*recover extra inputs from users, at current convergence iteration: */
	inputs->Recover("velocity",&vxvy_list[0][0],2,dofs,numgrids,(void**)nodes);

	/*Build alpha2_list used by drag stiffness matrix*/
	Friction* friction=NewFriction();
	
	/*Initialize all fields: */
	friction->element_type=(char*)xmalloc((strlen("2d")+1)*sizeof(char));
	strcpy(friction->element_type,"2d");
	
	friction->gravity=matpar->GetG();
	friction->rho_ice=matpar->GetRhoIce();
	friction->rho_water=matpar->GetRhoWater();
	friction->K=&k[0];
	friction->bed=&b[0];
	friction->thickness=&h[0];
	friction->velocities=&vxvy_list[0][0];
	friction->p=p;
	friction->q=q;

	/*Compute alpha2_list: */
	FrictionGetAlpha2(&alpha2_list[0],friction);

	/*Erase friction object: */
	DeleteFriction(&friction);

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("   alpha2_list [%g %g %g ]\n",alpha2_list[0],alpha2_list[1],alpha2_list[2]);
	}
	#endif

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("   gaussian points: \n");
		for(i=0;i<num_gauss;i++){
			printf("    %g %g %g : %g\n",first_gauss_area_coord[i],second_gauss_area_coord[i],third_gauss_area_coord[i],gauss_weights[i]);
		}
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);


		// If we have a slope > 6% for this element,  it means  we are on a mountain. In this particular case, 
		//velocity should be = 0. To achieve this result, we set alpha2_list to a very high value: */
		GetParameterDerivativeValue(&slope[0], &s[0],&xyz_list[0][0], gauss_l1l2l3);
		slope_magnitude=sqrt(pow(slope[0],2)+pow(slope[1],2));

		if (slope_magnitude>MAXSLOPE){
			alpha2_list[0]=pow((double)10,MOUNTAINKEXPONENT);
			alpha2_list[1]=pow((double)10,MOUNTAINKEXPONENT);
			alpha2_list[2]=pow((double)10,MOUNTAINKEXPONENT);
		}

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/*Now, take care of the basal friction if there is any: */
		GetParameterValue(&alpha2, &alpha2_list[0],gauss_l1l2l3);

		DL_scalar=alpha2*gauss_weight*Jdet;
		for (i=0;i<2;i++){
			DL[i][i]=DL_scalar;
		}
		
		/*  Do the triple producte tL*D*L: */
		TripleMultiply( &L[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&L[0][0],2,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}	

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreateKMatrixSlopeCompute"

void  Tria::CreateKMatrixSlopeCompute(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[1][3];
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.
	
	double Jdet;
	
	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set Ke_gg to 0: */
	for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gg[i][j]=0.0;

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		
		/*Get L matrix: */
		GetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		DL_scalar=gauss_weight*Jdet;

		/*  Do the triple producte tL*D*L: */
		TripleMultiply( &L[0][0],1,3,1,
					&DL_scalar,1,1,0,
					&L[0][0],1,3,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
	} //for (ig=0; ig<num_gauss; ig++

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);
		
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreatePVector"
void  Tria::CreatePVector(Vec pg,void* inputs,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum()){
		
		CreatePVectorDiagnosticHoriz( pg,inputs,analysis_type,sub_analysis_type);
	
	}
	else if (analysis_type==DiagnosticAnalysisEnum()){
		if (sub_analysis_type==HorizAnalysisEnum()){
		
			CreatePVectorDiagnosticHoriz( pg,inputs,analysis_type,sub_analysis_type);
		
		}
		else throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","sub_analysis: ",sub_analysis_type," not supported yet"));
	}
	else if (analysis_type==SlopeComputeAnalysisEnum()){
		
		CreatePVectorSlopeCompute( pg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==PrognosticAnalysisEnum()){

		CreatePVectorPrognostic( pg,inputs,analysis_type,sub_analysis_type);
	}
	else{
		throw ErrorException(__FUNCT__,exprintf("%s%i%s"," analysis ",analysis_type," not supported yet"));
	}

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreatePVectorDiagnosticHoriz"
void Tria::CreatePVectorDiagnosticHoriz( Vec pg, void* vinputs, int analysis_type,int sub_analysis_type){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* parameters: */
	double  plastic_stress; 
	double  slope[NDOF2];
	double  driving_stress_baseline;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof];
	double  pe_g_gaussian[numdof];

	/*input parameters for structural analysis (diagnostic): */
	double  thickness;

	ParameterInputs* inputs=NULL;

	/*First, if we are on water, return empty vector: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set pe_g to 0: */
	for(i=0;i<numdof;i++) pe_g[i]=0.0;


	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("gravity %g\n",matpar->GetG());
		printf("rho_ice %g\n",matpar->GetRhoIce());
		printf("thickness [%g,%g,%g]\n",h[0],h[1],h[2]);
		printf("surface[%g,%g,%g]\n",s[0],s[1],s[2]);
		printf("bed[%g,%g,%g]\n",b[0],b[1],b[2]);
		printf("drag [%g,%g,%g]\n",k[0],k[1],k[2]);
	}
	#endif


	/* Get gaussian points and weights: */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2); /*We need higher order because our load is order 2*/

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("   gaussian points: \n");
		for(i=0;i<num_gauss;i++){
			printf("    %g %g %g : %g\n",first_gauss_area_coord[i],second_gauss_area_coord[i],third_gauss_area_coord[i],gauss_weights[i]);
		}
	}
	#endif



	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Compute thickness at gaussian point: */
		GetParameterValue(&thickness, &h[0],gauss_l1l2l3);
	
		GetParameterDerivativeValue(&slope[0], &s[0],&xyz_list[0][0], gauss_l1l2l3);
		
		/*In case we have plastic basal drag, compute plastic stress at gaussian point from k1, k2 and k3 fields in the 
		 * element itself: */
		if(friction_type==1){
			GetParameterValue(&plastic_stress, &k[0],gauss_l1l2l3);
		}

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		 /*Get nodal functions: */
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Compute driving stress: */
		driving_stress_baseline=matpar->GetRhoIce()*matpar->GetG()*thickness;


	 	#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && id==ELID){ 
			printf("      gaussian %i\n",ig);
			printf("      thickness %g\n",thickness);
			printf("      slope(%g,%g)\n",slope[0],slope[1]);
			printf("      Jdet %g\n",Jdet);
			printf("      gaussweigth %g\n",gauss_weight);
			printf("      l1l2l3 (%g,%g,%g)\n",l1l2l3[0],l1l2l3[1],l1l2l3[2]);
			if(friction_type==1)printf("      plastic_stress(%g)\n",plastic_stress);
		}
		#endif

		/*Build pe_g_gaussian vector: */
		if(friction_type==1){
			for (i=0;i<numgrids;i++){
				for (j=0;j<NDOF2;j++){
					pe_g_gaussian[i*NDOF2+j]=(-driving_stress_baseline*slope[j]-plastic_stress)*Jdet*gauss_weight*l1l2l3[i]; 
				}
			}
		}
		else {
			for (i=0;i<numgrids;i++){
				for (j=0;j<NDOF2;j++){
					pe_g_gaussian[i*NDOF2+j]=-driving_stress_baseline*slope[j]*Jdet*gauss_weight*l1l2l3[i];
				}
			}
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

	} //for (ig=0; ig<num_gauss; ig++)

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("      pe_g->terms\n",ig);
		for( i=0; i<pe_g->nrows; i++){
			printf("%g ",*(pe_g->terms+i));
		}
		printf("\n");
		printf("      pe_g->row_indices\n",ig);
		for( i=0; i<pe_g->nrows; i++){
			printf("%i ",*(pe_g->row_indices+i));
		}
	}
	#endif

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreatePVectorSlopeCompute"

void Tria::CreatePVectorSlopeCompute( Vec pg, void* vinputs, int analysis_type,int sub_analysis_type){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof];
	double  pe_g_gaussian[numdof];
	double  param[3];
	double  slope[2];

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set pe_g to 0: */
	for(i=0;i<numdof;i++) pe_g[i]=0.0;

	if ( (sub_analysis_type==SurfaceXAnalysisEnum()) || (sub_analysis_type==SurfaceYAnalysisEnum())){
		for(i=0;i<numdof;i++) param[i]=s[i];
	}
	if ( (sub_analysis_type==BedXAnalysisEnum()) || (sub_analysis_type==BedYAnalysisEnum())){
		for(i=0;i<numdof;i++) param[i]=b[i];
	}

	/* Get gaussian points and weights: */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2); /*We need higher order because our load is order 2*/


	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		GetParameterDerivativeValue(&slope[0], &param[0],&xyz_list[0][0], gauss_l1l2l3);
		
		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		 /*Get nodal functions: */
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Build pe_g_gaussian vector: */
		if ( (sub_analysis_type==SurfaceXAnalysisEnum()) || (sub_analysis_type==BedXAnalysisEnum())){
			for(i=0;i<numdof;i++) pe_g_gaussian[i]=Jdet*gauss_weight*slope[0]*l1l2l3[i];
		}
		if ( (sub_analysis_type==SurfaceYAnalysisEnum()) || (sub_analysis_type==BedYAnalysisEnum())){
			for(i=0;i<numdof;i++) pe_g_gaussian[i]=Jdet*gauss_weight*slope[1]*l1l2l3[i];
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

	} //for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::UpdateFromInputs"
void  Tria::UpdateFromInputs(void* vinputs){

	int     dofs[1]={0};
	double  temperature_list[3];
	double  temperature_average;
	double  B_list[3];
	double  B_average;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*Update internal data if inputs holds new values: */
	inputs->Recover("thickness",&h[0],1,dofs,3,(void**)nodes);
	inputs->Recover("surface",&s[0],1,dofs,3,(void**)nodes);
	inputs->Recover("bed",&b[0],1,dofs,3,(void**)nodes);
	inputs->Recover("drag",&k[0],1,dofs,3,(void**)nodes);
	inputs->Recover("melting",&melting[0],1,dofs,3,(void**)nodes);
	inputs->Recover("accumulation",&accumulation[0],1,dofs,3,(void**)nodes);
	inputs->Recover("geothermalflux",&geothermalflux[0],1,dofs,3,(void**)nodes);
	
	//Update material if necessary
	if(inputs->Recover("temperature_average",&temperature_list[0],1,dofs,3,(void**)nodes)){
		temperature_average=(temperature_list[0]+temperature_list[1]+temperature_list[2])/3.0;
		B_average=Paterson(temperature_average);
		matice->SetB(B_average);
	}
	
	if(inputs->Recover("B",&B_list[0],1,dofs,3,(void**)nodes)){
		B_average=(B_list[0]+B_list[1]+B_list[2])/3.0;
		matice->SetB(B_average);
	}

}
		
void  Tria::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	for(i=0;i<3;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}

void  Tria::GetDofList1(int* doflist){

	int i;
	for(i=0;i<3;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetParameterValue"
void Tria::GetParameterValue(double* pp, double* plist, double* gauss_l1l2l3){
	
	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter value at gaussian 
	 * point specifie by gauss_l1l2l3: */
	
	/*nodal functions: */
	double l1l2l3[3];

	/*output: */
	double p;

	GetNodalFunctions(l1l2l3, gauss_l1l2l3);

	p=l1l2l3[0]*plist[0]+l1l2l3[1]*plist[1]+l1l2l3[2]*plist[2];

	/*Assign output pointers:*/
	*pp=p;
}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetParameterDerivativeValue"
void Tria::GetParameterDerivativeValue(double* p, double* plist,double* xyz_list, double* gauss_l1l2l3){
	 
	const int NDOF2=2;
	const int numgrids=3;
	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter derivative value at gaussian 
	 * point specified by gauss_l1l2l3:
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *
	 * p is a vector of size 2x1 already allocated.
	 */
	
	double dh1dh2dh3_basic[NDOF2][numgrids]; //nodal derivative functions in basic coordinate system.

	/*Get dh1dh2dh3 in basic coordinate system: */
	GetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],xyz_list, gauss_l1l2l3);

	*(p+0)=plist[0]*dh1dh2dh3_basic[0][0]+plist[1]*dh1dh2dh3_basic[0][1]+plist[2]*dh1dh2dh3_basic[0][2];
	*(p+1)=plist[0]*dh1dh2dh3_basic[1][0]+plist[1]*dh1dh2dh3_basic[1][1]+plist[2]*dh1dh2dh3_basic[1][2];

}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetStrainRate"
void Tria::GetStrainRate(double* epsilon, double* velocity, double* xyz_list, double* gauss_l1l2l3){

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double B[3][NDOF2*numgrids];

	/*Get B matrix: */
	GetB(&B[0][0], xyz_list, gauss_l1l2l3);

	/*Multiply B by velocity, to get strain rate: */
	MatrixMultiply( &B[0][0],3,NDOF2*numgrids,0,
			              velocity,NDOF2*numgrids,1,0,
						  epsilon,0);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetJacobianDeterminant2d" 
void Tria::GetJacobianDeterminant2d(double*  Jdet, double* xyz_list,double* gauss_l1l2l3){

	/*The Jacobian determinant is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double x1,x2,x3,y1,y2,y3;
	
	x1=*(xyz_list+3*0+0);
	y1=*(xyz_list+3*0+1);
	x2=*(xyz_list+3*1+0);
	y2=*(xyz_list+3*1+1);
	x3=*(xyz_list+3*2+0);
	y3=*(xyz_list+3*2+1);


	*Jdet=sqrt(3.0)/6.0*((x2-x1)*(y3-y1)-(y2-y1)*(x3-x1));


	if(Jdet<0){
		printf("%s%s\n",__FUNCT__," error message: negative jacobian determinant!");
	}
	
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetJacobianDeterminant3d" 
void Tria::GetJacobianDeterminant3d(double*  Jdet, double* xyz_list,double* gauss_l1l2l3){

	/*The Jacobian determinant is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double x1,x2,x3,y1,y2,y3,z1,z2,z3;
	
	x1=*(xyz_list+3*0+0);
	y1=*(xyz_list+3*0+1);
	z1=*(xyz_list+3*0+2);
	x2=*(xyz_list+3*1+0);
	y2=*(xyz_list+3*1+1);
	z2=*(xyz_list+3*1+2);
	x3=*(xyz_list+3*2+0);
	y3=*(xyz_list+3*2+1);
	z3=*(xyz_list+3*2+2);


	*Jdet=sqrt(3.0)/6.0*pow(pow(((y2-y1)*(z3-z1)-(z2-z1)*(y3-y1)),2.0)+pow(((z2-z1)*(x3-x1)-(x2-x1)*(z3-z1)),2.0)+pow(((x2-x1)*(y3-y1)-(y2-y1)*(x3-x1)),2.0),0.5);


	if(Jdet<0){
		printf("%s%s\n",__FUNCT__," error message: negative jacobian determinant!");
	}
	
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetB"

void Tria::GetB(double* B, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For grid i, Bi can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_basic=[ dh/dx    0    ]
	 *                [   0    dh/dy  ]
	 *                [ 1/2*dh/dy  1/2*dh/dx  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF2*numgrids)
	 */
	
	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double dh1dh2dh3_basic[NDOF2][numgrids];


	/*Get dh1dh2dh3 in basic coordinate system: */
	GetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],xyz_list, gauss_l1l2l3);

	#ifdef _ISSM_DEBUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  dh/dx=%lf dh/dy=%lf \n",i,dh1dh2dh3_basic[0][i],dh1dh2dh3_basic[1][i]);
	}
	#endif

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh2dh3_basic[0][i]; //B[0][NDOF2*i]=dh1dh2dh3_basic[0][i];
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh2dh3_basic[1][i];
		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh2dh3_basic[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh2dh3_basic[0][i]; 
	}
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetBPrime"

void Tria::GetBPrime(double* Bprime, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B'  matrix. B'=[B1' B2' B3'] where Bi' is of size 3*NDOF2. 
	 * For grid i, Bi' can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_prime__basic=[ 2*dh/dx dh/dy ]
	 *                       [ dh/dx  2*dh/dy]
	 *                       [dh/dy dh/dx]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B' has been allocated already, of size: 3x(NDOF2*numgrids)
	 */
	
	int i;
	const int NDOF2=2;
	const int numgrids=3;

	/*Same thing in the basic coordinate system: */
	double dh1dh2dh3_basic[NDOF2][numgrids];


	/*Get dh1dh2dh3 in basic coordinates system : */
	GetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],xyz_list,gauss_l1l2l3);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime+NDOF2*numgrids*0+NDOF2*i)=2*dh1dh2dh3_basic[0][i]; 
		*(Bprime+NDOF2*numgrids*0+NDOF2*i+1)=dh1dh2dh3_basic[1][i]; 
		*(Bprime+NDOF2*numgrids*1+NDOF2*i)=dh1dh2dh3_basic[0][i]; 
		*(Bprime+NDOF2*numgrids*1+NDOF2*i+1)=2*dh1dh2dh3_basic[1][i]; 
		*(Bprime+NDOF2*numgrids*2+NDOF2*i)=dh1dh2dh3_basic[1][i]; 
		*(Bprime+NDOF2*numgrids*2+NDOF2*i+1)=dh1dh2dh3_basic[0][i]; 
	}
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetL"

void Tria::GetL(double* L, double* xyz_list, double* gauss_l1l2l3,int numdof){

	/*Compute L  matrix. L=[L1 L2 L3] where Li is square and of size numdof. 
	 * For grid i, Li can be expressed in the basic coordinate system
	 * by: 
	 *       numdof=1: 
	 *       Li_basic=h;
	 *       numdof=2:
	 *       Li_basic=[ h    0    ]
	 *                [   0   h  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume L has been allocated already, of size: numgrids (numdof=1), or numdofx(numdof*numgrids) (numdof=2)
	 */
	
	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double l1l2l3[3];


	/*Get l1l2l3 in basic coordinate system: */
	GetNodalFunctions(l1l2l3, gauss_l1l2l3);

	#ifdef _DELUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  h=%lf \n",i,l1l2l3[i]);
	}
	#endif

	/*Build L: */
	if(numdof==1){
		for (i=0;i<numgrids;i++){
			L[i]=l1l2l3[i]; 
		}
	}
	else{
		for (i=0;i<numgrids;i++){
			*(L+numdof*numgrids*0+numdof*i)=l1l2l3[i]; //L[0][NDOF2*i]=dh1dh2dh3_basic[0][i];
			*(L+numdof*numgrids*0+numdof*i+1)=0;
			*(L+numdof*numgrids*1+numdof*i)=0;
			*(L+numdof*numgrids*1+numdof*i+1)=l1l2l3[i];
		}
	}
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetB_prog"

void Tria::GetB_prog(double* B_prog, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For grid i, Bi can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_basic=[ h ]
	 *                [ h ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B_prog has been allocated already, of size: 2x(NDOF1*numgrids)
	 */
	
	int i;
	const int NDOF1=1;
	const int numgrids=3;

	double l1l2l3[numgrids];


	/*Get dh1dh2dh3 in basic coordinate system: */
	GetNodalFunctions(&l1l2l3[0],gauss_l1l2l3);

	#ifdef _ISSM_DEBUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  h=%lf \n",i,l1l2l3[i]);
	}
	#endif

	/*Build B_prog: */
	for (i=0;i<numgrids;i++){
		*(B_prog+NDOF1*numgrids*0+NDOF1*i)=l1l2l3[i];
		*(B_prog+NDOF1*numgrids*1+NDOF1*i)=l1l2l3[i];
	}
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetBPrime_prog"

void Tria::GetBPrime_prog(double* Bprime_prog, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B'  matrix. B'=[B1' B2' B3'] where Bi' is of size 3*NDOF2. 
	 * For grid i, Bi' can be expressed in the basic coordinate system
	 * by: 
	 *       Bi_prime__basic=[ dh/dx ]
	 *                       [ dh/dy ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B' has been allocated already, of size: 3x(NDOF2*numgrids)
	 */
	
	int i;
	const int NDOF1=1;
	const int NDOF2=2;
	const int numgrids=3;

	/*Same thing in the basic coordinate system: */
	double dh1dh2dh3_basic[NDOF2][numgrids];

	/*Get dh1dh2dh3 in basic coordinates system : */
	GetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],xyz_list,gauss_l1l2l3);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime_prog+NDOF1*numgrids*0+NDOF1*i)=dh1dh2dh3_basic[0][i]; 
		*(Bprime_prog+NDOF1*numgrids*1+NDOF1*i)=dh1dh2dh3_basic[1][i]; 
	}
}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetNodalFunctions"
void Tria::GetNodalFunctions(double* l1l2l3, double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2l3[0]=gauss_l1l2l3[0];

	/*Second nodal function: */
	l1l2l3[1]=gauss_l1l2l3[1];

	/*Third nodal function: */
	l1l2l3[2]=gauss_l1l2l3[2];

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetNodalFunctionsDerivativesBasic"
void Tria::GetNodalFunctionsDerivativesBasic(double* dh1dh2dh3_basic,double* xyz_list, double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * basic coordinate system: */

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double dh1dh2dh3_param[NDOF2][numgrids];
	double Jinv[NDOF2][NDOF2];


	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsDerivativesParams(&dh1dh2dh3_param[0][0], gauss_l1l2l3); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss_l1l2l3);

	/*Build dh1dh2dh3_basic: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh2dh3_basic+numgrids*0+i)=Jinv[0][0]*dh1dh2dh3_param[0][i]+Jinv[0][1]*dh1dh2dh3_param[1][i];
		*(dh1dh2dh3_basic+numgrids*1+i)=Jinv[1][0]*dh1dh2dh3_param[0][i]+Jinv[1][1]*dh1dh2dh3_param[1][i];
	}

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetNodalFunctionsDerivativesParams"
void Tria::GetNodalFunctionsDerivativesParams(double* dl1dl2dl3,double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. */

	const int NDOF2=2;
	const int numgrids=3;

	double sqrt3=sqrt(3.0);

	/*First nodal function: */
	*(dl1dl2dl3+numgrids*0+0)=-1.0/2.0; 
	*(dl1dl2dl3+numgrids*1+0)=-1.0/(2.0*sqrt3);

	/*Second nodal function: */
	*(dl1dl2dl3+numgrids*0+1)=1.0/2.0;
	*(dl1dl2dl3+numgrids*1+1)=-1.0/(2.0*sqrt3);

	/*Third nodal function: */
	*(dl1dl2dl3+numgrids*0+2)=0;
	*(dl1dl2dl3+numgrids*1+2)=1.0/sqrt3;

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetJacobianInvert"
void Tria::GetJacobianInvert(double*  Jinv, double* xyz_list,double* gauss_l1l2l3){

	double Jdet;
	const int NDOF2=2;
	const int numgrids=3;
	
	/*Call Jacobian routine to get the jacobian:*/
	GetJacobian(Jinv, xyz_list, gauss_l1l2l3);

	/*Invert Jacobian matrix: */
	MatrixInverse(Jinv,NDOF2,NDOF2,NULL,0,&Jdet);
		
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetJacobian"
void Tria::GetJacobian(double* J, double* xyz_list,double* gauss_l1l2l3){

	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	const int NDOF2=2;
	const int numgrids=3;
	double x1,y1,x2,y2,x3,y3;
	double sqrt3=sqrt(3.0);
	
	x1=*(xyz_list+numgrids*0+0);
	y1=*(xyz_list+numgrids*0+1);
	x2=*(xyz_list+numgrids*1+0);
	y2=*(xyz_list+numgrids*1+1);
	x3=*(xyz_list+numgrids*2+0);
	y3=*(xyz_list+numgrids*2+1);


	*(J+NDOF2*0+0)=1.0/2.0*(x2-x1);
	*(J+NDOF2*1+0)=sqrt3/6.0*(2*x3-x1-x2);
	*(J+NDOF2*0+1)=1.0/2.0*(y2-y1);
	*(J+NDOF2*1+1)=sqrt3/6.0*(2*y3-y1-y2);
}

void* Tria::GetMatPar(){
	return matpar;
}

int   Tria::GetShelf(){
	return shelf;
}

		
void  Tria::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=(Node**)vpnodes;

	for(i=0;i<3;i++){
		pnodes[i]=nodes[i];
	}
}
		

int Tria::GetOnBed(){
	return onbed;
}

void          Tria::GetThicknessList(double* thickness_list){

	int i;
	for(i=0;i<3;i++)thickness_list[i]=h[i];
}
void          Tria::GetBedList(double* bed_list){
	
	int i;
	for(i=0;i<3;i++)bed_list[i]=b[i];

}
  

Object* Tria::copy() {
	
	return new Tria(*this); 

}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::Du"
void Tria::Du(Vec du_g,void* vinputs,int analysis_type,int sub_analysis_type){

	int i;
	
	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	int          dofs2[2]={0,1};

	/* grid data: */
	double vxvy_list[numgrids][2];
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vxvy_list[numgrids][2];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];
	double absolutex_list[numgrids];
	double absolutey_list[numgrids];
	double relativex_list[numgrids];
	double relativey_list[numgrids];
	double logarithmicx_list[numgrids];
	double logarithmicy_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  obs_velocity_mag,velocity_mag;
	double  absolutex,absolutey,relativex,relativey,logarithmicx,logarithmicy;

	/*element vector : */
	double  due_g[numdof];
	double  due_g_gaussian[numdof];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*relative and algorithmic fitting: */
	double scalex=0;
	double scaley=0;
	double scale=0;
	double  fit=-1;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set due_g to 0: */
	for(i=0;i<numdof;i++) due_g[i]=0.0;

	/* Recover input data: */
	if(!inputs->Recover("fit",&fit)) throw ErrorException(__FUNCT__," missing fit input parameter");
	if(!inputs->Recover("velocity_obs",&obs_vxvy_list[0][0],2,dofs2,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing velocity_obs input parameter");
	}
	if(!inputs->Recover("velocity",&vxvy_list[0][0],2,dofs2,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing velocity input parameter");
	}

	for(i=0;i<numgrids;i++){
		obs_vx_list[i]=obs_vxvy_list[i][0];
		obs_vy_list[i]=obs_vxvy_list[i][1];
		vx_list[i]=vxvy_list[i][0];
		vy_list[i]=vxvy_list[i][1];
	}

	/*Get Du at the 3 nodes (integration of the linearized function)*/
	if(fit==0){
		/*We are using an absolute misfit: */
		for (i=0;i<numgrids;i++){
			absolutex_list[i]=obs_vx_list[i]-vx_list[i];
			absolutey_list[i]=obs_vy_list[i]-vy_list[i];
		}
	}
	else if(fit==1){
		/*We are using a relative misfit: */
		for (i=0;i<numgrids;i++){
			scalex=pow(numpar->meanvel/(obs_vx_list[i]+numpar->epsvel),2);
			scaley=pow(numpar->meanvel/(obs_vy_list[i]+numpar->epsvel),2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			relativex_list[i]=scalex*(obs_vx_list[i]-vx_list[i]);
			relativey_list[i]=scaley*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(fit==2){
		/*We are using a logarithmic misfit: */
		for (i=0;i<numgrids;i++){
			velocity_mag=sqrt(pow(vx_list[i],2)+pow(vy_list[i],2))+numpar->epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],2)+pow(obs_vy_list[i],2))+numpar->epsvel; //epsvel to avoid observed velocity being nil.
			scale=-8*pow(numpar->meanvel,2)/pow(velocity_mag,2)*log(velocity_mag/obs_velocity_mag);
			logarithmicx_list[i]=scale*vx_list[i];
			logarithmicy_list[i]=scale*vy_list[i];
		}
	}
	else{
		/*Not supported yet! : */
		throw ErrorException(__FUNCT__,exprintf("%s%g","unsupported type of fit: ",fit));
	}
	
	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("   gaussian points: \n");
		for(i=0;i<num_gauss;i++){
			printf("    %g %g %g : %g\n",first_gauss_area_coord[i],second_gauss_area_coord[i],third_gauss_area_coord[i],gauss_weights[i]);
		}
	}
	#endif
	
	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		#ifdef _ISSM_DEBUG_ 
		printf("Element id %i Jacobian determinant: %g\n",GetId(),Jdet);
		#endif
		
		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Build due_g_gaussian vector: we have three cases here, according to which type of misfit we are using. */
		if(fit==0){
			/*We are using an absolute misfit: */

			/*Compute absolute(x/y) at gaussian point: */
			GetParameterValue(&absolutex, &absolutex_list[0],gauss_l1l2l3);
			GetParameterValue(&absolutey, &absolutey_list[0],gauss_l1l2l3);

			/*compute Du*/
			for (i=0;i<numgrids;i++){
				due_g_gaussian[i*NDOF2+0]=absolutex*Jdet*gauss_weight*l1l2l3[i]; 
				due_g_gaussian[i*NDOF2+1]=absolutey*Jdet*gauss_weight*l1l2l3[i]; 
			}
		}
		else if(fit==1){
			/*We are using a relative misfit: */

			/*Compute relative(x/y) at gaussian point: */
			GetParameterValue(&relativex, &relativex_list[0],gauss_l1l2l3);
			GetParameterValue(&relativey, &relativey_list[0],gauss_l1l2l3);

			/*compute Du*/
			for (i=0;i<numgrids;i++){
				due_g_gaussian[i*NDOF2+0]=relativex*Jdet*gauss_weight*l1l2l3[i]; 
				due_g_gaussian[i*NDOF2+1]=relativey*Jdet*gauss_weight*l1l2l3[i]; 
			}
		}
		else if(fit==2){
			/*We are using a logarithmic misfit: */

			/*Compute logarithmic(x/y) at gaussian point: */
			GetParameterValue(&logarithmicx, &logarithmicx_list[0],gauss_l1l2l3);
			GetParameterValue(&logarithmicy, &logarithmicy_list[0],gauss_l1l2l3);

			/*compute Du*/
			for (i=0;i<numgrids;i++){
				due_g_gaussian[i*NDOF2+0]=logarithmicx*Jdet*gauss_weight*l1l2l3[i]; 
				due_g_gaussian[i*NDOF2+1]=logarithmicy*Jdet*gauss_weight*l1l2l3[i]; 
			}
		}
		else{
			/*Not supported yet! : */
			throw ErrorException(__FUNCT__,exprintf("%s%g","unsupported type of fit: ",fit));
		}
		
		/*Add due_g_gaussian vector to due_g: */
		for( i=0; i<numdof; i++){
			due_g[i]+=due_g_gaussian[i];
		}
	}
	
	/*Add due_g to global vector du_g: */
	VecSetValues(du_g,numdof,doflist,(const double*)due_g,ADD_VALUES);
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
#undef __FUNCT__ 
#define __FUNCT__ "Tria::Gradj"
void  Tria::Gradj(Vec grad_g,void* inputs,int analysis_type,int sub_analysis_type,char* control_type){

	/*If on water, grad = 0: */
	if(onwater)return;

	if (strcmp(control_type,"drag")==0){
		GradjDrag( grad_g,inputs,analysis_type,sub_analysis_type);
	}
	else if (strcmp(control_type,"B")==0){
		GradjB( grad_g,inputs,analysis_type,sub_analysis_type);
	}
	else throw ErrorException(__FUNCT__,exprintf("%s%s","control type not supported yet: ",control_type));
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GradjDrag"
void  Tria::GradjDrag(Vec grad_g,void* vinputs,int analysis_type,int sub_analysis_type){


	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF2=2;
	const int    numdof=NDOF2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh2dh3_basic[NDOF2][numgrids];

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double vxvy_list[numgrids][2];
	double adjx_list[numgrids];
	double adjy_list[numgrids];
	double adjxadjy_list[numgrids][2];

	double drag;
	int    dofs1[1]={0};
	int    dofs2[2]={0,1};

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  dk[NDOF2]; 
	double  vx,vy;
	double  lambda,mu;
	double  bed,thickness,Neff;
	
	/*drag: */
	double  pcoeff,qcoeff;
	double alpha_complement_list[numgrids];
	double alpha_complement;

	/*element vector at the gaussian points: */
	double  grade_g[numgrids];
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*Get out if shelf*/
	if(shelf) return;

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/* Set grade_g to 0: */
	for(i=0;i<numgrids;i++) grade_g[i]=0.0;

	/* recover input parameters: */
	inputs->Recover("drag",&k[0],1,dofs1,numgrids,(void**)nodes);
	inputs->Recover("bed",&b[0],1,dofs1,numgrids,(void**)nodes);
	inputs->Recover("thickness",&h[0],1,dofs1,numgrids,(void**)nodes);
	if(!inputs->Recover("velocity",&vxvy_list[0][0],2,dofs2,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing velocity input parameter");
	}
	if(!inputs->Recover("adjoint",&adjxadjy_list[0][0],2,dofs2,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing adjoint input parameter");
	}

	/*Initialize parameter lists: */
	for(i=0;i<numgrids;i++){
		vx_list[i]=vxvy_list[i][0];
		vy_list[i]=vxvy_list[i][1];
		adjx_list[i]=adjxadjy_list[i][0];
		adjy_list[i]=adjxadjy_list[i][1];
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);

	#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("   gaussian points: \n");
		for(i=0;i<num_gauss;i++){
			printf("    %g %g %g : %g\n",first_gauss_area_coord[i],second_gauss_area_coord[i],third_gauss_area_coord[i],gauss_weights[i]);
		}
	}
	#endif
	
	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Build alpha_complement_list: */
		if (!shelf && (friction_type==2)){
		
			/*Allocate friction object: */
			Friction* friction=NewFriction();
			
			/*Initialize all fields: */
			friction->element_type=(char*)xmalloc((strlen("2d")+1)*sizeof(char));
			strcpy(friction->element_type,"2d");
			
			friction->gravity=matpar->GetG();
			friction->rho_ice=matpar->GetRhoIce();
			friction->rho_water=matpar->GetRhoWater();
			friction->K=&k[0];
			friction->bed=&b[0];
			friction->thickness=&h[0];
			friction->velocities=&vxvy_list[0][0];
			friction->p=p;
			friction->q=q;

			if(friction->p!=1) throw ErrorException(__FUNCT__,"non-linear friction not supported yet in control methods!");

			/*Compute alpha complement: */
			FrictionGetAlphaComplement(&alpha_complement_list[0],friction);

			/*Erase friction object: */
			DeleteFriction(&friction);
		}
		else{
			alpha_complement_list[0]=0;
			alpha_complement_list[1]=0;
			alpha_complement_list[2]=0;
		}
	
		/*Recover alpha_complement and k: */
		GetParameterValue(&alpha_complement, &alpha_complement_list[0],gauss_l1l2l3);
		GetParameterValue(&drag, &k[0],gauss_l1l2l3);
		#ifdef _ISSM_DEBUG_ 
			printf("Drag complement: %20.20lf Drag: %20.20lf\n",alpha_complement,drag);
		#endif

		/*recover lambda and mu: */
		GetParameterValue(&lambda, &adjx_list[0],gauss_l1l2l3);
		GetParameterValue(&mu, &adjy_list[0],gauss_l1l2l3);
		#ifdef _ISSM_DEBUG_ 
			printf("Adjoint vector %20.20lf %20.20lf\n",lambda,mu);
		#endif
			
		/*recover vx and vy: */
		GetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
		GetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);
		#ifdef _ISSM_DEBUG_ 
			printf("Velocity vector %20.20lf %20.20lf\n",vx,vy);
		#endif

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		#ifdef _ISSM_DEBUG_ 
		printf("Element id %i Jacobian determinant: %lf\n",GetId(),Jdet);
		#endif
		
		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get k derivative: dk/dx */
		GetParameterDerivativeValue(&dk[0], &k[0],&xyz_list[0][0], gauss_l1l2l3);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<numgrids;i++){

			//standard term dJ/dki
			grade_g_gaussian[i]=-2*drag*alpha_complement*((lambda*vx+mu*vy))*Jdet*gauss_weight*l1l2l3[i];

			//noise dampening d/dki(1/2*(dk/dx)^2)
			grade_g_gaussian[i]+=-numpar->cm_noisedmp*Jdet*gauss_weight*(dh1dh2dh3_basic[0][i]*dk[0]+dh1dh2dh3_basic[1][i]*dk[1]);
			
			//min dampening
			if(drag<numpar->cm_mindmp_value){ 
				grade_g_gaussian[i]+=numpar->cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(drag>numpar->cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - numpar->cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}
		}
		
		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<numgrids; i++)grade_g[i]+=grade_g_gaussian[i];
	}

	/*Add grade_g to global vector grad_g: */
	VecSetValues(grad_g,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GradjDragStokes"
void  Tria::GradjDragStokes(Vec grad_g,void* vinputs,int analysis_type,int sub_analysis_type){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh2dh3_basic[NDOF2][numgrids];

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double vz_list[numgrids];
	double vxvyvz_list[numgrids][3];
	double adjx_list[numgrids];
	double adjy_list[numgrids];
	double adjz_list[numgrids];
	double adjxyz_list[numgrids][3];

	double drag;
	int    dofs1[1]={0};
	int    dofs3[3]={0,1,2};

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  vx,vy,vz;
	double  lambda,mu,xi;
	double  bed,thickness,Neff;
	double  surface_normal[3];
	double  bed_normal[3];
	double  dk[NDOF2]; 

	/*drag: */
	double  pcoeff,qcoeff;
	double alpha_complement_list[numgrids];
	double alpha_complement;

	/*element vector at the gaussian points: */
	double  grade_g[numgrids];
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/* Set grade_g to 0: */
	for(i=0;i<numgrids;i++) grade_g[i]=0.0;

	/* recover input parameters: */
	inputs->Recover("drag",&k[0],1,dofs1,numgrids,(void**)nodes);
	inputs->Recover("bed",&b[0],1,dofs1,numgrids,(void**)nodes);
	inputs->Recover("thickness",&h[0],1,dofs1,numgrids,(void**)nodes);
	if(!inputs->Recover("velocity",&vxvyvz_list[0][0],3,dofs3,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing velocity input parameter");
	}
	if(!inputs->Recover("adjoint",&adjxyz_list[0][0],3,dofs3,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing adjoint input parameter");
	}

	/*Initialize parameter lists: */
	for(i=0;i<numgrids;i++){
		vx_list[i]=vxvyvz_list[i][0];
		vy_list[i]=vxvyvz_list[i][1];
		vz_list[i]=vxvyvz_list[i][2];
		adjx_list[i]=adjxyz_list[i][0];
		adjy_list[i]=adjxyz_list[i][1];
		adjz_list[i]=adjxyz_list[i][2];
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);

#ifdef _DEBUGELEMENTS_
	if(my_rank==RANK && id==ELID){ 
		printf("   gaussian points: \n");
		for(i=0;i<num_gauss;i++){
			printf("    %g %g %g : %g\n",first_gauss_area_coord[i],second_gauss_area_coord[i],third_gauss_area_coord[i],gauss_weights[i]);
		}
	}
#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Build alpha_complement_list: */
		if (!shelf && (friction_type==2)){

			/*Allocate friction object: */
			Friction* friction=NewFriction();

			/*Initialize all fields: */
			friction->element_type=(char*)xmalloc((strlen("2d")+1)*sizeof(char));
			strcpy(friction->element_type,"2d");

			friction->gravity=matpar->GetG();
			friction->rho_ice=matpar->GetRhoIce();
			friction->rho_water=matpar->GetRhoWater();
			friction->K=&k[0];
			friction->bed=&b[0];
			friction->thickness=&h[0];
			friction->velocities=&vxvyvz_list[0][0];
			friction->p=p;
			friction->q=q;

			if(friction->p!=1) throw ErrorException(__FUNCT__,"non-linear friction not supported yet in control methods!");

			/*Compute alpha complement: */
			FrictionGetAlphaComplement(&alpha_complement_list[0],friction);

			/*Erase friction object: */
			DeleteFriction(&friction);
		}
		else{
			alpha_complement_list[0]=0;
			alpha_complement_list[1]=0;
			alpha_complement_list[2]=0;
		}

		/*Recover alpha_complement and k: */
		GetParameterValue(&alpha_complement, &alpha_complement_list[0],gauss_l1l2l3);
		GetParameterValue(&drag, &k[0],gauss_l1l2l3);
#ifdef _ISSM_DEBUG_ 
		printf("Drag complement: %20.20lf Drag: %20.20lf\n",alpha_complement,drag);
#endif

		/*recover lambda mu and xi: */
		GetParameterValue(&lambda, &adjx_list[0],gauss_l1l2l3);
		GetParameterValue(&mu, &adjy_list[0],gauss_l1l2l3);
		GetParameterValue(&xi, &adjz_list[0],gauss_l1l2l3);
#ifdef _ISSM_DEBUG_ 
		printf("Adjoint vector %20.20lf %20.20lf\n",lambda,mu);
#endif

		/*recover vx vy and vz: */
		GetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
		GetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);
		GetParameterValue(&vz, &vz_list[0],gauss_l1l2l3);
#ifdef _ISSM_DEBUG_ 
		printf("Velocity vector %20.20lf %20.20lf\n",vx,vy);

		/*Get normal vecyor to the bed */
		SurfaceNormal(&surface_normal[0],xyz_list);

		bed_normal[0]=-surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
		bed_normal[1]=-surface_normal[1];
		bed_normal[2]=-surface_normal[2];
#endif

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
#ifdef _ISSM_DEBUG_ 
		printf("Element id %i Jacobian determinant: %lf\n",GetId(),Jdet);
#endif

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get k derivative: dk/dx */
		GetParameterDerivativeValue(&dk[0], &k[0],&xyz_list[0][0], gauss_l1l2l3);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<numgrids;i++){
			//standard gradient dJ/dki
			grade_g_gaussian[i]=(
						-lambda*(2*drag*alpha_complement*(vx - vz*bed_normal[0]*bed_normal[2]))
						-mu    *(2*drag*alpha_complement*(vy - vz*bed_normal[1]*bed_normal[2]))
						-xi    *(2*drag*alpha_complement*(-vx*bed_normal[0]*bed_normal[2]-vy*bed_normal[1]*bed_normal[2]))
						)*Jdet*gauss_weight*l1l2l3[i]; 

			//Add regularization term
			grade_g_gaussian[i]+= - numpar->cm_noisedmp*Jdet*gauss_weight*(dh1dh2dh3_basic[0][i]*dk[0]+dh1dh2dh3_basic[1][i]*dk[1]);

			//min dampening
			if(drag<numpar->cm_mindmp_value){ 
				grade_g_gaussian[i]+= numpar->cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(drag>numpar->cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - numpar->cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}
		}

		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<numgrids; i++)grade_g[i]+=grade_g_gaussian[i];
	}

	/*Add grade_g to global vector grad_g: */
	VecSetValues(grad_g,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::SurfaceNormal"

void Tria::SurfaceNormal(double* surface_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2) );

	*(surface_normal)=normal[0]/normal_norm;
	*(surface_normal+1)=normal[1]/normal_norm;
	*(surface_normal+2)=normal[2]/normal_norm;

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GradjB"
void  Tria::GradjB(Vec grad_g,void* vinputs,int analysis_type,int sub_analysis_type){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    NDOF2=2;
	const int    numdof=NDOF2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh2dh3_basic[NDOF2][numgrids];

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double vxvy_list[numgrids][2];
	double adjx_list[numgrids];
	double adjy_list[numgrids];
	double adjxadjy_list[numgrids][2];
	double B[numgrids];

	int    dofs1[1]={0};
	int    dofs2[2]={0,1};

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/*element vector at the gaussian points: */
	double  grade_g[numgrids];
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	/* parameters: */
	double  viscosity_complement;
	double  dvx[NDOF2];
	double  dvy[NDOF2]; 
	double  dadjx[NDOF2];
	double  dadjy[NDOF2];
	double  vx,vy;
	double  lambda,mu;
	double  thickness;
	int     dofs[1]={0};
	double  dB[NDOF2]; 
	double  B_gauss;
	
	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/* Set grade_g to 0: */
	for(i=0;i<numgrids;i++) grade_g[i]=0.0;

	/* recover input parameters: */
	inputs->Recover("thickness",&h[0],1,dofs,numgrids,(void**)nodes);
	if(!inputs->Recover("velocity",&vxvy_list[0][0],2,dofs2,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing velocity input parameter");
	}
	if(!inputs->Recover("adjoint",&adjxadjy_list[0][0],2,dofs2,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing adjoint input parameter");
	}
	if(!inputs->Recover("B",&B[0],1,dofs1,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"parameter B not found in input");
	}

	/*Initialize parameter lists: */
	for(i=0;i<numgrids;i++){
		vx_list[i]=vxvy_list[i][0];
		vy_list[i]=vxvy_list[i][1];
		adjx_list[i]=adjxadjy_list[i][0];
		adjy_list[i]=adjxadjy_list[i][1];
	}
	
	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);
	#ifdef _ISSM_DEBUG_ 
	for (i=0;i<num_gauss;i++){
		printf("Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(gauss_weights+i));
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Get thickness: */
		GetParameterValue(&thickness, &h[0],gauss_l1l2l3);
	
		/*Get strain rate, if velocity has been supplied: */
		GetStrainRate(&epsilon[0],&vxvy_list[0][0],&xyz_list[0][0],gauss_l1l2l3);
	
		/*Get viscosity complement: */
		matice->GetViscosityComplement(&viscosity_complement, &epsilon[0]);
		
		/*Get dvx, dvy, dadjx and dadjx: */
		GetParameterDerivativeValue(&dvx[0], &vx_list[0],&xyz_list[0][0], gauss_l1l2l3);
		GetParameterDerivativeValue(&dvy[0], &vy_list[0],&xyz_list[0][0], gauss_l1l2l3);
		GetParameterDerivativeValue(&dadjx[0], &adjx_list[0],&xyz_list[0][0], gauss_l1l2l3);
		GetParameterDerivativeValue(&dadjy[0], &adjy_list[0],&xyz_list[0][0], gauss_l1l2l3);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);
		#ifdef _ISSM_DEBUG_
			printf("viscositycomp %g thickness %g dvx [%g %g] dvy [%g %g]  dadjx [%g %g] dadjy[%g %g]\n",viscosity_complement,thickness,dvx[0],dvx[1],dvy[0],dvy[1],dadjx[0],dadjx[1],dadjy[0],dadjy[1]);
		#endif

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivativesBasic(&dh1dh2dh3_basic[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get B derivative: dB/dx */
		GetParameterDerivativeValue(&dB[0], &B[0],&xyz_list[0][0], gauss_l1l2l3);
		GetParameterValue(&B_gauss, &B[0],gauss_l1l2l3);

		/*Build gradje_g_gaussian vector (actually -dJ/dB): */
		for (i=0;i<numgrids;i++){
			//standard gradient dJ/dki
			grade_g_gaussian[i]=-viscosity_complement*thickness*( (2*dvx[0]+dvy[1])*2*dadjx[0]+(dvx[1]+dvy[0])*(dadjx[1]+dadjy[0])+(2*dvy[1]+dvx[0])*2*dadjy[1])*Jdet*gauss_weight*l1l2l3[i]; 

			//Add regularization term
			grade_g_gaussian[i]-=numpar->cm_noisedmp*Jdet*gauss_weight*(dh1dh2dh3_basic[0][i]*dB[0]+dh1dh2dh3_basic[1][i]*dB[1]);

			//min dampening
			if(B_gauss<numpar->cm_mindmp_value){ 
				grade_g_gaussian[i]+= numpar->cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(B_gauss>numpar->cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - numpar->cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

		}

		/*Add grade_g_gaussian to grade_g: */
		for( i=0; i<numgrids;i++) grade_g[i]+=grade_g_gaussian[i];
	}
	
	/*Add grade_g to global vector grad_g: */
	VecSetValues(grad_g,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::Misfit"
double Tria::Misfit(void* vinputs,int analysis_type,int sub_analysis_type){

	int i;
	
	/* output: */
	double Jelem=0;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	int          dofs1[1]={0};
	int          dofs2[2]={0,1};
	double       xyz_list[numgrids][3];

	/* grid data: */
	double vxvy_list[numgrids][2];
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vxvy_list[numgrids][2];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];
	double absolute_list[numgrids];
	double relative_list[numgrids];
	double logarithmic_list[numgrids];
	double B[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  k_gauss;
	double  B_gauss;

	/* parameters: */
	double  velocity_mag,obs_velocity_mag;
	double  absolute,relative,logarithmic;
	double  dk[NDOF2]; 
	double  dB[NDOF2]; 

	/* Jacobian: */
	double Jdet;
	
	/*relative and logarithmic control method :*/
	double scalex=1;
	double scaley=1;
	double  fit=-1;

	ParameterInputs* inputs=NULL;

	/*If on water, return 0: */
	if(onwater)return 0;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	
	/* Recover input data: */
	if(!inputs->Recover("fit",&fit)) throw ErrorException(__FUNCT__," missing fit input parameter");
	if(!inputs->Recover("velocity_obs",&obs_vxvy_list[0][0],2,dofs2,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing velocity_obs input parameter");
	}
	if(!inputs->Recover("velocity",&vxvy_list[0][0],2,dofs2,numgrids,(void**)nodes)){
		throw ErrorException(__FUNCT__,"missing velocity input parameter");
	}

	/*Initialize velocities: */
	for(i=0;i<numgrids;i++){
		obs_vx_list[i]=obs_vxvy_list[i][0];
		obs_vy_list[i]=obs_vxvy_list[i][1];
		vx_list[i]=vxvy_list[i][0];
		vy_list[i]=vxvy_list[i][1];
	}

	/*Compute Misfit at the 3 nodes (integration of the linearized function)*/
	if(fit==0){
		/*We are using an absolute misfit: */
		for (i=0;i<numgrids;i++){
			absolute_list[i]=0.5*(pow((vx_list[i]-obs_vx_list[i]),(double)2)+pow((vy_list[i]-obs_vy_list[i]),(double)2));
		}
	}
	else if(fit==1){
		/*We are using a relative misfit: */
		for (i=0;i<numgrids;i++){
			scalex=pow(numpar->meanvel/(obs_vx_list[i]+numpar->epsvel),(double)2);
			scaley=pow(numpar->meanvel/(obs_vy_list[i]+numpar->epsvel),(double)2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			relative_list[i]=0.5*(scalex*pow((vx_list[i]-obs_vx_list[i]),2)+scaley*pow((vy_list[i]-obs_vy_list[i]),2));
		}
	}
	else if(fit==2){
		/*We are using a logarithmic misfit: */
		for (i=0;i<numgrids;i++){
			velocity_mag=sqrt(pow(vx_list[i],(double)2)+pow(vy_list[i],(double)2))+numpar->epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],(double)2)+pow(obs_vy_list[i],(double)2))+numpar->epsvel; //epsvel to avoid observed velocity being nil.
			logarithmic_list[i]=4*pow(numpar->meanvel,(double)2)*pow(log(velocity_mag/obs_velocity_mag),(double)2);
		}
	}
	else{
		/*Not supported yet! : */
		throw ErrorException(__FUNCT__,exprintf("%s%g","unsupported type of fit: ",fit));
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);
	
	#ifdef _ISSM_DEBUG_ 
	for (i=0;i<num_gauss;i++){
		printf("Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(gauss_weights+i));
	}
	#endif

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		#ifdef _ISSM_DEBUG_ 
		printf("Element id %i Jacobian determinant: %lf\n",GetId(),Jdet);
		#endif
		
		/*Add dampening terms to misfit*/
		if (strcmp(numpar->control_type,"drag")==0){
			if (!shelf){

				//noise dampening
				GetParameterDerivativeValue(&dk[0], &k[0],&xyz_list[0][0], gauss_l1l2l3);
				Jelem+=numpar->cm_noisedmp*1/2*(pow(dk[0],2)+pow(dk[1],2))*Jdet*gauss_weight;

			}
		}
		else if (strcmp(numpar->control_type,"B")==0){
			if(!inputs->Recover("B",&B[0],1,dofs1,numgrids,(void**)nodes)){
				throw ErrorException(__FUNCT__,"parameter B not found in input");
			}
			//noise dampening
			GetParameterDerivativeValue(&dB[0], &B[0],&xyz_list[0][0], gauss_l1l2l3);
			Jelem+=numpar->cm_noisedmp*1/2*(pow(dB[0],2)+pow(dB[1],2))*Jdet*gauss_weight;

			//min dampening
			GetParameterValue(&B_gauss, &B[0],gauss_l1l2l3);
			if(B_gauss<numpar->cm_mindmp_value){ 
				Jelem+=numpar->cm_mindmp_slope*B_gauss*Jdet*gauss_weight;
			}

			//max dampening
			if(B_gauss>numpar->cm_maxdmp_value){ 
				Jelem+=numpar->cm_maxdmp_slope*B_gauss*Jdet*gauss_weight;
			}
		}
		else{
			throw ErrorException(__FUNCT__,exprintf("%s%s","unsupported control type: ",numpar->control_type));
		}

		/*Differents misfits are allowed: */
		if(fit==0){
			/*Compute absolute misfit at gaussian point: */
			GetParameterValue(&absolute, &absolute_list[0],gauss_l1l2l3);

			/*compute Misfit*/
			Jelem+=absolute*Jdet*gauss_weight;
		}
		else if(fit==1){
			/*Compute relative misfit at gaussian point: */
			GetParameterValue(&relative, &relative_list[0],gauss_l1l2l3);

			/*compute Misfit*/
			Jelem+=relative*Jdet*gauss_weight;
		}	
		else if(fit==2){
			/*Compute logarithmic misfit at gaussian point: */
			GetParameterValue(&logarithmic, &logarithmic_list[0],gauss_l1l2l3);

			/*compute Misfit*/
			Jelem+=logarithmic*Jdet*gauss_weight;
		}
		else throw ErrorException(__FUNCT__,exprintf("%s%i%s","fit type",fit," not supported yet!"));

	}
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

	/*Return: */
	return Jelem;
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::NodeConfiguration"
void  Tria::NodeConfiguration(int* tria_node_ids,Node* tria_nodes[3],int* tria_node_offsets){

	int i;
	for(i=0;i<3;i++){
		node_ids[i]=tria_node_ids[i];
		nodes[i]=tria_nodes[i];
		node_offsets[i]=tria_node_offsets[i];
	}

}
#undef __FUNCT__ 
#define __FUNCT__ "Tria::MaticeConfiguration"
void  Tria::MaticeConfiguration(Matice* tria_matice,int tria_matice_offset){
	matice=tria_matice;
	matice_offset=tria_matice_offset;
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::MaticeConfiguration"
void  Tria::MatparConfiguration(Matpar* tria_matpar,int tria_matpar_offset){

	matpar=tria_matpar;
	matpar_offset=tria_matpar_offset;

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::NumparConfiguration"
void  Tria::NumparConfiguration(Numpar* tria_numpar,int tria_numpar_offset){

	numpar=tria_numpar;
	numpar_offset=tria_numpar_offset;

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreateKMatrixDiagnosticSurfaceVert"
void  Tria::CreateKMatrixDiagnosticSurfaceVert(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){

	int i,j;
	
	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	
	/* surface normal: */
	double x4,y4,z4;
	double x5,y5,z5;
	double x6,y6,z6;
	double v46[3];
	double v56[3];
	double normal[3];
	double norm_normal;
	double nz;

	/*Matrices: */
	double DL_scalar;
	double L[3];
	double Jdet;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;
	
	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set Ke_gg to 0: */
	for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gg[i][j]=0.0;

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Build normal vector to the surface:*/
	
	x4=xyz_list[0][0];
	y4=xyz_list[0][1];
	z4=xyz_list[0][2];

	x5=xyz_list[1][0];
	y5=xyz_list[1][1];
	z5=xyz_list[1][2];

	x6=xyz_list[2][0];
	y6=xyz_list[2][1];
	z6=xyz_list[2][2];

	v46[0]=x4-x6;
	v46[1]=y4-y6;
	v46[2]=z4-z6;

	v56[0]=x5-x6;
	v56[1]=y5-y6;
	v56[2]=z5-z6;

	normal[0]=(y4-y6)*(z5-z6)-(z4-z6)*(y5-y6);
	normal[1]=(z4-z6)*(x5-x6)-(x4-x6)*(z5-z6);
	normal[2]=(x4-x6)*(y5-y6)-(y4-y6)*(x5-x6);

	norm_normal=sqrt(pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2));
	nz=1.0/norm_normal*normal[2];

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
	
		//Get L matrix if viscous basal drag present:
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

		/**********************Do not forget the sign**********************************/
		DL_scalar=- gauss_weight*Jdet*nz; 
		/******************************************************************************/
	
		/*  Do the triple producte tL*D*L: */
		TripleMultiply( L,1,3,1,
					&DL_scalar,1,1,0,
					L,1,3,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

		
	} //for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}
		

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreatePVectorDiagnosticBaseVert"
void  Tria::CreatePVectorDiagnosticBaseVert(Vec pg,void* vinputs,int analysis_type,int sub_analysis_type){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof];
	double  pe_g_gaussian[numdof];

	/* matrices: */
	double L[numgrids];

	/*input parameters for structural analysis (diagnostic): */
	double* velocity_param=NULL;
	double  vx_list[numgrids]={0,0,0};
	double  vy_list[numgrids]={0,0,0};
	double  vx,vy;
	double  meltingvalue;
	double  slope[2];
	double  dbdx,dbdy;
	int     dofs1[1]={0};
	int     dofs2[1]={1};

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* recover input parameters: */
	if(!inputs->Recover("velocity",&vx_list[0],1,dofs1,numgrids,(void**)nodes))throw ErrorException(__FUNCT__," cannot compute vertical velocity without horizontal velocity");
	    inputs->Recover("velocity",&vy_list[0],1,dofs2,numgrids,(void**)nodes);

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set pe_g to 0: */
	for(i=0;i<numdof;i++) pe_g[i]=0.0;

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*For icesheets: */
	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Get melting at gaussian point: */
		GetParameterValue(&meltingvalue, &melting[0],gauss_l1l2l3);

		/*Get velocity at gaussian point: */
		GetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
		GetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);

		/*Get bed slope: */
		GetParameterDerivativeValue(&slope[0], &b[0],&xyz_list[0][0], gauss_l1l2l3);
		dbdx=slope[0];
		dbdy=slope[1];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		//Get L matrix if viscous basal drag present:
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

		
		/*Build gaussian vector: */
		for(i=0;i<numgrids;i++){
			pe_g_gaussian[i]=-Jdet*gauss_weight*(vx*dbdx+vy*dbdy-meltingvalue)*L[i];
		}
	
		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];
	
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::ComputePressure"
void  Tria::ComputePressure(Vec pg){

	int i;
	const int numgrids=3;
	int doflist[numgrids];
	double pressure[numgrids];
	double rho_ice,g;

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	for(i=0;i<numgrids;i++){
		pressure[i]=rho_ice*g*h[i];
	}
	
	/*plug local pressure values into global pressure vector: */
	VecSetValues(pg,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreateKMatrixThermal"
void  Tria::CreateKMatrixThermal(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){

	int i,j;
	int found=0;
	
	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	double mixed_layer_capacity;
	double thermal_exchange_velocity;
	double rho_water;
	double rho_ice;
	double heatcapacity;
	double dt;

	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];

	/*matrices: */
	double  Jdet;
	double  K_terms[numdof][numdof]={0.0};
	double  Ke_gaussian[numdof][numdof]={0.0};
	double  l1l2l3[numgrids];
	double     tl1l2l3D[3];
	double  D_scalar;
	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover extra inputs from users, dt: */
	found=inputs->Recover("dt",&dt);
	if(!found)throw ErrorException(__FUNCT__," could not find dt in inputs!");

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	mixed_layer_capacity=matpar->GetMixedLayerCapacity();
	thermal_exchange_velocity=matpar->GetThermalExchangeVelocity();
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();


	GaussTria (&num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss (nodes on the bedrock) */
	for (ig=0; ig<num_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);
		
		//Get the Jacobian determinant
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0], gauss_coord);
		
		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);
				
		/*Calculate DL on gauss point */
		D_scalar=gauss_weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_velocity/(heatcapacity*rho_ice);
		if(dt){
			D_scalar=dt*D_scalar;
		}

		/*  Do the triple product tL*D*L: */
		MatrixMultiply(&l1l2l3[0],numdof,1,0,&D_scalar,1,1,0,&tl1l2l3D[0],0);
		MatrixMultiply(&tl1l2l3D[0],numdof,1,0,&l1l2l3[0],1,numdof,0,&Ke_gaussian[0][0],0);

		for(i=0;i<3;i++){
			for(j=0;j<3;j++){
				K_terms[i][j]+=Ke_gaussian[i][j];
			}
		}
	}
	
	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

	cleanup_and_return:
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreateKMatrixMelting"
void  Tria::CreateKMatrixMelting(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){
	
	/*indexing: */
	int i,j;

	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;

	/*Grid data: */
	double     xyz_list[numgrids][3];
		
	/*Material constants */
	double     heatcapacity,latentheat;

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];

	/*matrices: */
	double     Jdet;
	double     D_scalar;
	double     K_terms[numdof][numdof]={0.0};
	double     L[3];
	double     tLD[3];
	double     Ke_gaussian[numdof][numdof]={0.0};

	/*Recover constants of ice */
	latentheat=matpar->GetLatentHeat();
	heatcapacity=matpar->GetHeatCapacity();

	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights: */
	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss  (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);
		
		//Get the Jacobian determinant
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0], gauss_coord);
		
		/*Get L matrix : */
		GetL(&L[0], &xyz_list[0][0], gauss_coord,NDOF1);
				
		/*Calculate DL on gauss point */
		D_scalar=latentheat/heatcapacity*gauss_weight*Jdet;

		/*  Do the triple product tL*D*L: */
		MatrixMultiply(&L[0],numdof,1,0,&D_scalar,1,1,0,&tLD[0],0);
		MatrixMultiply(&tLD[0],numdof,1,0,&L[0],1,numdof,0,&Ke_gaussian[0][0],0);

		for(i=0;i<numgrids;i++){
			for(j=0;j<numgrids;j++){
			K_terms[i][j]+=Ke_gaussian[i][j];
			}
		}
	}
	
	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

	cleanup_and_return:
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}


#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreatePVectorThermalShelf"
void Tria::CreatePVectorThermalShelf( Vec pg, void* vinputs, int analysis_type,int sub_analysis_type){

	int i,found;
	
	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;
	double       xyz_list[numgrids][3];

	double mixed_layer_capacity;
	double thermal_exchange_velocity;
	double rho_water;
	double rho_ice;
	double heatcapacity;
	double beta;
	double meltingpoint;

	/*inputs: */
	double dt;
	double pressure_list[3];
	double pressure;

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];
	int     dofs1[1]={0};

	/*matrices: */
	double  Jdet;
	double  P_terms[numdof]={0.0};
	double  l1l2l3[numgrids];

	double  t_pmp;
	double  scalar_ocean;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;
	
	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	mixed_layer_capacity=matpar->GetMixedLayerCapacity();
	thermal_exchange_velocity=matpar->GetThermalExchangeVelocity();
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();
	beta=matpar->GetBeta();
	meltingpoint=matpar->GetMeltingPoint();


	/*recover extra inputs from users, dt and velocity: */
	found=inputs->Recover("dt",&dt);
	if(!found)throw ErrorException(__FUNCT__," could not find dt in inputs!");
	found=inputs->Recover("pressure",&pressure_list[0],1,dofs1,numgrids,(void**)nodes);
	if(!found)throw ErrorException(__FUNCT__," could not find pressure in inputs!");

	/* Ice/ocean heat exchange flux on ice shelf base */

	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);

		//Get the Jacobian determinant
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0], gauss_coord);

		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);

		/*Get geothermal flux and basal friction */
		GetParameterValue(&pressure,&pressure_list[0],gauss_coord);
		t_pmp=meltingpoint-beta*pressure;

		/*Calculate scalar parameter*/
		scalar_ocean=gauss_weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_velocity*(t_pmp)/(heatcapacity*rho_ice);
		if(dt){
			scalar_ocean=dt*scalar_ocean;
		}

		for(i=0;i<3;i++){
			P_terms[i]+=scalar_ocean*l1l2l3[i];
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
#undef __FUNCT__ 
#define __FUNCT__ "Tria::CreatePVectorThermalSheet"
void Tria::CreatePVectorThermalSheet( Vec pg, void* vinputs, int analysis_type,int sub_analysis_type){

	int i,found;
	
	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;
	double       xyz_list[numgrids][3];
	double     vxvyvz_list[numgrids][3];
	double     vx_list[numgrids];
	double     vy_list[numgrids];

	double rho_ice;
	double heatcapacity;

	/*inputs: */
	double dt;
	double pressure_list[3];
	double pressure;
	double alpha2_list[3];
	double basalfriction_list[3];
	double basalfriction;
	double geothermalflux_value;

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];
	int     dofs1[1]={0};

	/*matrices: */
	double  Jdet;
	double  P_terms[numdof]={0.0};
	double  l1l2l3[numgrids];
	double  scalar;

	int     dofs[3]={0,1,2};

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;
	
	/* Get node coordinates and dof list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();


	/*recover extra inputs from users, dt and velocity: */
	found=inputs->Recover("dt",&dt);
	if(!found)throw ErrorException(__FUNCT__," could not find dt in inputs!");
	
	found=inputs->Recover("velocity",&vxvyvz_list[0][0],3,dofs,numgrids,(void**)nodes);
	if(!found)throw ErrorException(__FUNCT__," could not find velocity in inputs!");

	for(i=0;i<numgrids;i++){
		vx_list[i]=vxvyvz_list[i][0];
		vy_list[i]=vxvyvz_list[i][1];
	}	

	/*Build alpha2_list used by drag stiffness matrix*/
	Friction* friction=NewFriction();
	
	/*Initialize all fields: */
	if (friction_type!=2)throw ErrorException(__FUNCT__," non-viscous friction not supported yet!");
	
	friction->element_type=(char*)xmalloc((strlen("3d")+1)*sizeof(char));
	strcpy(friction->element_type,"3d");
	
	friction->gravity=matpar->GetG();
	friction->rho_ice=matpar->GetRhoIce();
	friction->rho_water=matpar->GetRhoWater();
	friction->K=&k[0];
	friction->bed=&b[0];
	friction->thickness=&h[0];
	friction->velocities=&vxvyvz_list[0][0];
	friction->p=p;
	friction->q=q;

	/*Compute alpha2_list: */
	FrictionGetAlpha2(&alpha2_list[0],friction);

	/*Erase friction object: */
	DeleteFriction(&friction);

	/* Compute basal friction */
	for(i=0;i<numgrids;i++){
		basalfriction_list[i]= alpha2_list[i]*(pow(vx_list[i],(double)2.0)+pow(vy_list[i],(double)2.0));
	}
	
	/* Ice/ocean heat exchange flux on ice shelf base */
	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);

		//Get the Jacobian determinant
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0], gauss_coord);

		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);

		/*Get geothermal flux and basal friction */
		GetParameterValue(&geothermalflux_value,&geothermalflux[0],gauss_coord);
		GetParameterValue(&basalfriction,&basalfriction_list[0],gauss_coord);

		/*Calculate scalar parameter*/
		scalar=gauss_weight*Jdet*(basalfriction+geothermalflux_value)/(heatcapacity*rho_ice);
		if(dt){
			scalar=dt*scalar;
		}

		for(i=0;i<3;i++){
			P_terms[i]+=scalar*l1l2l3[i];
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::MassFlux"
double Tria::MassFlux( double* segment,double* ug){

	int i;

	const int    numgrids=3;
	const int    numdofs=2;
	int          numberofdofspernode;
	double mass_flux=0;
	int    doflist[numgrids*numdofs];
	double vx_list[3];
	double vy_list[3];
	double xyz_list[numgrids][3];
	double gauss_1[3];
	double gauss_2[3];
	double normal[2];
	double length;
	double x1,y1,x2,y2;
	double h1,h2;
	double vx1,vx2,vy1,vy2;
	double rho_ice;
	
	/*Get material parameters :*/
	rho_ice=this->matpar->GetRhoIce();

	/*First off, check that this segment belongs to this element: */
	if ((int)*(segment+4)!=this->id)throw ErrorException(__FUNCT__,exprintf("%s%i%s%i","error message: segment with id ",(int)*(segment+4)," does not belong to element with id:",this->id));

	/*Recover segment node locations: */
	x1=*(segment+0); y1=*(segment+1); x2=*(segment+2); y2=*(segment+3);
	
	/*Get xyz list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);

	/*recover velocity at three element nodes: */
	this->GetDofList(&doflist[0],&numberofdofspernode);
	for(i=0;i<3;i++){
		vx_list[i]=ug[doflist[numberofdofspernode*i+0]];
		vy_list[i]=ug[doflist[numberofdofspernode*i+1]];
	}

	/*get area coordinates of 0 and 1 locations: */
	for(i=0;i<3;i++){
		gauss_1[i]=this->GetAreaCoordinate(x1,y1,i+1);
		gauss_2[i]=this->GetAreaCoordinate(x2,y2,i+1);
	}

	/*get normal of segment: */
	normal[0]=cos(atan2(x1-x2,y2-y1));
	normal[1]=sin(atan2(x1-x2,y2-y1));

	/*get length of segment: */
	length=sqrt(pow(x2-x1,2.0)+pow(y2-y1,2));

	/*get thickness and velocity at two segment extremities: */
	GetParameterValue(&h1, &h[0],gauss_1);
	GetParameterValue(&h2, &h[0],gauss_2);
	GetParameterValue(&vx1, &vx_list[0],gauss_1);
	GetParameterValue(&vy1, &vy_list[0],gauss_1);
	GetParameterValue(&vx2, &vx_list[0],gauss_2);
	GetParameterValue(&vy2, &vy_list[0],gauss_2);


	mass_flux= rho_ice*length*(  
				  (1.0/3.0*(h1-h2)*(vx1-vx2)+1.0/2.0*h2*(vx1-vx2)+1.0/2.0*(h1-h2)*vx2+h2*vx2)*normal[0]+
				  (1.0/3.0*(h1-h2)*(vy1-vy2)+1.0/2.0*h2*(vy1-vy2)+1.0/2.0*(h1-h2)*vy2+h2*vy2)*normal[1]
				);
	return mass_flux;
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetArea"
double Tria::GetArea(void){

	double area=0;
	const int    numgrids=3;
	double xyz_list[numgrids][3];
	double x1,y1,x2,y2,x3,y3;

	/*Get xyz list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	x1=xyz_list[0][0]; y1=xyz_list[0][1];
	x2=xyz_list[1][0]; y2=xyz_list[1][1];
	x3=xyz_list[2][0]; y3=xyz_list[2][1];
 
	return x2*y3 - y2*x3 + x1*y2 - y1*x2 + x3*y1 - y3*x1;
}

#undef __FUNCT__ 
#define __FUNCT__ "Tria::GetAreaCoordinate"
double Tria::GetAreaCoordinate(double x, double y, int which_one){

	double area=0;
	const int    numgrids=3;
	double xyz_list[numgrids][3];
	double x1,y1,x2,y2,x3,y3;

	/*Get area: */
	area=this->GetArea();

	/*Get xyz list: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	x1=xyz_list[0][0]; y1=xyz_list[0][1];
	x2=xyz_list[1][0]; y2=xyz_list[1][1];
	x3=xyz_list[2][0]; y3=xyz_list[2][1];

	if(which_one==1){
		/*Get first area coordinate = det(x-x3  x2-x3 ; y-y3   y2-y3)/area*/
		return ((x-x3)*(y2-y3)-(x2-x3)*(y-y3))/area;
	}
	else if(which_one==2){
		/*Get second area coordinate = det(x1-x3  x-x3 ; y1-y3   y-y3)/area*/
		return ((x1-x3)*(y-y3)-(x-x3)*(y1-y3))/area;
	}
	else if(which_one==3){
		/*Get third  area coordinate 1-area1-area2: */
		return 1-((x-x3)*(y2-y3)-(x2-x3)*(y-y3))/area -((x1-x3)*(y-y3)-(x-x3)*(y1-y3))/area;
	}
	else throw ErrorException(__FUNCT__,exprintf("%s%i%s\n"," error message: area coordinate ",which_one," done not exist!"));
}


