#!/bin/bash

################################################################################
# To be used after running,
#
#	${ISSM_DIR}/jenkins/jenkins.sh ${ISSM_DIR}/jenkins/ross-debian_linux-binaries-matlab
#
# in the context of a Jenkins project.
#
# When no runtime errors occur, performs the following:
# - Checks resulting executables and libraries against test suite.
# - Packages and compresses executables and libraries.
# - Transmits it to ISSM Web site for distribution.
#
# Options:
# -s/--skiptests		Skip tests (use if this script fails for some reason 
#						after tests have successfully passed to save time)
# -t/--transferonly		Transfer package to ISSM Web site only (use if 
#						transfer fails for some reason to skip testing and
#						signing)
################################################################################

# Expand aliases within the context of this script
shopt -s expand_aliases

## Override certain aliases
#
alias grep=$(which grep)

## Constants
#
LIBGFORTRAN="/usr/lib/x86_64-linux-gnu/libgfortran.so.5.0.0" # Important that this is the library itself
LIBGFORTRAN_DIST="${ISSM_DIR}/lib/libgfortran.so.5" # Important the file name matches the SONAME entry in the binaries and other shared libraries which link to it
PYTHON_NROPTIONS="--benchmark all --exclude 125 126 234 235 418 420 435 444 445 701 702 703 1101 1102 1103 1104 1105 1106 1107 1108 1109 1110 1201 1202 1203 1204 1205 1206 1207 1208 1301 1302 1303 1304 1401 1402 1601 1602 2002 2003 2005 2006 2010 2020 2021 2051 2052 2053 2101 2102 3001:3200 3201 3202 3300 3480 3481 4001 4002 4003" # NOTE: Combination of test suites from basic, Dakota, and Solid Earth builds, with tests that require a restart and those that require the JVM excluded
PKG="ISSM-Linux-Python" # Name of directory to copy distributable files to

COMPRESSED_PKG="${PKG}.tar.gz"

## Environment
#
export PATH="${ISSM_DIR}/bin:$(getconf PATH)" # Ensure that we pick up binaries from 'bin' directory rather than 'externalpackages'

## Parse options
#
if [ $# -gt 1 ]; then
	echo "Can use only one option at a time"
	exit 1
fi

skip_tests=0
transfer_only=0
while [ $# -gt 0 ]; do
    case $1 in
        -s|--skiptests) skip_tests=1; shift ;;
        -t|--transferonly) transfer_only=1; shift ;;
        *) echo "Unknown parameter passed: $1"; exit 1 ;;
    esac
    shift
done

if [ ${transfer_only} -eq 0 ]; then
	# Check if MATLAB exists
	if ! [ -d ${MATLAB_PATH} ]; then
		echo "${MATLAB_PATH} does not point to a MATLAB installation! Please modify MATLAB_PATH variable in $(basename $0) and try again."
		exit 1
	fi

	# Clean up from previous packaging
	echo "Cleaning up existing assets"
	cd ${ISSM_DIR}
	rm -rf ${PKG} ${COMPRESSED_PKG}
	mkdir ${PKG}

	# Add/modify required binaries and libraries
	cd ${ISSM_DIR}/bin

	echo "Modify generic"
	cat generic_static.py | sed -e "s/generic_static/generic/g" > generic.py

	echo "Moving MPICH binaries to bin/"
	if [ -f ${ISSM_DIR}/externalpackages/petsc/install/bin/mpiexec ]; then
		cp ${ISSM_DIR}/externalpackages/petsc/install/bin/mpiexec .
		cp ${ISSM_DIR}/externalpackages/petsc/install/bin/hydra_pmi_proxy .
	elif [ -f ${ISSM_DIR}/externalpackages/mpich/install/bin/mpiexec ]; then
		cp ${ISSM_DIR}/externalpackages/mpich/install/bin/mpiexec .
		cp ${ISSM_DIR}/externalpackages/mpich/install/bin/hydra_pmi_proxy .
	else
		echo "MPICH not found"
		exit 1
	fi

	echo "Moving GDAL binaries to bin/"
	if [ -f ${ISSM_DIR}/externalpackages/gdal/install/bin/gdal-config ]; then
		cp ${ISSM_DIR}/externalpackages/gdal/install/bin/gdalsrsinfo .
		cp ${ISSM_DIR}/externalpackages/gdal/install/bin/gdaltransform .
	else
		echo "GDAL not found"
		exit 1
	fi

	echo "Moving GMT binaries to bin/"
	if [ -f ${ISSM_DIR}/externalpackages/gmt/install/bin/gmt-config ]; then
		cp ${ISSM_DIR}/externalpackages/gmt/install/bin/gmt .
		cp ${ISSM_DIR}/externalpackages/gmt/install/bin/gmtselect .
	else
		echo "GMT not found"
		exit 1
	fi

	echo "Moving Gmsh binaries to bin/"
	if [ -f ${ISSM_DIR}/externalpackages/gmsh/install/bin/gmsh ]; then
		cp ${ISSM_DIR}/externalpackages/gmsh/install/bin/gmsh .
	else
		echo "Gmsh not found"
		exit 1
	fi

	echo "Moving libgfortran to lib/"
	cp ${LIBGFORTRAN} ${LIBGFORTRAN_DIST} 2> /dev/null


	# Run tests
	if [ ${skip_tests} -eq 0 ]; then
		echo "Running tests"
		cd ${ISSM_DIR}/test/NightlyRun
		rm python.log 2> /dev/null

		# Set Python environment
		export PYTHONPATH="${ISSM_DIR}/src/m/dev"
		export PYTHONSTARTUP="${PYTHONPATH}/devpath.py"
		export PYTHONUNBUFFERED=1 # We don't want Python to buffer output, otherwise issm.exe output is not captured

		# Run tests, redirecting output to logfile and suppressing output to console
		./runme.py ${PYTHON_NROPTIONS} &> python.log 2>&1

		# Check that Python did not exit in error
		pythonExitCode=`echo $?`
		pythonExitedInError=`grep -E "Error|Standard exception|Traceback|bad interpreter" python.log | wc -l`

		if [[ ${pythonExitCode} -ne 0 || ${pythonExitedInError} -ne 0 ]]; then
			echo "----------Python exited in error!----------"
			cat python.log
			echo "-----------End of python.log-----------"

			# Clean up execution directory
			rm -rf ${ISSM_DIR}/execution/*

			exit 1
		fi

		# Check that all tests passed
		numTestsFailed=`cat python.log | grep -c -e "FAILED|ERROR"`

		if [[ ${numTestsFailed} -ne 0 ]]; then
			echo "One or more tests FAILED"
			exit 1
		else
			echo "All tests PASSED"
		fi
	else
		echo "Skipping tests"
	fi

	# Create package
	cd ${ISSM_DIR}
	svn cleanup --remove-ignored --remove-unversioned test # Clean up test directory (before copying to package)
	echo "Copying assets to package: ${PKG}"
	cp -rf bin examples lib scripts test ${PKG}/
	mkdir ${PKG}/execution
	${ISSM_DIR}/scripts/py_to_pyc.sh ${PKG}/bin # Compile Python source files
	echo "Cleaning up unneeded/unwanted files"
	rm -f ${PKG}/bin/*.py # Remove all Python scripts
	rm -f ${PKG}/bin/generic_static.* # Remove static versions of generic cluster classes
	rm -f ${PKG}/lib/*.a # Remove static libraries from package
	rm -f ${PKG}/lib/*.la # Remove libtool libraries from package
	rm -rf ${PKG}/test/SandBox # Remove testing sandbox from package

	# Compress package
	echo "Compressing package"
	tar -czf ${COMPRESSED_PKG} ${PKG}
	ls -lah ${ISSM_DIR}/${COMPRESSED_PKG}
fi

# Transfer package to ISSM Web site
echo "Transferring package to ISSM Web site"
scp -i ~/.ssh/debian_linux-vm_to_ross ${COMPRESSED_PKG} jenkins@ross.ics.uci.edu:/var/www/html/${COMPRESSED_PKG}

if [ $? -ne 0 ]; then
	echo "Transfer failed! Verify connection then build this project again."
	exit 1
fi
