%DSL class definition
%
%   Usage:
%      dsl=dsl(); %dynamic sea level class, based on CMIP5 outputs

classdef dsl
	properties (SetAccess=public) 

		global_average_thermosteric_sea_level_change; %corresponds to zostoga field in CMIP5 archives. Specified as a temporally variable global rate (mm/yr)
		sea_surface_height_change_above_geoid; %corresponds to zos field in CMIP5 archives. Spatial average is 0. Specified as a spatio-temporally variable rate (mm/yr)
		sea_water_pressure_change_at_sea_floor; %corresponds to bpo field in CMIP5 archives. Specified as a spatio-temporally variable rate (in mm/yr equivalent, not in Pa/yr!) for each ensemble
		compute_fingerprints; %do we use the sea water pressure change to compute fingerprints and correct sea_surface_height_change_above_geoid

	end
	methods
		function self = extrude(self,md) % {{{
			self.sea_surface_height_change_above_geoid=project3d(md,'vector',self.sea_surface_height_change_above_geoid,'type','node','layer',1);
			self.sea_water_pressure_change_at_sea_floor=project3d(md,'vector',self.sea_water_pressure_change_at_sea_floor,'type','node','layer',1);
		end % }}}
		function self = dsl(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					self=structtoobj(dsl(),varargin{1});
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			self.global_average_thermosteric_sea_level_change=0;
			self.sea_surface_height_change_above_geoid=NaN;
			self.sea_water_pressure_change_at_sea_floor=NaN;
			self.compute_fingerprints=0;

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			%Early return
			if ~ismember('SealevelriseAnalysis',analyses), return; end
			if (strcmp(solution,'TransientSolution') & md.transient.isslr == 0), return; end
			md = checkfield(md,'fieldname','dsl.global_average_thermosteric_sea_level_change','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','dsl.sea_surface_height_change_above_geoid','NaN',1,'Inf',1,'timeseries',1);
			md = checkfield(md,'fieldname','dsl.sea_water_pressure_change_at_sea_floor','NaN',1,'Inf',1,'timeseries',1);
			md = checkfield(md,'fieldname','dsl.compute_fingerprints','NaN',1,'Inf',1,'values',[0,1]);
			if self.compute_fingerprints,
				%check geodetic flag of slr is on: 
				if md.slr.geodetic==0,
					error('DSL checkconsistency error message: if bottom pressure fingerprints computations are requested, slr class should have geodetic flag on');
				end
			end

		end % }}}
		function disp(self) % {{{

			disp(sprintf('   dsl parameters:'));
			fielddisplay(self,'global_average_thermosteric_sea_level_change','corresponds to zostoga field in CMIP5 archives. Specified as a temporally variable global rate (mm/yr)');
			fielddisplay(self,'sea_surface_height_change_above_geoid','corresponds to zos field in CMIP5 archives. Spatial average is 0. Specified as a spatio-temporally variable rate (mm/yr)');
			fielddisplay(self,'sea_water_pressure_change_at_sea_floor','corresponds to bpo field in CMIP5 archives. Specified as a spatio-temporally variable rate (in Pa/yr)');
			fielddisplay(self,'compute_fingerprints','%do we use the sea water pressure change to compute fingerprints and correct sea_surface_height_change_above_geoid');

		end % }}}
		function marshall(self,prefix,md,fid) % {{{

			WriteData(fid,prefix,'name','md.dsl.model','data',1,'format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','compute_fingerprints','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','global_average_thermosteric_sea_level_change','format','DoubleMat','mattype',1,'timeseries',1,'timeserieslength',2,'yts',md.constants.yts,'scale',1e-3/md.constants.yts);
			WriteData(fid,prefix,'object',self,'fieldname','sea_water_pressure_change_at_sea_floor','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts);
			WriteData(fid,prefix,'object',self,'fieldname','sea_surface_height_change_above_geoid','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts);

		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejs1Darray(fid,[modelname '.dsl.global_average_thermosteric_sea_level_change'],self.global_average_thermosteric_sea_level_change);
			writejs1Darray(fid,[modelname '.dsl.compute_fingerprints'],self.compute_fingerprints);
			writejs1Darray(fid,[modelname '.dsl.sea_surface_height_change_above_geoid'],self.sea_surface_height_change_above_geoid);
			writejs1Darray(fid,[modelname '.dsl.sea_water_pressure_change_at_sea_floor'],self.sea_water_pressure_change_at_sea_floor);

		end % }}}
	end
end
