function Names=FieldFindVarNames(filename)
%FIELDFINDVARNAMES - find names of variables in a data set file
%
%   This routines looks at the variables contained in a file and finds out
%   the names of the variables that are needed for an interpolation (x,y,data)
%   or (index,x,y,data)
%
%   Usage:
%      Names=FieldFindVarNames(filename)
%
%   Example:
%      Names=FieldFindVarNames('thickness.mat')
%
%   See also: INTERPFROMFILE, GRIDDATA

%some checks
if nargin~=1 | nargout~=1
	help FieldFindVarNames
	error('FieldFindVarNames error message: bad usage');
end
if ~exist(filename)
	error(['FieldFindVarNames error message: file ' filename  ' does not exist']);
end

%Get variables
A=whos('-file',filename);

%find x,y,vx and vy
xenum=NaN; yenum=NaN; dataenum=NaN; indexenum=NaN;
if length(A)==3,
	isgrid=1;
	for i=1:4
		if strcmpi(A(i).name(1),'x');
			xenum=i;
		elseif strcmpi(A(i).name(1),'y');
			yenum=i;
		elseif (strncmpi(A(i).name,filename,3) | strncmpi(A(i).name,'data',4)),
			dataenum=i;
		else
			%nothing
		end
	end
elseif length(A)==4,
	isgrid=0;
	for i=1:5
		if strcmpi(A(i).name(1),'x');
			xenum=i;
		elseif strcmpi(A(i).name(1),'y');
			yenum=i;
		elseif (strncmpi(A(i).name,'index',5) | strncmpi(A(i).name,'elements',7));
			indexenum=i;
		elseif (strncmpi(A(i).name,filename,3) | strncmpi(A(i).name,'data',4)),
			dataenum=i;
		else
			%nothing
		end
	end
else
	error(['FieldFindVarNames error message: file ' filename  ' not supported yet (it should hold 3 variables x,y and data (for grids) OR 4 variables  x,y,index and data (for mesh))']);
end

%2: if only one item is missing, find it by elimination
if ~isgrid,
	pos=find(isnan([xenum yenum indexenum dataenum]));
	if length(pos)==1,
		list=[xenum yenum indexenum dataenum]; list(pos)=[];
		if pos==1,
			xenum=setdiff(1:4,list);
		elseif pos==2,
			yenum=setdiff(1:4,list);
		elseif pos==3,
			indexenum=setdiff(1:4,list);
		elseif pos==4,
			dataenum=setdiff(1:4,list);
		end
	end
else
	pos=find(isnan([xenum yenum dataenum]));
	if length(pos)==1,
		list=[xenum yenum indexenum dataenum]; list(pos)=[];
		if pos==1,
			xenum=setdiff(1:3,list);
		elseif pos==2,
			yenum=setdiff(1:3,list);
		elseif pos==3,
			dataenum=setdiff(1:3,list);
		end
	end
end

%assum that we have found at least xenum and yenum
if ( isnan(xenum) | isnan(yenum))
	error(['FieldFindVarNames error message: file ' filename  ' not supported yet (the coordinates vectors should be named x and y)']);
end

%find index
if (~isgrid & isnan(indexenum)),
	for i=1:4
		lengthi=min(A(i).size);
		if (lengthi==3),
			indexenum=i;
		end
	end
	if isnan(indexenum),
		error(['FieldFindVarNames error message: file ' filename  ' not supported yet (index not found)']);
	end
end

%4: last chance
if ~isgrid,
	pos=find(isnan([xenum yenum indexenum dataenum]));
	if length(pos)==1,
		list=[xenum yenum indexenum dataenum]; list(pos)=[];
		if pos==1,
			xenum=setdiff(1:4,list);
		elseif pos==2,
			yenum=setdiff(1:4,list);
		elseif pos==3,
			indexenum=setdiff(1:4,list);
		elseif pos==4,
			dataenum=setdiff(1:4,list);
		end
	end
else
	pos=find(isnan([xenum yenum dataenum]));
	if length(pos)==1,
		list=[xenum yenum indexenum dataenum]; list(pos)=[];
		if pos==1,
			xenum=setdiff(1:3,list);
		elseif pos==2,
			yenum=setdiff(1:3,list);
		elseif pos==3,
			dataenum=setdiff(1:3,list);
		end
	end
end

%last check
if isnan(dataenum)
	error(['FieldFindVarNames error message: file ' filename  ' not supported yet (data not found)']);
end

%create output
Names=struct();
Names.xname=A(xenum).name;
Names.yname=A(yenum).name;
Names.dataname=A(dataenum).name;
if ~isgrid,
	Names.indexname=A(indexenum).name; 
	Names.interp='mesh';
else
	Names.interp='grid';
end
