%SOLIDEARTH class definition
%
%   Usage:
%      solidearth=solidearth();

classdef solidearth
	properties (SetAccess=public) 
		sealevel               = NaN;
		settings               = solidearthsettings(); 
		surfaceload            = surfaceload();
		lovenumbers            = lovenumbers();
		rotational             = rotational();
		planetradius           = planetradius('earth');
		requested_outputs      = {};
		transitions            = {};
		partitionice           = [];
		partitionhydro         = [];
	end
	methods
		function self = solidearth(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{
		
		%output default:
		self.requested_outputs={'default'};

		%transitions should be a cell array of vectors: 
		self.transitions={};
		
		%no partitions requested for barystatic contribution: 
		self.partitionice=[];
		self.partitionhydro=[];

		%earth radius
		self.planetradius= planetradius('earth');
		
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			if ~ismember('SealevelriseAnalysis',analyses) | (strcmp(solution,'TransientSolution') & md.transient.isslr==0), 
				return; 
			end

			md = checkfield(md,'fieldname','solidearth.sealevel','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','solidearth.requested_outputs','stringrow',1);

			self.settings.checkconsistency(md,solution,analyses);
			self.surfaceload.checkconsistency(md,solution,analyses);
			self.lovenumbers.checkconsistency(md,solution,analyses);
			self.rotational.checkconsistency(md,solution,analyses);

			
		end % }}}
		function list=defaultoutputs(self,md) % {{{
			list = {'Sealevel'};
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   solidearth inputs, forcings and settings:'));

			fielddisplay(self,'sealevel','current sea level (prior to computation) [m]');
			fielddisplay(self,'planetradius','planet radius [m]');
			fielddisplay(self,'transitions','indices into parts of the mesh that will be icecaps');
			fielddisplay(self,'requested_outputs','additional outputs requested');
			fielddisplay(self,'partitionice','ice partition vector for barystatic contribution');
			fielddisplay(self,'partitionhydro','hydro partition vector for barystatic contribution');
			self.settings.disp();
			self.surfaceload.disp();
			self.lovenumbers.disp();
			self.rotational.disp();

		end % }}}
		function marshall(self,prefix,md,fid) % {{{
			
			WriteData(fid,prefix,'object',self,'fieldname','sealevel','mattype',1,'format','DoubleMat','timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts);
			WriteData(fid,prefix,'object',self,'fieldname','planetradius','format','Double');
			WriteData(fid,prefix,'object',self,'fieldname','transitions','format','MatArray');
		
			if ~isempty(self.partitionice),
				npartice=max(self.partitionice)+2;
			else
				npartice=0;
			end
			if ~isempty(self.partitionhydro),
				nparthydro=max(self.partitionhydro)+2;
			else
				nparthydro=0;
			end

			
			WriteData(fid,prefix,'object',self,'fieldname','partitionice','mattype',1,'format','DoubleMat');
			WriteData(fid,prefix,'data',npartice,'format','Integer','name','md.solidearth.npartice');
			WriteData(fid,prefix,'object',self,'fieldname','partitionhydro','mattype',1,'format','DoubleMat');
			WriteData(fid,prefix,'data',nparthydro,'format','Integer','name','md.solidearth.nparthydro');
			
			self.settings.marshall(prefix,md,fid);
			self.surfaceload.marshall(prefix,md,fid);
			self.lovenumbers.marshall(prefix,md,fid);
			self.rotational.marshall(prefix,md,fid);
			
			%process requested outputs
			outputs = self.requested_outputs;
			pos  = find(ismember(outputs,'default'));
			if ~isempty(pos),
				outputs(pos) = [];                         %remove 'default' from outputs
				outputs      = [outputs defaultoutputs(self,md)]; %add defaults
			end
			WriteData(fid,prefix,'data',outputs,'name','md.solidearth.requested_outputs','format','StringArray');

		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejs1Darray(fid,[modelname '.solidearth.sealevel'],self.sealevel);
			self.settings.savemodeljs(fid,modelname);
			self.surfaceload.savemodeljs(fid,modelname);
			self.lovenumbers.savemodeljs(fid,modelname);
			self.rotational.savemodeljs(fid,modelname);
			writejscellstring(fid,[modelname '.solidearth.requested_outputs'],self.requested_outputs);
			writejscellarray(fid,[modelname '.solidearth.transitions'],self.transitions);
			writejscellarray(fid,[modelname '.solidearth.partition'],self.partition);
		end % }}}
		function self = extrude(self,md) % {{{
			self.sealevel=project3d(md,'vector',self.sealevel,'type','node');
		end % }}}
	end
end
