# From https://developer.apple.com/documentation/macos-release-notes/macos-catalina-10_15-release-notes,
#
#	Command line tool support for Subversion — including svn, git-svn, and 
#	related commands — is no longer provided by Xcode. (50266910)
#
# which results in,
#
#	svn: error: The subversion command line tools are no longer provided by 
#	Xcode.
#
# when calling svn, even when subversion is installed via Homebrew and its path 
# is available in PATH.
#
shopt -s expand_aliases
alias svn='/usr/local/bin/svn'

## Override certain other aliases
#
alias cp=$(which cp)
alias grep=$(which grep)

AD_IDENTITY="**********"
AD_USERNAME="**********"
ALTOOL_PASSWORD="@keychain:**********"
ASC_PROVIDER="**********"
NOTARIZATION_CHECK_ATTEMPTS=60
NOTARIZATION_CHECK_PERIOD=60
NOTARIZATION_LOGFILE="notarization.log"
NOTARIZATION_LOGFILE_PATH="."
PKG="ISSM-macOS-MATLAB"
PRIMARY_BUNDLE_ID="**********.issm.matlab"
SIGNED_REPO_COPY="./signed"
SIGNED_REPO_URL="https://issm.ess.uci.edu/svn/issm-binaries/mac/matlab/signed"
SIGNING_LOCK_FILE="signing.lock"
SUCCESS_LOGFILE="${SIGNED_REPO_COPY}/success.log"
UNSIGNED_REPO_COPY="./unsigned"
UNSIGNED_REPO_URL="https://issm.ess.uci.edu/svn/issm-binaries/mac/matlab/unsigned"

COMPRESSED_PKG="${PKG}.zip"

# Clean up from previous packaging (not necessary for single builds on Jenkins, 
# but useful when testing packaging locally)
echo 'Cleaning up existing assets'
rm -rf ${PKG} ${COMPRESSED_PKG} ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE} ${SIGNED_REPO_COPY} ${UNSIGNED_REPO_COPY}
mkdir ${PKG}

# Check out copy of SVN repository for unsigned macOS packages
echo 'Checking out copy of SVN respository for unsigned packages'
svn co \
	--username ${USERNAME} \
	--password ${PASSWORD} \
	${UNSIGNED_REPO_URL} \
	${UNSIGNED_REPO_COPY}

# Uncompress package
echo 'Uncompressing package'
ditto -xk ${UNSIGNED_REPO_COPY}/${COMPRESSED_PKG} .

# Clear extended attributes on all files
xattr -cr ${PKG}

# Build list of executables
EXECUTABLES=$(\
	echo ${PKG}/bin/mpiexec; \
	echo ${PKG}/bin/hydra_pmi_proxy; \
	echo ${PKG}/bin/gdalsrsinfo; \
	echo ${PKG}/bin/gdaltransform; \
	echo ${PKG}/bin/gmt; \
	echo ${PKG}/bin/gmtselect; \
	echo ${PKG}/bin/gmsh; \
	find ${PKG} -type f -name *.exe; \
	find ${PKG} -type f -name *.mexmaci64; \
)

# Sign all executables in package
echo 'Signing all executables in package'
codesign -s ${AD_IDENTITY} --timestamp --options=runtime ${EXECUTABLES}

# NOTE: Skipping signature validation because this is not a true package nor app

# Compress signed package
echo 'Compressing signed package'
ditto -ck --sequesterRsrc --keepParent ${PKG} ${COMPRESSED_PKG}

# Submit compressed package for notarization
echo 'Submitting signed package for notarization'
xcrun altool --notarize-app --primary-bundle-id ${PRIMARY_BUNDLE_ID} --username ${AD_USERNAME} --password ${ALTOOL_PASSWORD} --asc-provider ${ASC_PROVIDER} --file ${COMPRESSED_PKG} &> ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE}

# Sleep until notarization request response is received
echo 'Waiting for notarizaion request response'
while [[ ! -f ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE} || ! -z $(find ${NOTARIZATION_LOGFILE_PATH} -empty -name ${NOTARIZATION_LOGFILE}) ]]; do
	sleep 30
done

echo 'Notarizaion request response received'

# Check if UUID exists in response
HAS_UUID=$(grep 'RequestUUID = ' ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE}) # NOTE: Checking for "RequestUUID = " because "RequestUUID" shows up in some error messages
if [[ -z "${HAS_UUID}" ]]; then
	echo 'Notarization failed!'
	echo '----------------------- Contents of notarization logfile -----------------------'
	cat ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE}
	echo '--------------------------------------------------------------------------------'

	# Clean up
	rm -rf ${PKG} ${COMPRESSED_PKG}

	exit 1
fi

# Get UUID from notarization request response
UUID=$(echo ${HAS_UUID} | sed 's/[[:space:]]*RequestUUID = //')
echo "UUID: ${UUID}" 

# Check notarization status
#
# NOTE: Currently, this checks if notarization was successful, but we are not 
#		able to staple notarization as this is not a true package nor app and, 
#		at the very least, MATLAB Mex files cannot be stapled. As such, clients 
#		will not be able to clear Gatekeeper if they are offline.
#
echo 'Checking notarization status'
SUCCESS=0
for ATTEMPT in $(seq 1 ${NOTARIZATION_CHECK_ATTEMPTS}); do
	echo "    Attempt #${ATTEMPT}..."
	xcrun altool --notarization-info ${UUID} --username ${AD_USERNAME} --password ${ALTOOL_PASSWORD} &> ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE}
	if [[ -f ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE} && -z $(find ${NOTARIZATION_LOGFILE_PATH} -empty -name ${NOTARIZATION_LOGFILE}) ]]; then

		# First, check if there is an error
		ERROR_CHECK=$(grep 'Error' ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE})
		if [[ ! -z "${ERROR_CHECK}" ]]; then
			break
		fi

		# No error, so check status
		STATUS=$(grep 'Status:' ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE} | sed -e 's/[[:space:]]*Status: //')
		if [[ "${STATUS}" == "success" ]]; then
			# Staple notarization to all elements of package that were previously signed
			#xcrun stapler staple ${EXECUTABLES} # NOTE: Fails with "Stapler is incapable of working with MATLAB Mex files."

			# Validate stapling of notarization
			#xcrun stapler validation ${EXECUTABLES} # NOTE: Skipping notarization stapling validation because this is not a true package nor app

			# Compress signed and notarized package
			ditto -ck --sequesterRsrc --keepParent ${PKG} ${COMPRESSED_PKG}

			# Set flag indicating notarization was successful
			SUCCESS=1

			break
		elif [[ "${STATUS}" == "in progress" ]]; then
			echo "    ...in progress still; checking again in ${NOTARIZATION_CHECK_PERIOD} seconds."
			sleep ${NOTARIZATION_CHECK_PERIOD}
		elif [[ "${STATUS}" == "invalid" ]]; then
			break
		fi
	else
		if [ $ATTEMPT -lt ${NOTARIZATION_CHECK_ATTEMPTS} ]; then
			echo "    ...not ready yet; checking again in ${NOTARIZATION_CHECK_PERIOD} seconds."
			sleep ${NOTARIZATION_CHECK_PERIOD}
		else
			echo '    ...maximum attempts reached, but no response, or something else went wrong.'
			echo '    If contents of notarization status check logfile appear to be valid, increase NOTARIZATION_CHECK_ATTEMPTS and run again.'
			break
		fi
	fi
done

# Check out copy of SVN repository for signed macOS packages
echo 'Checking out copy of SVN respository for signed packages'
svn co \
	--username ${USERNAME} \
	--password ${PASSWORD} \
	${SIGNED_REPO_URL} \
	${SIGNED_REPO_COPY}

# Copy notarization file to repository for signed binaries
cp ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE} ${SIGNED_REPO_COPY}
svn add ${SIGNED_REPO_COPY}/${NOTARIZATION_LOGFILE} > /dev/null 2>&1

# Remove lock file from repository for signed binaries
svn delete ${SIGNED_REPO_COPY}/${SIGNING_LOCK_FILE} > /dev/null 2>&1

if [ ${SUCCESS} -eq 1 ]; then
	echo 'Notarization successful!'

	# Copy signed package to repository for signed binaries
	cp ${COMPRESSED_PKG} ${SIGNED_REPO_COPY}
	svn add ${SIGNED_REPO_COPY}/${COMPRESSED_PKG} > /dev/null 2>&1

	# Commit changes
	svn commit --message "CHG: New signed package (success)" ${SIGNED_REPO_COPY}
else
	echo 'Notarization failed!'
	echo '----------------------- Contents of notarization logfile -----------------------'
	cat ${NOTARIZATION_LOGFILE_PATH}/${NOTARIZATION_LOGFILE}
	echo '--------------------------------------------------------------------------------'

	# Commit changes
	svn commit --message "CHG: New signed package (failure)" ${SIGNED_REPO_COPY}

	exit 1
fi
