/*!\file:  issm_post.cpp
 * \brief: ISSM POST processing main program
 */ 
/*includes and prototypes:*/
#include "./issm.h"

int main(int argc,char **argv){ /*{{{*/
	
	char* method=NULL;

	int nfields;
	char*  string=NULL;
	char** fields=NULL;
	char*  field=NULL;
	int    offset;
	char*  stepstring=NULL;
	int    step1,step2;
	char*  pattern=NULL;
	int    nsteps; 
	int*   steps=NULL;
	int    nindices;
	int*   indices=NULL;
	int    nbins;
	
	/*datasets*/
	Parameters  *parameters   = NULL;
	Results  *results   = NULL;

	/*Initialize environment (MPI, PETSC, MUMPS, etc ...)*/
	ISSM_MPI_Comm comm=EnvironmentInit(argc,argv);

	/*First things first, store the communicator, and set it as a global variable: */
	IssmComm::SetComm(comm);

	/*Initialize and retrieve parameters:{{{*/
	parameters   = new Parameters();
	results   = new Results();

	/*Method and Solution:*/
	method=argv[1];
	parameters->AddObject(new IntParam(SolutionTypeEnum,StatisticsSolutionEnum));
	parameters->AddObject(new StringParam(AnalysisTypeEnum,method));

	/*Directory:*/
	parameters->AddObject(new StringParam(DirectoryNameEnum,argv[2]));

	/*Model name:*/
	parameters->AddObject(new StringParam(InputFileNameEnum,argv[3]));

	/*Number of samples:*/
	parameters->AddObject(new IntParam(QmuNsampleEnum,atoi(argv[4])));

	/*Retrieve fields:*/
	nfields=atoi(argv[5]);
	fields=xNew<char*>(nfields);
	for(int i=0;i<nfields;i++){
		string= argv[6+i];
		field=xNew<char>(strlen(string)+1);
		xMemCpy<char>(field,string,(strlen(string)+1));
		fields[i]=field;
	}
	parameters->AddObject(new StringArrayParam(FieldsEnum,fields,nfields));
	offset=6+nfields;

	/*free some memory: */
	for(int i=0;i<nfields;i++){
		char* field=fields[i]; 
		xDelete<char>(field);
	}
	xDelete<char*>(fields);

	/*Retrieve time steps:*/
	stepstring=argv[offset]; 
	pattern=strstr(stepstring,":");
	if (pattern==NULL){
		step1=atoi(stepstring);
		step2=step1;
	}
	else{
		step2=atoi(pattern+1);
		stepstring[pattern-stepstring]='\0';
		step1=atoi(stepstring);
	}
	nsteps=step2-step1+1; 
	steps=xNew<int>(nsteps);
	for (int i=step1;i<=step2;i++)steps[i-step1]=i;
	parameters->AddObject(new IntVecParam(StepsEnum,steps,nsteps));
	offset++;

	/*free some memory:*/
	xDelete<int>(steps);

	/*}}}*/
	
	/*Key off method:*/
	if(strcmp(method,"MeanVariance")==0){

		ComputeMeanVariance(parameters,results,0,ISSM_MPI_COMM_WORLD);
	}
	else if(strcmp(method,"Histogram")==0){
		
		/*Retrieve the size of the histogram (number of bins):*/
		nbins=atoi(argv[offset]); offset++;
		parameters->AddObject(new IntParam(NbinsEnum,nbins));
		
		ComputeHistogram(parameters,results,0,ISSM_MPI_COMM_WORLD);

	}
	else if(strcmp(method,"SampleSeries")==0){

		/*Retrieve the vertex indices where we'll retrieve our sample time series:*/
		nindices=atoi(argv[offset]); offset++;
		indices=xNew<int>(nindices);
		for (int i=0;i<nindices;i++){
			indices[i]=atoi(argv[offset+i]);
		}
		parameters->AddObject(new IntVecParam(IndicesEnum,indices,nindices));

		/*free some memory:*/
		xDelete<int>(indices);
		
		ComputeSampleSeries(parameters,results,0,ISSM_MPI_COMM_WORLD);
	}

	else _error_("unknown method: " << method << "\n");

	/*output results:*/
	ISSM_MPI_Barrier(ISSM_MPI_COMM_WORLD); _printf0_("Output file.\n");
	OutputStatistics(parameters,results,0,ISSM_MPI_COMM_WORLD);

	/*Delete ressources:*/
	delete parameters; 
	delete results;

	/*Finalize ISSM:*/
	ISSM_MPI_Finalize();

	/*Return unix success: */
	return 0; 

} /*}}}*/
