/*!\file:  issm_dakota.cpp
 * \brief: ISSM DAKOTA main program
 */ 

#include "./issm.h"
#include <sys/stat.h>

/*Dakota includes: */
#if defined(_HAVE_DAKOTA_) && _DAKOTA_MAJOR_ >= 6
#include "ParallelLibrary.hpp"
#include "ProblemDescDB.hpp"
#include "LibraryEnvironment.hpp"
#include "DakotaModel.hpp"
#include "DakotaInterface.hpp"
#endif

/*prototypes:*/
int dirstructure(int argc,char** argv);
int issm_dakota_statistics(int argc,char** argv);

int main(int argc,char **argv){ /*{{{*/

	#if defined(_HAVE_DAKOTA_) && _DAKOTA_MAJOR_ >= 6

	bool parallel=true;
	char* dakota_input_file=NULL;
	char* dakota_output_file = NULL;
	char* dakota_error_file = NULL;

	/*Define MPI_DEBUG in dakota_global_defs.cpp to cause a hold here*/
	Dakota::mpi_debug_hold();

	/*Initialize MPI: */
	ISSM_MPI_Init(&argc, &argv); // initialize MPI

	/*Recover file name for dakota input file:*/
	dakota_input_file=xNew<char>((strlen(argv[2])+strlen(argv[3])+strlen(".qmu.in")+2));
	sprintf(dakota_input_file,"%s/%s%s",argv[2],argv[3],".qmu.in");

	dakota_output_file=xNew<char>((strlen(argv[2])+strlen(argv[3])+strlen(".qmu.out")+2));
	sprintf(dakota_output_file,"%s/%s%s",argv[2],argv[3],".qmu.out");

	dakota_error_file=xNew<char>((strlen(argv[2])+strlen(argv[3])+strlen(".qmu.err")+2));
	sprintf(dakota_error_file,"%s/%s%s",argv[2],argv[3],".qmu.err");

	/*Create directory structure for model outputs:*/
	dirstructure(argc,argv);

	/* Parse input and construct Dakota LibraryEnvironment, performing input data checks*/
	Dakota::ProgramOptions opts;
	opts.input_file(dakota_input_file);
	opts.output_file(dakota_output_file);
	opts.error_file(dakota_error_file);

	/* Defaults constructs the MPIManager, which assumes COMM_WORLD*/
	Dakota::LibraryEnvironment env(opts);

	/* get the list of all models matching the specified model, interface, driver:*/
	Dakota::ModelList filt_models = env.filtered_model_list("single", "direct", "matlab");
	if (filt_models.empty()) {
		Cerr << "Error: no parallel interface plugin performed.  Check compatibility "
			<< "between parallel\n       configuration and selected analysis_driver."
			<< std::endl;
		Dakota::abort_handler(-1);
	}

	Dakota::ProblemDescDB& problem_db = env.problem_description_db();
	Dakota::ModelLIter ml_iter;
	size_t model_index = problem_db.get_db_model_node(); // for restoration
	for (ml_iter = filt_models.begin(); ml_iter != filt_models.end(); ++ml_iter) {
		// set DB nodes to input specification for this Model
		problem_db.set_db_model_nodes(ml_iter->model_id());

		Dakota::Interface& model_interface = ml_iter->derived_interface();

		// Parallel case: plug in derived Interface object with an analysisComm.
		// Note: retrieval and passing of analysisComm is necessary only if
		// parallel operations will be performed in the derived constructor.

		// retrieve the currently active analysisComm from the Model.  In the most
		// general case, need an array of Comms to cover all Model configurations.
		const MPI_Comm& analysis_comm = ml_iter->analysis_comm();

		// don't increment ref count since no other envelope shares this letter
		model_interface.assign_rep(new
				SIM::IssmParallelDirectApplicInterface(problem_db, analysis_comm, argc, argv), false);
	}
	problem_db.set_db_model_nodes(model_index);            // restore

	/* Execute the environment:*/
	env.execute();

	/* Run statistics if requested:*/
	issm_dakota_statistics(argc,argv);

	/*free allocations:*/
	xDelete<char>(dakota_input_file);
	xDelete<char>(dakota_output_file);
	xDelete<char>(dakota_error_file);

	/*Return unix success: */
	return 0; 
	#else 
	Cout <<  "ISSM Dakota  executable was compiled without support of Dakota! Will just return now!" << "\n";
	return 1;
	#endif

} /*}}}*/
int dirstructure(int argc,char** argv){ /*{{{*/

	char* input_file; 
	FILE* fid;
	IoModel* iomodel=NULL;
	int check;

	//qmu statistics
	bool statistics    = false;
	int  numdirectories = 0;

	/*First things first, set the communicator as a global variable: */
	IssmComm::SetComm(MPI_COMM_WORLD);

	/*Barrier:*/
	ISSM_MPI_Barrier(IssmComm::GetComm());
	_printf0_("Preparing directory structure for model outputs:" << "\n");

	//open model input file for reading
	input_file=xNew<char>((strlen(argv[2])+strlen(argv[3])+strlen(".bin")+2));
	sprintf(input_file,"%s/%s%s",argv[2],argv[3],".bin");
	fid=fopen(input_file,"rb");
	if (fid==NULL) Cerr << "dirstructure error message: could not open model " << input_file << " to retrieve qmu statistics parameters" << std::endl;

	//initialize IoModel, but light version, we just need it to fetch one constant: 
	iomodel=new IoModel();
	iomodel->fid=fid;
	iomodel->FetchConstants();

	//early return if statistics not requested: 
	iomodel->FindConstant(&statistics,"md.qmu.statistics");
	if(!statistics){
		delete iomodel;
		fclose(fid); 
		return 0;
	}

	iomodel->FindConstant(&numdirectories,"md.qmu.statistics.ndirectories");

	/*Ok, we have everything we need to create the directory structure:*/
	if(IssmComm::GetRank()==0){
		for (int i=0;i<numdirectories;i++){
			char directory[1000];
			sprintf(directory,"./%i",i+1);

			check = mkdir(directory,ACCESSPERMS);
			if (check) _error_("dirstructure error message: could not create directory " << directory << "\n");
		}
	}

	//close model file: 
	fclose(fid);
} /*}}}*/
int issm_dakota_statistics(int argc,char** argv){ /*{{{*/

	char* input_file; 
	FILE* fid;
	IoModel* iomodel=NULL;
	ISSM_MPI_Comm statcomm;
	int my_rank;

	//qmu statistics
	bool statistics    = false;
	int  numstatistics = 0;
	int  numdirectories = 0;
	int  nfilesperdirectory = 0;
	char string[1000];
	char* name = NULL;
	char** fields = NULL;
	int    nfields; 
	int*   steps=NULL;
	int    nsteps;
	int    nbins;
	int*   indices=NULL;
	int    nindices;
	int    nsamples;
	int    dummy;
	char*  directory=NULL;
	char*  model=NULL;
	Results* results=NULL;
	Parameters* parameters=NULL;
	int color;

	/*First things first, set the communicator as a global variable: */
	IssmComm::SetComm(MPI_COMM_WORLD);
	my_rank=IssmComm::GetRank();

	/*Barrier:*/
	ISSM_MPI_Barrier(IssmComm::GetComm());
	_printf0_("Dakota Statistic Computation" << "\n");

	//open model input file for reading
	input_file=xNew<char>((strlen(argv[2])+strlen(argv[3])+strlen(".bin")+2));
	sprintf(input_file,"%s/%s%s",argv[2],argv[3],".bin");
	fid=fopen(input_file,"rb");
	if (fid==NULL) Cerr << "issm_dakota_statistics error message: could not open model " << input_file << " to retrieve qmu statistics parameters" << std::endl;

	//initialize IoModel, but light version, we'll need it to fetch constants:
	iomodel=new IoModel();
	iomodel->fid=fid;
	iomodel->FetchConstants();

	//early return if statistics not requested: 
	iomodel->FindConstant(&statistics,"md.qmu.statistics");
	if(!statistics){
		delete iomodel;
		fclose(fid); 
		return 0;
	}

	//create parameters datasets with al the qmu statistics settings we need: 
	if(statistics){

		/*Initialize parameters and results:*/
		results   = new Results();
		parameters=new Parameters();
	
		//root  directory
		directory=xNew<char>(strlen(argv[2])+1);
		xMemCpy<char>(directory,argv[2],strlen(argv[2])+1);
		parameters->AddObject(new StringParam(DirectoryNameEnum,directory));

		//model  name
		model=xNew<char>(strlen(argv[3])+1);
		xMemCpy<char>(model,argv[3],strlen(argv[3])+1);
		parameters->AddObject(new StringParam(InputFileNameEnum,model));

		//nsamples
		iomodel->FindConstant(&nsamples,"md.qmu.method.params.samples");
		parameters->AddObject(new IntParam(QmuNsampleEnum,nsamples));

		//ndirectories
		iomodel->FindConstant(&numdirectories,"md.qmu.statistics.ndirectories");
		parameters->AddObject(new IntParam(QmuNdirectoriesEnum,numdirectories));

		//nfiles per directory
		iomodel->FindConstant(&nfilesperdirectory,"md.qmu.statistics.nfiles_per_directory");
		parameters->AddObject(new IntParam(QmuNfilesPerDirectoryEnum,nfilesperdirectory));

		//At this point, we don't want to go forward any longer, we want to create an MPI 
		//communicator on which to carry out the computations:
		if ((my_rank+1)*nfilesperdirectory>nsamples)color=MPI_UNDEFINED;
		else color=0;
		ISSM_MPI_Comm_split(ISSM_MPI_COMM_WORLD,color, my_rank, &statcomm);


		iomodel->FindConstant(&numstatistics,"md.qmu.statistics.numstatistics");
		for (int i=1;i<=numstatistics;i++){

			char* directory=NULL;
			char* model=NULL;
			int   nsamples;
			_printf0_("Dealing with qmu statistical computation #" << i << "\n");
		
			sprintf(string,"md.qmu.statistics.method(%i).name",i);
			iomodel->FindConstant(&name,string);

			sprintf(string,"md.qmu.statistics.method(%i).fields",i);
			iomodel->FindConstant(&fields,&nfields,string);
			parameters->AddObject(new StringArrayParam(FieldsEnum,fields,nfields));

			sprintf(string,"md.qmu.statistics.method(%i).steps",i);
			iomodel->FetchData(&steps,&nsteps,&dummy,string);
			parameters->AddObject(new IntVecParam(StepsEnum,steps,nsteps));

			if (strcmp(name,"Histogram")==0){
				/*fetch nbins: */
				sprintf(string,"md.qmu.statistics.method(%i).nbins",i);
				iomodel->FindConstant(&nbins,string);
				parameters->AddObject(new IntParam(NbinsEnum,nbins));
				ComputeHistogram(parameters,results,color,statcomm);
			}
			else if (strcmp(name,"SampleSeries")==0){
				/*fetch indices: */
				sprintf(string,"md.qmu.statistics.method(%i).indices",i);
				iomodel->FetchData(&indices,&nindices,&dummy,string);
				parameters->AddObject(new IntVecParam(IndicesEnum,indices,nindices));
		
				ComputeSampleSeries(parameters,results,color,statcomm);
			}
			else if (strcmp(name,"MeanVariance")==0){
				ComputeMeanVariance(parameters,results,color,statcomm);
			}
			else _error_(" error creating qmu statistics methods parameters: unsupported method " << name);
		}
	}
	//close model file: 
	fclose(fid);

	/*output results:*/
	ISSM_MPI_Barrier(ISSM_MPI_COMM_WORLD); _printf0_("Output file.\n");
	OutputStatistics(parameters,results);

	/*Delete ressources:*/
	delete parameters; 
	delete results;


	
} /*}}}*/
