/*\file Marshalling.h
 *\brief: macros to help automate the marshalling, demarshalling, and marshalling size routines. 
 */

#ifndef _MARSHALLING_H_
#define _MARSHALLING_H_

#include <string.h>
#include "../../Exceptions/exceptions.h"
#include "../../MemOps/MemOps.h"
#include "../../Numerics/recast.h"

/*Define Marshall operation Enums first*/
enum MarshallOpEnum{
	MARSHALLING_WRITE,
	MARSHALLING_LOAD,
	MARSHALLING_SIZE,
};

/*Define virtual Marshall Handle*/
class MarshallHandle{
	public:
		MarshallOpEnum operation_enum;
		MarshallHandle(MarshallOpEnum op_in) : operation_enum(op_in){}
		~MarshallHandle(){}
		virtual void Echo(void)=0;
		int OperationNumber(){return operation_enum;}
		template<typename T> void call(T  & value);
		template<typename T> void call(T* & value,int size);
};

/*Make sure to pass all fields by reference!*/
class WriteCheckpointFunctor: public MarshallHandle{
	public:
		char** pmarshalled_data;
		WriteCheckpointFunctor(char** pmarshalled_data_in) : MarshallHandle(MARSHALLING_WRITE),pmarshalled_data(pmarshalled_data_in){}
		template<typename T> void call(T & value){
			memcpy(*pmarshalled_data,&value,sizeof(T));
			*pmarshalled_data+=sizeof(T);
		}
		void Echo(void){
			printf("WriteCheckpointFunctor Echo:\n");
			printf("   pmarshalled_data: %p\n",pmarshalled_data);
		}
		template<typename T> void call(T* & value,int size){
			bool pointer_null = true;
			if(value) pointer_null = false;
			call(pointer_null);
			if(value){
				memcpy(*pmarshalled_data,value,size*sizeof(T));
				*pmarshalled_data+=size*sizeof(T);
			}
		}
};

class LoadCheckpointFunctor: public MarshallHandle{
	public:
		char** pmarshalled_data;
		LoadCheckpointFunctor(char** pmarshalled_data_in) : MarshallHandle(MARSHALLING_LOAD),pmarshalled_data(pmarshalled_data_in){}
		void Echo(void){
			printf("LoadCheckpointFunctor Echo:\n");
			printf("   pmarshalled_data: %p\n",pmarshalled_data);
		}
		template<typename T> void call(T & value){
			memcpy(&value,*pmarshalled_data,sizeof(T));
			*pmarshalled_data+=sizeof(T);
		}
		template<typename T> void call(T* & value,int size){
			bool pointer_null;
			call(pointer_null);
			if(!pointer_null){
				value=xNew<T>(size);
				memcpy(value,*pmarshalled_data,size*sizeof(T));
				*pmarshalled_data+=size*sizeof(T);
			}
			else{
				value = NULL;
			}
		}
};

class SizeCheckpointFunctor: public MarshallHandle{
	public:
		int marshalled_data_size;
		SizeCheckpointFunctor(void) : MarshallHandle(MARSHALLING_SIZE),marshalled_data_size(0){}
		int MarshalledSize(void){return this->marshalled_data_size;};
		void Echo(void){
			printf("SizeCheckpointFunctor Echo:\n");
			printf("   marshalled_data_size: %i\n",marshalled_data_size);
		}
		template<typename T> void call(T & value){
			marshalled_data_size+=sizeof(T);
		}
		template<typename T> void call(T* & value,int size){
			bool pointer_null = true;
			if(value) pointer_null = false;
			this->call(pointer_null);
			if(!pointer_null){
				marshalled_data_size+=size*sizeof(T);
			}
		}
};

template<typename T> void MarshallHandle::call(T & value){
	switch(OperationNumber()){
		case MARSHALLING_WRITE:{WriteCheckpointFunctor* temp = xDynamicCast<WriteCheckpointFunctor*>(this); temp->call(value); break;}
		case MARSHALLING_LOAD: {LoadCheckpointFunctor*  temp = xDynamicCast<LoadCheckpointFunctor*>(this);  temp->call(value); break;}
		case MARSHALLING_SIZE: {SizeCheckpointFunctor*  temp = xDynamicCast<SizeCheckpointFunctor*>(this);  temp->call(value); break;}
		default: _error_("not supported yet");
	}
}
template<typename T> void MarshallHandle::call(T* & value,int size){
	switch(OperationNumber()){
		case MARSHALLING_WRITE:{WriteCheckpointFunctor* temp = xDynamicCast<WriteCheckpointFunctor*>(this); temp->call(value,size); break;}
		case MARSHALLING_LOAD: {LoadCheckpointFunctor*  temp = xDynamicCast<LoadCheckpointFunctor*>(this);  temp->call(value,size); break;}
		case MARSHALLING_SIZE: {SizeCheckpointFunctor*  temp = xDynamicCast<SizeCheckpointFunctor*>(this);  temp->call(value,size); break;}
		default: _error_("not supported yet");
	}
}

#define MARSHALLING_ENUM(EN)\
	int type_enum=EN;\
	if(marshall_direction==MARSHALLING_WRITE){\
		memcpy(*pmarshalled_data,&type_enum,sizeof(int));\
		*pmarshalled_data+=sizeof(int);\
	}\
	else if(marshall_direction==MARSHALLING_SIZE){\
		*pmarshalled_data_size+=sizeof(int);\
	}\
	else if(marshall_direction==MARSHALLING_LOAD){\
		*pmarshalled_data+=sizeof(int);\
	}\
	else _error_("Wrong direction during the Marshall process");\


#define MARSHALLING(FIELD)\
	\
	if(marshall_direction==MARSHALLING_WRITE){\
		memcpy(*pmarshalled_data,&FIELD,sizeof(FIELD));\
		*pmarshalled_data+=sizeof(FIELD);\
	}\
	else if(marshall_direction==MARSHALLING_SIZE){\
		*pmarshalled_data_size+=sizeof(FIELD);\
	}\
	else if(marshall_direction==MARSHALLING_LOAD){\
		memcpy(&FIELD,*pmarshalled_data,sizeof(FIELD));\
		*pmarshalled_data+=sizeof(FIELD);\
	}\
	else _error_("Wrong direction during the Marshall process");


#define MARSHALLING_DYNAMIC(FIELD,TYPE,SIZE) \
	\
	{\
		bool field_null=true;\
		if (FIELD)field_null=false;\
		MARSHALLING(field_null);\
		\
		if(!field_null){\
			if(marshall_direction==MARSHALLING_WRITE){\
					memcpy(*pmarshalled_data,FIELD,SIZE*sizeof(TYPE));\
					*pmarshalled_data+=SIZE*sizeof(TYPE);\
			}\
			else if(marshall_direction==MARSHALLING_SIZE){\
				*pmarshalled_data_size+=SIZE*sizeof(TYPE);\
			}\
			else if(marshall_direction==MARSHALLING_LOAD){\
				FIELD=xNew<TYPE>(SIZE);\
				memcpy(FIELD,*pmarshalled_data,SIZE*sizeof(TYPE));\
				*pmarshalled_data+=SIZE*sizeof(TYPE);\
			}\
			else _error_("Wrong direction during the Marshall process");\
		}\
	}

#endif	
