function [eust,rsl] = SESAWslr(index,greens,para) 
%SESAWslr :: computes GRD slr due to applied surface loads based on SESAW method. 
% 
%Reference: Adhikari et al., 2016, GMD: https://doi.org/10.5194/gmd-9-1087-2016 
%
%   Usage:
%      [eus,rsl]=greensfunctions(md.mesh.elements,greens,para) 
%      
%      greens.Grigid = Gravitational Green's function for the rigid planet. 
%      greens.Gelast = Gravitational Green's function for the elastic planet.
%      greens.Uelast = Deformational (radial displacement) Green's function.
% 
%      para.ocean_element = ocean funnction mapped onto elemental centroid. 
%      para.loads_element = land loads (ice or water) [m] 
%      para.area_element = area of elements [m^2]  
%      para.earth_density = averae density of the solid earth [kg/m^3] 
%      para.ocean_density = ocean water density [kg/m^3] 
%      para.loads_density = land loads (ice or freshwater) density [kg/m^3] 
%      para.rel_tol = relative tolerance for iterative SLR solver 
%      para.solidearth = rheological model for solid Earth: 'rigid' or 'elastic' 
%      para.isgrd_R = Do we want to account for Rotational feedback? 0 or 1. 

area_element = para.area_element; 
ocean_element = para.ocean_element; 
loads_element = para.loads_element; 
rho_o = para.ocean_density; 
rho_l = para.loads_density; % land loads, ice or water... 
rho_e = para.earth_density; 
rel_tol = para.rel_tol; 

% total Green's function for elastic earth, i.e. (1+k_l-h_l)
if strcmpi(para.solidearth,'rigid') 
	Galpha = greens.Grigid; 
elseif strcmpi(para.solidearth,'elastic') 
	Galpha = greens.Grigid + greens.Gelast- greens.Uelast;		
else
	error(['Unknown solidearth model:' para.solidearth, '; Should be rigid or elastic']); 
end

density_o_e = rho_o/rho_e; 
density_l_e = rho_l/rho_e; 
density_l_o = rho_l/rho_o; 

% eustatic term 
eust=-density_l_o*sum(loads_element.*area_element)/sum(ocean_element.*area_element); 

term1=3*density_l_e.*sum(bsxfun(@times,Galpha,(loads_element.*area_element)'),2)./sum(area_element); 
func3=mean(term1(index),2).*ocean_element;
term3=sum(func3.*area_element)./sum(ocean_element.*area_element); 

% computation of sea level change 
rsl=eust+term1-term3;

norm_diff = 10;			p = 0; 

while norm_diff > rel_tol
	norm_old = sqrt(sum(rsl.^2)); 
	% 
	term2=3*density_o_e.*sum(bsxfun(@times,Galpha,(mean(rsl(index),2).*ocean_element.*area_element)'),2)./sum(area_element); 
	func4=mean(term2(index),2).*ocean_element;
	term4=sum(func4.*area_element)./sum(ocean_element.*area_element); 
	% 
	rsl=eust+term1+term2-term3-term4; 
	norm_new  = sqrt(sum(rsl.^2));
	norm_diff = abs(norm_new-norm_old)./norm_old;
	p = p+1;
	if norm_diff > rel_tol
		disp(['     iteration # ', num2str(p), ' :: difference in norm = ', num2str(norm_diff)]);
	else
		disp(['     iteration # ', num2str(p), ' :: difference in norm = ', num2str(norm_diff)]);
		disp(['     solution converged! ']);
	end
end 


