%DSLMME class definition
%
%   Usage:
%      dsl=dslmme(); %dynamic sea level class based on a multi-model ensemble of CMIP5 outputs

classdef dslmme
	properties (SetAccess=public) 

		modelid; %index into the multi-model ensemble, determine which field will be used.
		global_average_thermosteric_sea_level_change; %corresponds to zostoga fields in CMIP5 archives. Specified as a temporally variable global rate (mm/yr) for each ensemble.
		sea_surface_height_change_above_geoid; %corresponds to zos fields in CMIP5 archives. Spatial average is 0. Specified as a spatio-temporally variable rate (mm/yr) for each ensemble
		sea_water_pressure_change_at_sea_floor; %corresponds to bpo fields in CMIP5 archives. Specified as a spatio-temporally variable rate (in mm/yr equivalent, not in Pa/yr!) for each ensemble
		compute_fingerprints; %do we use the sea water pressure change to compute fingerprints and correct sea_surface_height_change_above_geoid

	end
	methods
		function self = dslmme(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					self=structtoobj(dsl(),varargin{1});
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			self.modelid=0;
			self.global_average_thermosteric_sea_level_change={};
			self.sea_surface_height_change_above_geoid={};
			self.sea_water_pressure_change_at_sea_floor={};
			self.compute_fingerprints=0;

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			%Early return
			if ~ismember('SealevelriseAnalysis',analyses), return; end
			if (strcmp(solution,'TransientSolution') & md.transient.isslr == 0), return; end
			for i=1:length(self.global_average_thermosteric_sea_level_change),
				md = checkfield(md,'field',self.global_average_thermosteric_sea_level_change{i},'NaN',1,'Inf',1);
				md = checkfield(md,'field',self.sea_surface_height_change_above_geoid{i},'NaN',1,'Inf',1,'timeseries',1);
				md = checkfield(md,'field',self.sea_water_pressure_change_at_sea_floor{i},'NaN',1,'Inf',1,'timeseries',1);
			end
			md = checkfield(md,'field',self.modelid,'NaN',1,'Inf',1,'>=',1,'<=',length(self.global_average_thermosteric_sea_level_change));
			if self.compute_fingerprints,
				%check geodetic flag of slr is on: 
				if md.solidearth.settings.computesealevelchange==0,
					error('DSL checkconsistency error message: if bottom pressure fingerprints computations are requested, slr class should have geodetic flag on');
				end
			end


		end % }}}
		function disp(self) % {{{

			disp(sprintf('   dsl mme parameters:'));
			fielddisplay(self,'modelid','index into the multi-model ensemble, determine which field will be used.');
			fielddisplay(self,'global_average_thermosteric_sea_level_change','corresponds to zostoga fields in CMIP5 archives. Specified as a temporally variable global rate (mm/yr) for each ensemble.');
			fielddisplay(self,'sea_surface_height_change_above_geoid','corresponds to zos fields in CMIP5 archives. Spatial average is 0. Specified as a spatio-temporally variable rate (mm/yr) for each ensemble.');
			fielddisplay(self,'sea_water_pressure_change_at_sea_floor','corresponds to bpo fields in CMIP5 archives. Specified as a spatio-temporally variable rate (in mm/yr) for each ensemble.');
			fielddisplay(self,'compute_fingerprints','%do we use the sea water pressure change to compute fingerprints and correct sea_surface_height_change_above_geoid');
		end % }}}
		function marshall(self,prefix,md,fid) % {{{

			WriteData(fid,prefix,'name','md.dsl.model','data',2,'format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','compute_fingerprints','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','modelid','format','Double');
			WriteData(fid,prefix,'name','md.dsl.nummodels','data',length(self.global_average_thermosteric_sea_level_change),'format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','global_average_thermosteric_sea_level_change','format','MatArray','timeseries',1,'timeserieslength',2,'yts',md.constants.yts,'scale',1e-3/md.constants.yts);
			WriteData(fid,prefix,'object',self,'fieldname','sea_water_pressure_change_at_sea_floor','format','MatArray','timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts,'scale',1e-3);
			WriteData(fid,prefix,'object',self,'fieldname','sea_surface_height_change_above_geoid','format','MatArray','timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts,'scale',1e-3/md.constants.yts);

		end % }}}
		function self = extrude(self,md) % {{{
			for i=1:length(self.global_average_thermosteric_sea_level_change),
				self.sea_surface_height_change_above_geoid{i}=project3d(md,'vector',self.sea_surface_height_change_above_geoid{i},'type','node','layer',1);
				self.sea_water_pressure_change_at_sea_floor{i}=project3d(md,'vector',self.sea_water_pressure_change_at_sea_floor{i},'type','node','layer',1);
			end
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
			
			writejsdouble(fid,[modelname '.dsl.modelid'],self.modelid);
			writejscellarray(fid,[modelname '.dsl.global_average_thermosteric_sea_level_change'],self.global_average_thermosteric_sea_level_change);
			writejscellarray(fid,[modelname '.dsl.sea_surface_height_change_above_geoid'],self.sea_surface_height_change_above_geoid);
			writejs1Darray(fid,[modelname '.dsl.compute_fingerprints'],self.compute_fingerprints);
			writejscellarray(fid,[modelname '.dsl.sea_water_pressure_change_at_sea_floor'],self.sea_water_pressure_change_at_sea_floor);

		end % }}}
	end
end
