import numpy as np

from checkfield import checkfield
from fielddisplay import fielddisplay
from lovenumbers import lovenumbers
from planetradius import planetradius
from project3d import project3d
from rotational import rotational
from solidearthsettings import solidearthsettings
from surfaceload import surfaceload
from WriteData import WriteData


class solidearth(object):
    '''
    SOLIDEARTH class definition

        Usage:
            solidearth = solidearth()
    '''

    def __init__(self, *args):  #{{{
        self.sealevel = np.nan
        self.settings = solidearthsettings()
        self.surfaceload = surfaceload()
        self.lovenumbers = lovenumbers()
        self.rotational = rotational()
        self.planetradius = planetradius('earth')
        self.requested_outputs = ['default']
        self.transitions = []

        nargin = len(args)

        if nargin == 0:
            self.setdefaultparameters()
        else:
            raise Exception('constructor not supported')
    #}}}

    def __repr__(self):  # {{{
        s = '   solidearthinputs, forcings and settings:\n'
        s += '{}\n'.format(fielddisplay(self, 'sealevel', 'current sea level (prior to computation) [m]'))
        s += '{}\n'.format(fielddisplay(self, 'planetradius', 'planet radius [m]'))
        s += '{}\n'.format(fielddisplay(self, 'transitions', 'indices into parts of the mesh that will be icecaps'))
        s += '{}\n'.format(fielddisplay(self, 'requested_outputs', 'additional outputs requested'))
        s += '{}\n'.format(fielddisplay(self, 'miscellaneous', 'miscellaneous fields'))
        s += '{}\n'.format(self.settings)
        s += '{}\n'.format(self.surfaceload)
        s += '{}\n'.format(self.lovenumbers)
        s += '{}\n'.format(self.rotational)
        return s
    #}}}

    def setdefaultparameters(self):  # {{{
        return
    #}}}

    def checkconsistency(self, md, solution, analyses):  # {{{
        if ('SealevelriseAnalysis' not in analyses) or (solution == 'TransientSolution' and md.transient.isslr == 0):
            return md

        md = checkfield(md, 'fieldname', 'solidearth.sealevel', 'NaN', 1, 'Inf', 1, 'size', [md.mesh.numberofvertices])
        md = checkfield(md, 'fieldname', 'solidearth.requested_outputs', 'stringrow', 1)

        self.settings.checkconsistency(md, solution, analyses)
        self.surfaceload.checkconsistency(md, solution, analyses)
        self.lovenumbers.checkconsistency(md, solution, analyses)
        self.rotational.checkconsistency(md, solution, analyses)

        return md
    #}}}

    def defaultoutputs(self, md):  #{{{
        return ['Sealevel']
    #}}}

    def marshall(self, prefix, md, fid):  #{{{
        WriteData(fid, prefix, 'object', self, 'fieldname', 'sealevel', 'mattype', 1, 'format', 'DoubleMat', 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', md.constants.yts)
        WriteData(fid, prefix, 'object', self, 'fieldname', 'planetradius', 'format', 'Double')
        WriteData(fid, prefix, 'object', self, 'fieldname', 'transitions', 'format', 'MatArray')

        self.settings.marshall(prefix, md, fid)
        self.surfaceload.marshall(prefix, md, fid)
        self.lovenumbers.marshall(prefix, md, fid)
        self.rotational.marshall(prefix, md, fid)

        #process requested outputs
        outputs = self.requested_outputs
        pos = np.where(np.asarray(outputs) == 'default')[0]
        if len(pos):
            outputs = np.delete(outputs, pos) #remove 'default' from outputs
            outputs = np.append(outputs, self.defaultoutputs(md)) #add defaults
        WriteData(fid, prefix, 'data', outputs, 'name', 'md.solidearth.requested_outputs', 'format', 'StringArray')
    #}}}

    def extrude(self, md):  #{{{
        self.sealevel = project3d(md, 'vector', self.sealevel, 'type', 'node')

        return self
    #}}}
