%FOURIERLOVE class definition
%
%   Usage:
%      md.love=fourierlove();

classdef fourierlove
	properties (SetAccess=public) 
		nfreq                = 0;
		frequencies          = 0;
		sh_nmax              = 0;
		sh_nmin              = 0;
		g0                   = 0;
		r0                   = 0;
		mu0                  = 0;
		allow_layer_deletion = 0;
		love_kernels         = 0;
		forcing_type         = 0;
	end
	methods (Static)
		function self = loadobj(self) % {{{
			% This function is directly called by matlab when a model object is
			% loaded. Update old properties here
		end% }}}
	end
	methods
		function self = extrude(self,md) % {{{
		end % }}}
		function self = fourierlove(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{
			%we setup an elastic love number computation by default.
			self.nfreq=1; 
			self.frequencies=[0]; %Hz
			self.sh_nmax=256; % .35 degree, 40 km at the equator.
			self.sh_nmin=1;
			% work on matlab script for computing g0 for given Earth's structure. 
			self.g0=9.81; % m/s^2; 
			self.r0=6371*1e3; %m;
			self.mu0=10^11; % Pa
			self.allow_layer_deletion=1; 
			self.love_kernels=0; 
			self.forcing_type = 11; 
		end % }}}
		function disp(self) % {{{
			fielddisplay(self,'nfreq','number of frequencies sampled (default 1, elastic) [Hz]');
			fielddisplay(self,'frequencies','frequencies sampled (convention defaults to 0 for the elastic case) [Hz]');
			fielddisplay(self,'sh_nmax','maximum spherical harmonic degree (default 256, .35 deg, or 40 km at equator)');
			fielddisplay(self,'sh_nmin','minimum spherical harmonic degree (default 1)');
			fielddisplay(self,'g0','adimensioning constant for gravity (default 10) [m/s^2]');
			fielddisplay(self,'r0','adimensioning constant for radius (default 6378*10^3) [m]');
			fielddisplay(self,'mu0','adimensioning constant for stress (default 10^11) [Pa]');
			fielddisplay(self,'allow_layer_deletion','allow for migration of the integration boundary with increasing spherical harmonics degree (default 1)');
			fielddisplay(self,'love_kernels','compute love numbers at depth? (default 0)');
			fielddisplay(self,'forcing_type',{'integer indicating the nature and depth of the forcing for the Love number calculation (default 11) :','1:  Inner core boundary -- Volumic Potential','2:  Inner core boundary -- Pressure','3:  Inner core boundary -- Loading','4:  Inner core boundary -- Tangential traction','5:  Core mantle boundary -- Volumic Potential','6:  Core mantle boundary -- Pressure','7:  Core mantle boundary -- Loading','8:  Core mantle boundary -- Tangential traction','9:  Surface -- Volumic Potential','10: Surface -- Pressure','11: Surface -- Loading','12: Surface -- Tangential traction '}); 

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','love.nfreq','NaN',1,'Inf',1,'numel',1,'>',0);
			md = checkfield(md,'fieldname','love.frequencies','NaN',1,'Inf',1,'numel',md.love.nfreq);
			md = checkfield(md,'fieldname','love.sh_nmax','NaN',1,'Inf',1,'numel',1,'>',0);
			md = checkfield(md,'fieldname','love.sh_nmin','NaN',1,'Inf',1,'numel',1,'>',0);
			md = checkfield(md,'fieldname','love.g0','NaN',1,'Inf',1,'numel',1,'>',0);
			md = checkfield(md,'fieldname','love.r0','NaN',1,'Inf',1,'numel',1,'>',0);
			md = checkfield(md,'fieldname','love.mu0','NaN',1,'Inf',1,'numel',1,'>',0);
			md = checkfield(md,'fieldname','love.allow_layer_deletion','values',[0 1]);
			md = checkfield(md,'fieldname','love.love_kernels','values',[0 1]);
			md = checkfield(md,'fieldname','love.forcing_type','NaN',1,'Inf',1,'numel',1,'>',0, '<=', 12);
			if md.love.sh_nmin<=1 & md.love.forcing_type==9
				error('Degree 1 not supported for Volumetric Potential forcing. Use sh_min>=2 for this kind of calculation.')
			end
		end % }}}
		function marshall(self,prefix,md,fid) % {{{
		
			WriteData(fid,prefix,'object',self,'fieldname','nfreq','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','frequencies','format','DoubleMat','mattype',3);
			WriteData(fid,prefix,'object',self,'fieldname','sh_nmax','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','sh_nmin','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','g0','format','Double');
			WriteData(fid,prefix,'object',self,'fieldname','r0','format','Double');
			WriteData(fid,prefix,'object',self,'fieldname','mu0','format','Double');
			WriteData(fid,prefix,'object',self,'fieldname','allow_layer_deletion','format','Boolean');
			WriteData(fid,prefix,'object',self,'fieldname','love_kernels','format','Boolean');
			WriteData(fid,prefix,'object',self,'fieldname','forcing_type','format','Integer');

		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
			error('not implemented yet!');
		end % }}}
	end
end
