/*!\file InputUpdateFromDakotax
 * \brief: update datasets using  parameter inputs
 */

#include "./InputUpdateFromDakotax.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"
#include "../../classes/Inputs2/TransientInput2.h"
#include "../../classes/Inputs2/DatasetInput2.h"
#include "../../classes/Inputs2/TriaInput2.h"
#include "../InputUpdateFromMatrixDakotax/InputUpdateFromMatrixDakotax.h"
#include "../InputUpdateFromConstantx/InputUpdateFromConstantx.h"
#include "../InputUpdateFromVectorDakotax/InputUpdateFromVectorDakotax.h"
			
void InputUpdateFromDakotax(FemModel* femmodel,double* variables,char* *variables_descriptors,int numdakotavariables){ /*{{{*/

	int     i,j,k,l;

	int     numberofvertices;
	int     numberofelements;
	int     nrows;
	int     ncols;
	IssmDouble **variable_partitions         = NULL;
	IssmDouble * variable_partition         = NULL;
	int * variable_partitions_npart         = NULL;
	int * variable_partitions_nt         = NULL;
	int          variable_partitions_num;
	int          npart;
	int          nt;
	int variablecount=0;

	double *distributed_values = NULL;
	double *parameter          = NULL;
	char   *descriptor         = NULL;
	char    root[50]; //root name of variable, ex: DragCoefficent, RhoIce, etc ...

	if (VerboseQmu())_printf0_("dakota variables updates\n");

	/*retrieve parameters: */
	femmodel->parameters->FindParam(&variable_partitions,&variable_partitions_num,NULL,NULL,QmuVariablePartitionsEnum); 
	femmodel->parameters->FindParam(&variable_partitions_npart,NULL,NULL,QmuVariablePartitionsNpartEnum); 
	femmodel->parameters->FindParam(&variable_partitions_nt,NULL,NULL,QmuVariablePartitionsNtEnum); 
	
	numberofvertices=femmodel->vertices->NumberOfVertices();
	numberofelements=femmodel->elements->NumberOfElements();

	/*Go through all dakota descriptors, ex: "rho_ice","thermal_conductivity","thickness1","thickness2", etc ..., and 
	 * for each descriptor, take the variable value and plug it into the inputs: */
	for(i=0;i<numdakotavariables;i++){ //these are the dakota variables, for all partitions. 

		descriptor=variables_descriptors[i];
	
		if (VerboseQmu())_printf0_("   updating variable " << descriptor << "\n");

		/*From descriptor, figure out if the variable is scaled, indexed, nodal, or just a simple variable: */
		if (strncmp(descriptor,"scaled_",7)==0){
		
			/*recover partition vector: */
			variable_partition=variable_partitions[variablecount];
			npart=variable_partitions_npart[variablecount];
			nt=variable_partitions_nt[variablecount];

			/*Variable is scaled. Determine root name of variable (ex: scaled_DragCoefficient_1 -> DragCoefficient). Allocate distributed_values and fill the 
			 * distributed_values with the next npart variables: */

			//strcpy(root,strstr(descriptor,"_")+1); *strstr(root,"_")='\0';
			memcpy(root,strstr(descriptor,"_")+1,(strlen(strstr(descriptor,"_")+1)+1)*sizeof(char));
			*strstr(root,"_")='\0';

			distributed_values=xNew<double>(npart*nt);
			for(j=0;j<npart*nt;j++){
				distributed_values[j]=variables[i+j];
			}

			
			/*Now, pick up the parameter corresponding to root: */
			femmodel->parameters->FindParamInDataset(&parameter,&nrows,&ncols,QmuVariableDescriptorsEnum,StringToEnumx(root));

			/*We've got the parameter, we need to update it using the partition vector, and the distributed_values. 
			 In addition, the parameter can be either a static or transient (nrows+1) vector. Finally, the partition vectors can include
			 -1 (meaning, don't update). */
			
			//_printf_("nrows: " << nrows << " numberofvertices: " << numberofvertices << " numberofelements: " << numberofelements << "\n");

			if(ncols!=nt){
				/*we are trying to update a col sized transient input by scaling with a matrix of col size nt. This can only work if nt==1, otherwise, error out: */
				if (nt!=1) _error_("InputUpdateFromDakotax error message: transient input being updated should be the same col size as the number of time step in the qmu variable specificationi");
			}

			if(nt==1){
				/*scale all the columns by the same vector:*/
				if (nrows==numberofvertices || nrows==(numberofvertices+1)){
					for(k=0;k<numberofvertices;k++){
						if (variable_partition[k]==-1)continue;
						else{
							for(l=0;l<ncols;l++){
								*(parameter+ncols*k+l)=*(parameter+ncols*k+l)*distributed_values[(int)variable_partition[k]];
							}
						}
					}
				}
				else if (nrows==numberofelements || nrows==(numberofelements+1)){
					for(k=0;k<numberofelements;k++){
						if (variable_partition[k]==-1)continue;
						else{
							for(l=0;l<ncols;l++){
								*(parameter+ncols*k+l)=*(parameter+ncols*k+l)*distributed_values[(int)variable_partition[k]];
							}
						}
					}

				}
				else _error_("partitioning vector should be either elements or vertex sized!");

			}
			else{
				/*scale all the columns by the scalar matrix:*/
				if (nrows==numberofvertices || nrows==(numberofvertices+1)){
					for(k=0;k<numberofvertices;k++){
						if (variable_partition[k]==-1)continue;
						else{
							for(l=0;l<ncols;l++){
								*(parameter+ncols*k+l)=*(parameter+ncols*k+l)*distributed_values[(int)variable_partition[k]*nt+l];
							}
						}
					}
				}
				else if (nrows==numberofelements || nrows==(numberofelements+1)){
					for(k=0;k<numberofelements;k++){
						if (variable_partition[k]==-1)continue;
						else{
							for(l=0;l<ncols;l++){
								*(parameter+ncols*k+l)=*(parameter+ncols*k+l)*distributed_values[(int)variable_partition[k]*nt+l];
							}
						}
					}

				}
				else _error_("partitioning vector should be either elements or vertex sized!");
			}

			#ifdef _DEBUG_
				PetscSynchronizedPrintf(IssmComm::GetComm(),"Parameter matrix:");
				PetscSynchronizedFlush(IssmComm::GetComm());
				for(l=0;l<ncols;l++){
					PetscSynchronizedPrintf(IssmComm::GetComm()," time %i\n",l);
					PetscSynchronizedFlush(IssmComm::GetComm());

					for(k=0;k<numberofvertices;k++){
						PetscSynchronizedPrintf(IssmComm::GetComm()," node %i value %g\n",k+1,*(parameter+k*ncols+l));
						PetscSynchronizedFlush(IssmComm::GetComm());
					}
				}
				PetscSynchronizedPrintf(IssmComm::GetComm()," descriptor: %s root %s enum: %i\n",descriptor,root,StringToEnumx(root));
				PetscSynchronizedFlush(IssmComm::GetComm());
			#endif

			/*Update inputs using the parameter matrix: */
			if(nrows==numberofvertices || (nrows==numberofvertices+1))
				InputUpdateFromMatrixDakotax(femmodel, parameter, nrows,ncols,StringToEnumx(root), VertexEnum);
			else
				InputUpdateFromMatrixDakotax(femmodel, parameter, nrows,ncols,StringToEnumx(root), ElementEnum);

			/*increment i to skip the distributed values just collected: */
			i+=npart*nt-1; //careful, the for loop will add 1.
			
			/*Free allocations: */
			xDelete<double>(parameter);
			xDelete<double>(distributed_values);
		}
		else if (strncmp(descriptor,"distributed_",12)==0){
			
			/*recover partition vector: */
			variable_partition=variable_partitions[variablecount];
			npart=variable_partitions_npart[variablecount];

			/*Variable is distributed. Determine root name of variable (ex: distributed_DragCoefficient_1 -> DragCoefficient). 
			 * Allocate distributed_values and fill the distributed_values with the next npart variables: */

			memcpy(root,strstr(descriptor,"_")+1,(strlen(strstr(descriptor,"_")+1)+1)*sizeof(char));
			*strstr(root,"_")='\0';

			distributed_values=xNew<double>(npart);
			for(j=0;j<npart;j++){
				distributed_values[j]=variables[i+j];
			}

			//for (int j=0;j<npart;j++)_printf_(j << ":" << distributed_values[j] << "\n");
			
			//Call specialty code:
			InputUpdateSpecialtyCode(femmodel,distributed_values,variable_partition,npart,root);
			
			/*increment i to skip the distributed values just collected: */
			i+=npart-1; //careful, the for loop will add 1.
			
			/*Free allocations: */
			xDelete<double>(parameter);
			xDelete<double>(distributed_values);
		}
		else if (strncmp(descriptor,"indexed_",8)==0){
			_error_("indexed variables not supported yet!");
		}
		else if (strncmp(descriptor,"nodal_",8)==0){
			_error_("nodal variables not supported yet!");
		}
		else{
			/*Ok, standard variable, just update inputs using the variable: */
			InputUpdateFromConstantx(femmodel,variables[i],StringToEnumx(descriptor));
		}
		variablecount++;
	}

	/*Free ressources:*/
	for(i=0;i<variable_partitions_num;i++){
		IssmDouble* matrix=variable_partitions[i];
		xDelete<IssmDouble>(matrix);
	}
	xDelete<IssmDouble*>(variable_partitions);
	xDelete<int>(variable_partitions_npart);
	xDelete<int>(variable_partitions_nt);

} /*}}}*/
void  InputUpdateSpecialtyCode(FemModel* femmodel,IssmDouble* distributed_values,IssmDouble* variable_partition,int npart,char* root){ //{{{

	/*Here, we put all the code that cannot be handled any other place: */
	if (strncmp(root,"SurfaceloadModelid",18)==0){

		if(VerboseQmu()){
			_printf0_("Updating SurfaceloadModelid MME, with ids: ");
			for (int i=0;i<npart;i++)_printf0_((int)distributed_values[i]+1 << " ");
			_printf0_("\n");
		}


		if (femmodel->inputs2->GetInputObjectEnum(SurfaceloadIceThicknessChangeEnum)==DatasetInput2Enum)
			MmeToInput(femmodel,distributed_values,variable_partition,npart,SurfaceloadIceThicknessChangeEnum, P0Enum);

		if (femmodel->inputs2->GetInputObjectEnum(MaskIceLevelsetEnum)==DatasetInput2Enum)
			MmeToInput(femmodel,distributed_values,variable_partition,npart,MaskIceLevelsetEnum, P1Enum);

		if (femmodel->inputs2->GetInputObjectEnum(MaskOceanLevelsetEnum)==DatasetInput2Enum)
			MmeToInput(femmodel,distributed_values,variable_partition,npart,MaskOceanLevelsetEnum, P1Enum);

	}
	else _error_("InputUpdateSpecialtyCode error message: " << root << " not supported yet!");

}	//}}}
void  MmeToInput(FemModel* femmodel,IssmDouble* distributed_values,IssmDouble* variable_partition,int npart,int rootenum, int interpolationenum){ /*{{{*/

	TransientInput2* transientinput  = NULL;
	TransientInput2* transientinput2 = NULL;
	Tria* element                    = NULL;
	IssmDouble value;
	IssmDouble* values               = NULL;
	IssmDouble* times                = NULL;
	int N;
	int id;

	/*find thickness dataset: */
	DatasetInput2* datasetinput = femmodel->inputs2->GetDatasetInput2(rootenum);

	/*Initialize new transient input: */
	transientinput = datasetinput->GetTransientInputByOffset(0); _assert_(transientinput);
	transientinput->GetAllTimes(&times,&N);
	femmodel->inputs2->SetTransientInput(DummyEnum,times,N);
	transientinput2 = femmodel->inputs2->GetTransientInput(DummyEnum);
		
	for (int i=0;i<femmodel->elements->Size();i++){

		Tria*   element=xDynamicCast<Tria*>(femmodel->elements->GetObjectByOffset(i));

		if((int)variable_partition[element->Sid()]==-1)id=0; //grab background field
		else id=distributed_values[(int)variable_partition[element->Sid()]]-1; //grab partition field

		/*recover the right field from the mme: */
		transientinput = datasetinput->GetTransientInputByOffset(id); _assert_(transientinput);

		/*copy values from the transientinput to the final transientinput2: */
		for (int j=0;j<N;j++){
			TriaInput2* tria_input=transientinput->GetTriaInput(j);
			element->InputServe(tria_input);
			if(interpolationenum==P0Enum){
				value=tria_input->element_values[0];
				transientinput2->AddTriaTimeInput( j,1,&(element->lid),&value,P0Enum); 
			}
			else if(interpolationenum==P1Enum){

				/*Get values and lid list*/
				const int   numvertices     = element->GetNumberOfVertices();
				int        *vertexlids      = xNew<int>(numvertices);
				int        *vertexsids      = xNew<int>(numvertices);

				/*Recover vertices ids needed to initialize inputs*/
				element->GetVerticesLidList(&vertexlids[0]);
				element->GetVerticesSidList(&vertexsids[0]);
				values=tria_input->element_values;
				transientinput2->AddTriaTimeInput( j,numvertices,vertexlids,values,P1Enum); 
			}
		}
	}

	/*wipe out existing SurfaceloadIceThicknessChangeEnum dataset:*/
	femmodel->inputs2->ChangeEnum(DummyEnum,rootenum);

	//reconfigure:
	transientinput2->Configure(femmodel->parameters);
}	//}}}
