#Test Name: EarthSlr
import numpy as np

from gmshplanet import *
from gmtmask import *
from lovenumbers import *
from materials import *
from model import *
from parameterize import *
from paterson import *
from solve import *


#mesh earth:
md = model()
md.mesh = gmshplanet('radius', 6.371012 * 1e3, 'resolution', 700.) #700 km resolution mesh

#parameterize solidearth solution:
#solidearth loading:
md.solidearth.surfaceload.icethicknesschange = np.zeros((md.mesh.numberofelements, 1))
md.solidearth.sealevel = np.zeros((md.mesh.numberofvertices, 1))
md.dsl.global_average_thermosteric_sea_level_change = np.zeros((1, 1))
md.dsl.sea_surface_height_change_above_geoid = np.zeros((md.mesh.numberofvertices + 1, 1))
md.dsl.sea_water_pressure_change_at_sea_floor = np.zeros((md.mesh.numberofvertices + 1, 1))

#antarctica
late = np.sum(md.mesh.lat[md.mesh.elements - 1], axis=1) / 3
longe = np.sum(md.mesh.long[md.mesh.elements - 1], axis=1) / 3
pos = np.where(late < -80)[0]
md.solidearth.surfaceload.icethicknesschange[pos] = -100
#greenland
pos = np.where(np.logical_and.reduce((late > 70, late < 80, longe > -60, longe < -30)))[0]
md.solidearth.surfaceload.icethicknesschange[pos] = -100

#elastic loading from love numbers:
md.solidearth.lovenumbers = lovenumbers('maxdeg', 100)
#}}}

#mask:  {{{
mask = gmtmask(md.mesh.lat, md.mesh.long)
icemask = np.ones(md.mesh.numberofvertices)
pos = np.where(mask == 0)[0]
icemask[pos] = -1
pos = np.where(np.sum(mask[md.mesh.elements - 1], axis=1) < 3)
icemask[md.mesh.elements[pos, :] - 1] = -1
md.mask.ice_levelset = icemask
md.mask.ocean_levelset = -icemask

#make sure that the ice level set is all inclusive:
md.mask.land_levelset = np.zeros((md.mesh.numberofvertices))
md.mask.ocean_levelset = -np.ones((md.mesh.numberofvertices))

#make sure that the elements that have loads are fully grounded
pos = np.nonzero(md.solidearth.surfaceload.icethicknesschange)[0]
md.mask.ocean_levelset[md.mesh.elements[pos, :] - 1] = 1

#make sure wherever there is an ice load, that the mask is set to ice:
#pos = np.nonzero(md.solidearth.surfaceload.icethicknesschange)[0] # Do we need to do this twice?
md.mask.ice_levelset[md.mesh.elements[pos, :] - 1] = 1
# }}}

md.solidearth.settings.ocean_area_scaling = 0

#geometry for the bed, arbitrary
md.geometry.bed = -np.ones((md.mesh.numberofvertices, 1))

#materials
md.materials=materials('hydro')

#Miscellaneous
md.miscellaneous.name = 'test2002'

#Solution parameters
md.solidearth.settings.reltol = np.nan
md.solidearth.settings.abstol = 1e-3
md.solidearth.settings.computesealevelchange = 1

#max number of iteration reverted back to 10 (i.e., the original default value)
md.solidearth.settings.maxiter = 10

#eustatic run:
md.solidearth.settings.rigid = 0
md.solidearth.settings.elastic = 0
md.solidearth.settings.rotation = 0
md = solve(md, 'Sealevelrise')
Seustatic = md.results.SealevelriseSolution.Sealevel

#eustatic + rigid run:
md.solidearth.settings.rigid = 1
md.solidearth.settings.elastic = 0
md.solidearth.settings.rotation = 0
md = solve(md, 'Sealevelrise')
Srigid = md.results.SealevelriseSolution.Sealevel

#eustatic + rigid + elastic run:
md.solidearth.settings.rigid = 1
md.solidearth.settings.elastic = 1
md.solidearth.settings.rotation = 0
md = solve(md, 'Sealevelrise')
Selastic = md.results.SealevelriseSolution.Sealevel

#eustatic + rigid + elastic + rotation run:
md.solidearth.settings.rigid = 1
md.solidearth.settings.elastic = 1
md.solidearth.settings.rotation = 1
md = solve(md, 'Sealevelrise')
Srotation = md.results.SealevelriseSolution.Sealevel

#Fields and tolerances to track changes
field_names = ['Eustatic', 'Rigid', 'Elastic', 'Rotation']
field_tolerances = [1e-13, 1e-13, 1e-13, 1e-13]
field_values = [Seustatic, Srigid, Selastic, Srotation]
