%LOVE numbers class definition
%
%   Usage:
%      love=love() %will setup love numbers deg 1001 by default
%      love=love('maxdeg',10001);  %supply numbers of degrees required (here, 10001)

classdef love
	properties (SetAccess=public) 
		
		%regular love numbers:
		h           = []; %provided by PREM model
		k           = []; %idem
		l           = []; %idem
		
		%tidal love numbers for computing rotational feedback:
		th          = []; 
		tk          = []; 
		tl          = []; 
		tk2secular  = 0;  %deg 2 secular number.

	end
	methods
		function self = love(varargin) % {{{
			options=pairoptions(varargin{:});
			maxdeg=getfieldvalue(options,'maxdeg',1000);
			self=setdefaultparameters(self,maxdeg);
		end % }}}
		function self = setdefaultparameters(self,maxdeg) % {{{
		
			%initialize love numbers:
			self.h=love_numbers('type','loadingverticaldisplacement','referenceframe','CM','maxdeg',maxdeg);
			self.k=love_numbers('type','loadinggravitationalpotential','referenceframe','CM','maxdeg',maxdeg);
			self.l=love_numbers('type','loadinghorizontaldisplacement','referenceframe','CM','maxdeg',maxdeg);
			self.th=love_numbers('type','tidalverticaldisplacement','referenceframe','CM','maxdeg',maxdeg);
			self.tk=love_numbers('type','tidalgravitationalpotential','referenceframe','CM','maxdeg',maxdeg);
			self.tl=love_numbers('type','tidalhorizontaldisplacement','referenceframe','CM','maxdeg',maxdeg);

			%secular fluid love number: 
			self.tk2secular=0.942; 


		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			if ~ismember('SealevelriseAnalysis',analyses) | (strcmp(solution,'TransientSolution') & md.transient.islove==0), 
				return; 
			end

			md = checkfield(md,'fieldname','solidearth.love.h','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','solidearth.love.k','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','solidearth.love.l','NaN',1,'Inf',1);
			
			
			md = checkfield(md,'fieldname','solidearth.love.th','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','solidearth.love.tk','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','solidearth.love.tk2secular','NaN',1,'Inf',1);

			%check that love numbers are provided at the same level of accuracy: 
			if (size(self.h,1)~=size(self.k,1) | size(self.h,1)~=size(self.l,1)),
				error('love error message: love numbers should be provided at the same level of accuracy');
			end


		end % }}}
		function list=defaultoutputs(self,md) % {{{
			list = {};
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   love parameters:'));

			fielddisplay(self,'h','load Love number for radial displacement');
			fielddisplay(self,'k','load Love number for gravitational potential perturbation');
			fielddisplay(self,'l','load Love number for horizontal displacements');

			fielddisplay(self,'th','tidal load Love number (deg 2)');
			fielddisplay(self,'tk','tidal load Love number (deg 2)');
			fielddisplay(self,'tk2secular','secular fluid Love number');

		end % }}}
		function marshall(self,prefix,md,fid) % {{{

			WriteData(fid,prefix,'object',self,'fieldname','h','name','md.solidearth.love.h','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','k','name','md.solidearth.love.k','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','l','name','md.solidearth.love.l','format','DoubleMat','mattype',1);

			WriteData(fid,prefix,'object',self,'fieldname','th','name','md.solidearth.love.th','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','tk','name','md.solidearth.love.tk','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'fieldname','tl','name','md.solidearth.love.tl','format','DoubleMat','mattype',1);
			WriteData(fid,prefix,'object',self,'data',self.tk2secular,'fieldname','love.tk2secular','format','Double');

		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
			writejs1Darray(fid,[modelname '.love.h'],self.h);
			writejs1Darray(fid,[modelname '.love.k'],self.k);
			writejs1Darray(fid,[modelname '.love.l'],self.l);
		end % }}}
		function self = extrude(self,md) % {{{
		end % }}}
	end
end
