import numpy as np

from checkfield import checkfield
from fielddisplay import fielddisplay
from WriteData import WriteData


class solidearthsettings(object):
    '''
    SOLIDEARTHSETTINGS class definition

        Usage:
            solidearthsettings = solidearthsettings()
    '''

    def __init__(self, *args): #{{{
        nargin = len(args)

        if nargin == 0:
            self.setdefaultparameters()
        else:
            raise Exception('constructor not supported')
    #}}}

    def __repr__(self): # {{{
        s = '   solidearth settings:\n'
        s += '{}\n'.format(fielddisplay(self, 'reltol', 'sea level rise relative convergence criterion, (default, NaN: not applied)'))
        s += '{}\n'.format(fielddisplay(self, 'abstol', 'sea level rise absolute convergence criterion, NaN: not applied'))
        s += '{}\n'.format(fielddisplay(self, 'maxiter', 'maximum number of nonlinear iterations'))
        s += '{}\n'.format(fielddisplay(self, 'ocean_area_scaling', 'correction for model representation of ocean area [default: No correction]'))
        s += '{}\n'.format(fielddisplay(self, 'computesealevelchange', 'compute sealevel change from GRD in addition to steric?) default 0'))
        s += '{}\n'.format(fielddisplay(self, 'runfrequency', 'how many time steps we skip before we run solidearthsettings solver during transient (default: 1)'))
        s += '{}\n'.format(fielddisplay(self, 'rigid', 'rigid earth graviational potential perturbation'))
        s += '{}\n'.format(fielddisplay(self, 'elastic', 'elastic earth graviational potential perturbation'))
        s += '{}\n'.format(fielddisplay(self, 'degacc', 'accuracy (default .01 deg) for numerical discretization of the Green\'s functions'))

        return s
    #}}}

    def setdefaultparameters(self): # {{{
        #Convergence criterion: absolute, relative, and residual
        self.reltol = 0.01 # 1 percent
        self.abstol = np.nan # default

        #maximum of non-linear iterations
        self.maxiter = 5

        #computational flags
        self.rigid = 1
        self.elastic = 1
        self.rotation = 1
        self.ocean_area_scaling = 0
        self.computesealevelchange = 0

        #numerical discetization accuracy
        self.degacc = .01

        #how many time steps we skip before we run solidearthsettings solver during transient
        self.runfrequency = 1

        #horizontal displacemnet? (not by default)
        self.horiz = 0

        return self
    #}}}

    def checkconsistency(self, md, solution, analyses): # {{{
        if not 'SealevelriseAnalysis' in analyses or solution == 'TransientSolution' and md.transient.issolidearthsettings == 0:
            return md

        md = checkfield(md, 'fieldname', 'solidearth.settings.reltol', 'size', [1, 1])
        md = checkfield(md, 'fieldname', 'solidearth.settings.abstol', 'size', [1, 1])
        md = checkfield(md, 'fieldname', 'solidearth.settings.maxiter', 'size', [1, 1], '>=', 1)
        md = checkfield(md, 'fieldname', 'solidearth.settings.runfrequency', 'size', [1, 1], '>=', 1)
        md = checkfield(md, 'fieldname', 'solidearth.settings.degacc', 'size', [1, 1], '>=', 1e-10)
        md = checkfield(md, 'fieldname', 'solidearth.settings.horiz', 'NaN', 1, 'Inf', 1, 'values', [0, 1])

        #a coupler to planet model is provided
        if self.computesealevelchange:
            if md.transient.iscoupler:
                #we are good
                pass
            else:
                if md.mesh.__class__.__name__ == 'mesh3dsurface':
                    #we are good
                    pass
                else:
                    raise Exception('model is requesting sealevel computations without being a mesh3dsurface, or being coupled to one!')

        return md
    #}}}

    def marshall(self, prefix, md, fid): #{{{
        WriteData(fid, prefix, 'object', self, 'fieldname', 'reltol', 'name', 'md.solidearth.settings.reltol', 'format', 'Double')
        WriteData(fid,prefix,'object', self,'fieldname', 'abstol', 'name', 'md.solidearth.settings.abstol', 'format', 'Double')
        WriteData(fid,prefix,'object', self,'fieldname', 'maxiter', 'name', 'md.solidearth.settings.maxiter', 'format', 'Integer')
        WriteData(fid,prefix,'object', self,'fieldname', 'rigid', 'name', 'md.solidearth.settings.rigid', 'format', 'Boolean')
        WriteData(fid,prefix,'object', self,'fieldname', 'elastic', 'name', 'md.solidearth.settings.elastic', 'format', 'Boolean')
        WriteData(fid,prefix,'object', self,'fieldname', 'rotation', 'name', 'md.solidearth.settings.rotation', 'format', 'Boolean')
        WriteData(fid,prefix,'object', self,'fieldname', 'ocean_area_scaling', 'name', 'md.solidearth.settings.ocean_area_scaling', 'format', 'Boolean')
        WriteData(fid,prefix,'object', self,'fieldname', 'runfrequency', 'name', 'md.solidearth.settings.runfrequency', 'format', 'Integer')
        WriteData(fid,prefix,'object', self,'fieldname', 'degacc', 'name', 'md.solidearth.settings.degacc', 'format', 'Double')
        WriteData(fid,prefix,'object', self,'fieldname', 'horiz', 'name', 'md.solidearth.settings.horiz', 'format', 'Integer')
        WriteData(fid,prefix,'object', self,'fieldname', 'computesealevelchange', 'name', 'md.solidearth.settings.computesealevelchange', 'format', 'Integer')
    #}}}

    def extrude(self, md): #{{{
        return self
    #}}}
