%GIAMME class definition
%
%   Usage:
%      gia=giamme(); %gia class based on a multi-model 
%                     ensemble  (ex: Caron et al 2017 statistics)

classdef giamme
	properties (SetAccess=public) 

		modelid; %index into the multi-model ensemble
		Ngia;  %geoid ensemble
		Ugia;  %uplift ensemble

	end
	methods
		function self = extrude(self,md) % {{{
			for i=1:length(self.Ngia),
				self.Ngia{i}=project3d(md,'vector',self.Ngia{i},'type','node','layer',1);
				self.Ugia{i}=project3d(md,'vector',self.Ugia{i},'type','node','layer',1);
			end
		end % }}}
		function self = giamme(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					self=structtoobj(gia(),varargin{1});
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			self.modelid=0;
			self.Ngia=[];
			self.Ugia=[];

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			%Early return
			if ~ismember('SealevelriseAnalysis',analyses), return; end
			if (strcmp(solution,'TransientSolution') & md.transient.isslr == 0), return; end
			md = checkfield(md,'field',self.Ngia,'NaN',1,'Inf',1);
			md = checkfield(md,'field',self.Ugia,'NaN',1,'Inf',1);
			md = checkfield(md,'field',self.modelid,'NaN',1,'Inf',1,'>=',0,'<=',length(self.Ngia));
		end % }}}
		function disp(self) % {{{

			disp(sprintf('   gia mme parameters:'));
			fielddisplay(self,'modelid','index into the multi-model ensemble, determines which field will be used.');
			fielddisplay(self,'Ngia','geoid (m/yr).');
			fielddisplay(self,'Ugia','uplift (m/yr).');
		end % }}}
		function marshall(self,prefix,md,fid) % {{{

			WriteData(fid,prefix,'name','md.gia.model','data',3,'format','Integer');
			WriteData(fid,prefix,'name','md.gia.nummodels','data',size(self.Ngia,2),'format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','modelid','format','Double');
			if size(self.Ngia,2)==0,
				WriteData(fid,prefix,'data',zeros(md.mesh.numberofvertices,1),'format','DoubleMat','mattype',1,'name','md.gia.Ngia');
				WriteData(fid,prefix,'data',zeros(md.mesh.numberofvertices,1),'format','DoubleMat','mattype',1,'name','md.gia.Ugia');
			else
				WriteData(fid,prefix,'object',self,'fieldname','Ngia','format','DoubleMat');
				WriteData(fid,prefix,'object',self,'fieldname','Ugia','format','DoubleMat');
			end
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
			
			writejsdouble(fid,[modelname '.gia.modelid'],self.modelid);
			writejscellarray(fid,[modelname '.gia.Ngia'],self.Ngia);
			writejscellarray(fid,[modelname '.gia.Ugia'],self.Ugia);

		end % }}}
	end
end
