%ISMIP6 class definition
%
%   Usage:
%      is6 = ismip6('root',rootdir,'dirs',listdirectories);
%   Example: 
%      is6 = ismip6('root','./gis/','dirs',{'UVW','ISSM'});

classdef ismip6 < handle
	properties (SetAccess=public) %Model fields
		
		root = ''; %where are the files for CMIP5
		n   = 0;   %number of files
		directories   = {};   %directories where the files are
		experiments   = {};   %names of experiments
		thickness     = {};   %placeholder for thicknesses
		deltathickness     = {};   %placeholder for delta thicknesses
		icemask     = {};   %placeholder for ice masks
		oceanmask     = {};   %placeholder for ocean masks
		time     = {};   %placeholder for times
	end
	methods
		function self = ismip6(varargin) % {{{

			if nargin==0, 
				self=setdefaultparameters(self);
			else 
				self=setdefaultparameters(self);

				options=pairoptions(varargin{:});

				self.root=getfieldvalue(options,'root');
				self.directories=getfieldvalue(options,'directories');
				self.n=length(self.directories);

				%verify the directories exist: 
				for i=1:self.n,
					if ~exist([self.root '/' self.directories{i}],'dir'),
						error(['ismip6  constructor error: ' self.root '/' self.directories{i} ' does not exist']);
					end
				end

				%figure out names of experiments: 
				self.experiments=self.directories;
				for i=1:self.n,
					dir=self.directories{i};
					ind=findstr(dir,'exp');
					name=dir(1:ind-2);
					name=strrep(name,'/','-');
					self.experiments{i}=name;
				end

				%initialize fields: 
				self.thickness=cell(self.n,1);
				self.deltathickness=cell(self.n,1);
				self.icemask=cell(self.n,1);
				self.oceanmask=cell(self.n,1);

			end
		end
		%}}}
		function self = setdefaultparameters(self) % {{{
		end
		%}}}
		function disp(self) % {{{
			disp('   CMIP5 Ocean MIP:');

			fielddisplay(self,'n','number of files');
			fielddisplay(self,'root','where are the files for ISMIP6');
			fielddisplay(self,'directories','IMSIP6 directories');
		end % }}}
		function listexp(self) % {{{
			disp('ISMIP6  list of experiments:');
			for i=1:self.n,
				disp(['   ' self.experiments{i}]);
			end

		end % }}}
		function [output,time]=read(self,experiment,field) % {{{

			%go through list of experiments and find the right one: 
			if strcmpi(class(experiment),'double'),
				ind=experiment;
			elseif strcmpi(class(experiment),'char'),
				for i=1:self.n,
					if strcmpi(experiment,self.experiments{i}),
						ind=i;
						break;
					end
				end
			else 
				error(['ismip6 read error message: experiment should be a string or index']);
			end

			if ind==0 
				error(['ismip6 read error message: experiment ' experiment ' could not be found!']);
			end;

			%figure out the files in this directory: 
			dir=self.directories{ind};
			currentdir=pwd;
			cd([self.root '/' dir]); 
			list=listfiles;
			cd(currentdir);

			%go through list of files and figure out which one starts with the field: 
			for i=1:length(list),
				file=list{i};
				ind=findstr(file,'_');
				file_field=file(1:ind-1);
				if strcmpi(file_field,field),
					break;
				end
			end

			%read file: 
			%output=ncread([self.root '/' dir '/' file],'file_field');
			time=ncread([self.root '/' dir '/' file],'time');
			output=ncread([self.root '/' dir '/' file ],field);

		end % }}}
		function interpolate(self,field,ismip2mesh,ismip2mesh_correction) % {{{

			for i=1:self.n,
				disp(['reading and interpolating field ' field ' for model ' self.experiments{i}]);

				%read field from disk: 
				[h,t]=self.read(i,field); nt=length(t);

				%map onto 1 dimension field: 
				ht=zeros(size(h,1)*size(h,2),nt);
				for j=1:size(h,3),
					hj= h(:,:,j)'; hj=hj(:); ht(:,j)=double(hj);
				end

				%map onto mesh: 
				hg=ismip2mesh_correction.*(ismip2mesh*ht); 
				pos=find(isnan(hg)); hg(pos)=0;

				%keep field:
				if strcmpi(field,'lithk'),
					self.thickness{i}=hg; 
				end
				self.time{i}=t;
			end
		end  % }}}
	end
end
