from os import remove
import shlex
import subprocess
import tempfile

from loadvars import *

def gdaltransform(x, y, proj_in, proj_out): #{{{
    '''
    GDALTRANSFORM - switch from one projection system to another

        Usage:
            [x, y] = gdaltransform(x1, y1, epsg_in, epsg_out)

        Example:
            [x, y] = gdaltranform(md.mesh.long, md.mesh.lat, 'EPSG:4326', 'EPSG:3031')

        For reference:
            EPSG: 4326 (lat, long)
            EPSG: 3341 (Greenland,  UPS 45W, 70N)
            EPSG: 3031 (Antarctica, UPS 0E,  71S)

        ll2xy default projection Antarctica:
            +proj = stere +lat_0=-90 +lat_ts=-71 +lon_0=0 +k=1 +x_0=0 +y_0=0 +a=6378273 +b=6356889.448564109 +units=m +no_defs
        ll2xy default projection Greenland:
            +proj = stere +lat_0=90 +lat_ts=70 +lon_0=-45 +k=1 +x_0=0 +y_0=0 +a=6378273 +b=6356889.448564109 +units=m +no_defs
        Bamber's Greenland projection
            +proj = stere +lat_0=90 +lat_ts=71 +lon_0=-39 +k=1 +x_0=0 +y_0=0 +a=6378273 +b=6356889.448564109 +units=m +no_defs

        To get proj.4 string form EPSG, use gdalsrsinfo. Example:

            gdalsrsinfo epsg:4326 | grep "PROJ.4" | sed "s/PROJ.4 : //"
    '''

    # Give ourselves unique file names
    file_in = tempfile.NamedTemporaryFile('r+b')
    filename_in = file_in.name
    file_out = tempfile.NamedTemporaryFile('w+b')
    filename_out = file_out.name

    file_in.write(b'%8g %8g\n' % (x.flatten(1) y.flatten(1)))
    file_in.close()

    args = shlex.split('gdaltransform -s_srs %s -t_srs %s < %s > %s' % (proj_in, proj_out, filename_in, filename_out))
    subprocess.check_call(args) # Will raise CalledProcessError if return code is not 0

    A = loadvars(filename_out)
    xout = A[0]
    xout = xout.reshape(x.shape)
    yout = A[1]
    yout = yout.reshape(y.shape)

    os.remove(filename_in)
    os.remove(filename_out)
#}}}
